﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrv_DisplayBuffer.h"
#include "capsrv_DisplayBuffer-module.nvnflinger.h"

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nvrm_surface.h>
#include <nvgr.h>

#include <nvddk_vic.h>

#include "capsrv_ResultCapture.h"
#include "capsrv_CaptureModule-module.nvnflinger.h"
#include "capsrv_ImageBuffer-module.nvnflinger.h"
#include "detail/capsrv_VicOperation-module.nvnflinger.h"

#include "detail/capsrv_ImageFormatInfo-module.nvnflinger.h"

namespace nn{ namespace capsrv{ namespace capture{

    static const uint32_t BufferUsageBits = GRALLOC_USAGE_HW_COMPOSER | GRALLOC_USAGE_HW_VIDEO_ENCODER | GRALLOC_USAGE_RENDERSCRIPT;

    //------------------------------------
    // Data
    //------------------------------------

    DisplayBufferData::DisplayBufferData() NN_NOEXCEPT
        : m_pModule(nullptr)
        , m_PlaneCount(0)
        , m_Width(0)
        , m_Height(0)
        , m_Format()
        , m_hBuffer()
        , m_pGraphicBuffer()
        , m_IsContentTrivial(true)
    {
    }

    void DisplayBufferData::InitializeRgba(CaptureModule* pModule, ImageFormat format, ImageBuffer* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(format == ImageFormat_Rgba8 || format == ImageFormat_SharedTexture);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES(pBuffer->IsInitialized());
        NN_SDK_REQUIRES_EQUAL(pBuffer->GetData()->GetImageFormat(), format);

        auto pBufferData = pBuffer->GetData();

        m_pModule = pModule;
        m_PlaneCount = 1;
        m_pImageBufferList[0] = pBuffer;
        m_SurfaceList[0] = *pBufferData->GetSurface();
        m_Width  = pBufferData->GetWidth();
        m_Height = pBufferData->GetHeight();
        m_Format = format;

        // create graphic buffer
        nvgr_import_nvrmsurface(&m_SurfaceList[0], m_PlaneCount, &m_hBuffer);

        m_pGraphicBuffer = new android::GraphicBuffer(
            static_cast<uint32_t>(pBufferData->GetWidth()),
            static_cast<uint32_t>(pBufferData->GetHeight()),
            detail::ImageFormatInfo::GetComposerImageFormat(m_Format),
            BufferUsageBits,
            static_cast<uint32_t>(nvgr_get_stride(m_hBuffer)),
            const_cast<native_handle*>(m_hBuffer),
            false
        );
    }

    void DisplayBufferData::InitializeYuvPlanar(
        CaptureModule* pModule,
        ImageFormat format,
        ImageBuffer* pBufferY,
        ImageBuffer* pBufferU,
        ImageBuffer* pBufferV
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsYuvFormat(format));
        NN_SDK_REQUIRES_NOT_NULL(pBufferY);
        NN_SDK_REQUIRES_NOT_NULL(pBufferU);
        NN_SDK_REQUIRES_NOT_NULL(pBufferV);
        NN_SDK_REQUIRES(pBufferY->IsInitialized());
        NN_SDK_REQUIRES(pBufferU->IsInitialized());
        NN_SDK_REQUIRES(pBufferV->IsInitialized());
        NN_SDK_REQUIRES_EQUAL(pBufferY->GetData()->GetImageFormat(), ImageFormat_Y_Bt601);
        NN_SDK_REQUIRES_EQUAL(pBufferU->GetData()->GetImageFormat(), ImageFormat_U_Bt601);
        NN_SDK_REQUIRES_EQUAL(pBufferV->GetData()->GetImageFormat(), ImageFormat_V_Bt601);

        auto pBufferDataY = pBufferY->GetData();
        auto pBufferDataU = pBufferU->GetData();
        auto pBufferDataV = pBufferV->GetData();

        m_pModule = pModule;
        m_PlaneCount = 3;
        m_pImageBufferList[0] = pBufferY;
        m_pImageBufferList[1] = pBufferU;
        m_pImageBufferList[2] = pBufferV;
        m_SurfaceList[0] = *pBufferDataY->GetSurface();
        m_SurfaceList[1] = *pBufferDataU->GetSurface();
        m_SurfaceList[2] = *pBufferDataV->GetSurface();
        m_Width  = pBufferDataY->GetWidth();
        m_Height = pBufferDataY->GetHeight();
        m_Format = format;

        // create graphic buffer
        int import_result = nvgr_import_nvrmsurface(&m_SurfaceList[0], m_PlaneCount, &m_hBuffer);
        NN_ABORT_UNLESS_EQUAL(import_result, 0);

        m_pGraphicBuffer = new android::GraphicBuffer(
            static_cast<uint32_t>(pBufferDataY->GetWidth()),
            static_cast<uint32_t>(pBufferDataY->GetHeight()),
            detail::ImageFormatInfo::GetComposerImageFormat(m_Format),
            BufferUsageBits,
            static_cast<uint32_t>(nvgr_get_stride(m_hBuffer)),
            const_cast<native_handle*>(m_hBuffer),
            false
        );
        NN_ABORT_UNLESS_NOT_NULL(m_pGraphicBuffer);
    }

    void DisplayBufferData::InitializeYuvSemiPlanar(
        CaptureModule* pModule,
        ImageFormat format,
        ImageBuffer* pBufferY,
        ImageBuffer* pBufferUv
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsYuvFormat(format));
        NN_SDK_REQUIRES_NOT_NULL(pBufferY);
        NN_SDK_REQUIRES_NOT_NULL(pBufferUv);
        NN_SDK_REQUIRES(pBufferY->IsInitialized());
        NN_SDK_REQUIRES(pBufferUv->IsInitialized());
        NN_SDK_REQUIRES_EQUAL(pBufferY->GetData()->GetImageFormat(), ImageFormat_Y_NV12);
        NN_SDK_REQUIRES_EQUAL(pBufferUv->GetData()->GetImageFormat(), ImageFormat_Uv_NV12);

        auto pBufferDataY = pBufferY->GetData();
        auto pBufferDataUv = pBufferUv->GetData();

        m_pModule = pModule;
        m_PlaneCount = 2;
        m_pImageBufferList[0] = pBufferY;
        m_pImageBufferList[1] = pBufferUv;
        m_SurfaceList[0] = *pBufferDataY->GetSurface();
        m_SurfaceList[1] = *pBufferDataUv->GetSurface();
        m_Width  = pBufferDataY->GetWidth();
        m_Height = pBufferDataY->GetHeight();
        m_Format = format;

        // create graphic buffer
        int import_result = nvgr_import_nvrmsurface(&m_SurfaceList[0], m_PlaneCount, &m_hBuffer);
        NN_ABORT_UNLESS_EQUAL(import_result, 0);

        m_pGraphicBuffer = new android::GraphicBuffer(
            static_cast<uint32_t>(pBufferDataY->GetWidth()),
            static_cast<uint32_t>(pBufferDataY->GetHeight()),
            detail::ImageFormatInfo::GetComposerImageFormat(m_Format),
            BufferUsageBits,
            static_cast<uint32_t>(nvgr_get_stride(m_hBuffer)),
            const_cast<native_handle*>(m_hBuffer),
            false
        );
        NN_ABORT_UNLESS_NOT_NULL(m_pGraphicBuffer);
    }

    void DisplayBufferData::Finalize() NN_NOEXCEPT
    {
        m_pGraphicBuffer.clear();
        auto pGralloc = m_pModule->GetGralloc();
        pGralloc->free(pGralloc, m_hBuffer);
        m_hBuffer = nullptr;
        m_pModule = nullptr;
    }

    void DisplayBufferData::FlushCache() const NN_NOEXCEPT
    {
        for(int i = 0; i < m_PlaneCount; i++)
        {
            m_pImageBufferList[i]->FlushCache();
        }
    }

    bool DisplayBufferData::IsContentTrivial() const NN_NOEXCEPT
    {
        return m_IsContentTrivial;
    }

    nn::Result DisplayBufferData::CopyToImageBuffer(
        ImageBuffer* pDstBuffer,
        CaptureModule* pModule,
        const Rectangle* pDstRect,
        FilterMode filterMode
        ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDstBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pModule);

        detail::VicCopyDestinationRectangle dstRect = {};
        if(pDstRect)
        {
            dstRect.left   = pDstRect->x;
            dstRect.top    = pDstRect->y;
            dstRect.right  = pDstRect->x + pDstRect->width;
            dstRect.bottom = pDstRect->y + pDstRect->height;
        }

        detail::VicCopySourceInfo srcInfo = {};
        srcInfo.width = m_Width;
        srcInfo.height = m_Height;
        srcInfo.pSurfaceList = &m_SurfaceList[0];
        srcInfo.surfaceCount = m_PlaneCount;

        detail::VicCopyOption option = {};
        switch(filterMode)
        {
        case FilterMode_Nearest:
            {
                option.filter = detail::VicCopyFilter_Nearest;
                break;
            }
        case FilterMode_Nicest:
            {
                option.filter = detail::VicCopyFilter_Nicest;
                break;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }

        NN_RESULT_DO(detail::VicOperation::CopyToSurface(
            pDstBuffer,
            pModule,
            pDstRect ? &dstRect : nullptr,
            srcInfo,
            option
        ));

        NN_RESULT_SUCCESS;
    }



    //------------------------------------
    // Interface
    //------------------------------------

    DisplayBuffer::DisplayBuffer() NN_NOEXCEPT
    {
        NN_CAPSRV_CAPTURE_DATASTORAGE_CHECK(DataType, m_DataStorage);
        std::memset(this, 0, sizeof(*this));
    }

    void DisplayBuffer::InitializeRgba(CaptureModule* pModule, ImageFormat format, ImageBuffer* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        m_pData = new(&m_DataStorage) DataType();

        return m_pData->InitializeRgba(pModule, format, pBuffer);
    }

    void DisplayBuffer::InitializeYuvPlanar(CaptureModule* pModule, ImageFormat format, ImageBuffer* pBufferY, ImageBuffer* pBufferU, ImageBuffer* pBufferV) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        m_pData = new(&m_DataStorage) DataType();

        return m_pData->InitializeYuvPlanar(pModule, format, pBufferY, pBufferU, pBufferV);
    }

    void DisplayBuffer::InitializeYuvSemiPlanar(CaptureModule* pModule, ImageFormat format, ImageBuffer* pBufferY, ImageBuffer* pBufferUv) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        m_pData = new(&m_DataStorage) DataType();

        return m_pData->InitializeYuvSemiPlanar(pModule, format, pBufferY, pBufferUv);
    }

    void DisplayBuffer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->Finalize();
        m_pData->~DataType();
        m_pData = nullptr;
    }

    ImageFormat DisplayBuffer::GetImageFormat() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->m_Format;
    }

    const ImageBuffer* DisplayBuffer::GetImageBuffer() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_pData->m_PlaneCount, 1);
        return m_pData->m_pImageBufferList[0];
    }

    void DisplayBuffer::GetImageBufferYuvPlanar(
        const ImageBuffer** pOutBufferY,
        const ImageBuffer** pOutBufferU,
        const ImageBuffer** pOutBufferV
    ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_pData->m_PlaneCount, 3);
        *pOutBufferY = m_pData->m_pImageBufferList[0];
        *pOutBufferU = m_pData->m_pImageBufferList[1];
        *pOutBufferV = m_pData->m_pImageBufferList[2];
    }

    void DisplayBuffer::GetImageBufferYuvPlanar(
        ImageBuffer** pOutBufferY,
        ImageBuffer** pOutBufferU,
        ImageBuffer** pOutBufferV
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_pData->m_PlaneCount, 3);
        *pOutBufferY = m_pData->m_pImageBufferList[0];
        *pOutBufferU = m_pData->m_pImageBufferList[1];
        *pOutBufferV = m_pData->m_pImageBufferList[2];
    }

    void DisplayBuffer::FlushCache() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->FlushCache();
    }

    bool DisplayBuffer::IsContentTrivial() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->IsContentTrivial();
    }

    nn::Result DisplayBuffer::CopyToImageBuffer(
        ImageBuffer* pDstBuffer,
        CaptureModule* pModule,
        const Rectangle* pDstRect,
        FilterMode filterMode
        ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pData->CopyToImageBuffer(pDstBuffer, pModule, pDstRect, filterMode);
    }

}}}
