﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrv_CopyImageBuffer.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>
#include "capsrv_ImageBuffer-module.nvnflinger.h"
#include "detail/capsrv_VicOperation-module.nvnflinger.h"

#include <nn/nn_SdkLog.h>

namespace nn{ namespace capsrv{ namespace capture{

    nn::Result CopyImageBuffer(
        CaptureModule* pModule,
        ImageBuffer* pDstImage,
        const Rectangle& dstRect,
        const ImageBuffer*const* pSrcImageList,
        int srcImageCount,
        int srcWidth,
        int srcHeight,
        FilterMode filterMode
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NN_SDK_REQUIRES_NOT_NULL(pDstImage);
        NN_SDK_REQUIRES_NOT_NULL(pSrcImageList);

        return CopyImageBuffer(pModule, &pDstImage, 1, dstRect, pSrcImageList, srcImageCount, srcWidth, srcHeight, filterMode);
    }

    nn::Result CopyImageBuffer(
        CaptureModule* pModule,
        ImageBuffer*const* pDstImageList,
        int dstImageCount,
        const Rectangle& dstRect,
        const ImageBuffer*const* pSrcImageList,
        int srcImageCount,
        int srcWidth,
        int srcHeight,
        FilterMode filterMode
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NN_SDK_REQUIRES_NOT_NULL(pDstImageList);
        NN_SDK_REQUIRES_NOT_NULL(pSrcImageList);
        NN_SDK_REQUIRES_MINMAX(dstImageCount, 1, CopyDestinationImageBufferCountMax);
        NN_SDK_REQUIRES_MINMAX(srcImageCount, 1, CopySourceImageBufferCountMax);

        NN_RESULT_THROW_UNLESS(srcImageCount > 0 && srcImageCount <= CopySourceImageBufferCountMax, ResultScreenShotError());
        NN_RESULT_THROW_UNLESS(dstImageCount > 0 && dstImageCount <= CopyDestinationImageBufferCountMax, ResultScreenShotError());

        NvRmSurface srcSurfaces[CopySourceImageBufferCountMax] = {};
        for(int i = 0; i < srcImageCount; i++)
        {
            srcSurfaces[i] = *pSrcImageList[i]->GetData()->GetSurface();
        }

        NvRmSurface dstSurfaces[CopyDestinationImageBufferCountMax] = {};
        for(int i = 0; i < dstImageCount; i++)
        {
            dstSurfaces[i] = *pDstImageList[i]->GetData()->GetSurface();
        }


        capture::detail::VicCopySourceInfo vicSrcInfo = {};
        vicSrcInfo.width = srcWidth;
        vicSrcInfo.height = srcHeight;
        vicSrcInfo.surfaceCount = srcImageCount;
        vicSrcInfo.pSurfaceList = srcSurfaces;

        capture::detail::VicCopyDestinationInfo vicDstInfo = {};
        vicDstInfo.rectangle.left = dstRect.x;
        vicDstInfo.rectangle.right = dstRect.x + dstRect.width;
        vicDstInfo.rectangle.top = dstRect.y;
        vicDstInfo.rectangle.bottom = dstRect.y + dstRect.height;
        vicDstInfo.surfaceCount = dstImageCount;
        vicDstInfo.pSurfaceList = dstSurfaces;

        capture::detail::VicCopyOption vicOption = {};
        switch(filterMode)
        {
        case FilterMode_Nearest:
            vicOption.filter = capture::detail::VicCopyFilter_Nearest;
            break;
        default:
            vicOption.filter = capture::detail::VicCopyFilter_Nicest;
        }

        NN_RESULT_DO(capture::detail::VicOperation::CopyToSurfaceList(
            pModule,
            vicDstInfo,
            vicSrcInfo,
            vicOption
        ));
        NN_RESULT_SUCCESS;
    }

}}}
