﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/album/album_MovieMaker.h>
#include <nn/album/album_MovieMaker.internal.h>

#include <mutex>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/grc/grc_Application.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>
#include <nn/album/album_Result.h>
#include <nn/album/album_ImageOrientation.h>
#include <nn/capsrv/capsrv_ScreenShotOrientation.h>
#include <nn/capsrv/capsrv_AlbumAccessForApplication.h>
#include "album_Log.h"
#include "album_ConvertCapsrvResult.h"
#include "album_LibraryState.h"
#include <nn/am/am_Shim.h>

namespace nn{ namespace album{

    namespace {
        bool g_IsMovieMakerErrorConversionEnabled = true;

        struct MovieMakerState
        {
            uint64_t m_LayerHandle;
            nn::grc::MovieLayer* m_pLayer;

            nn::os::SystemEventType m_FinishMovieFileReadyEvent;
            nn::os::SystemEventType m_EncodeAudioReadyEvent;

            bool m_IsRecorderManaged;
            bool m_IsMovieMakerRunning;
        };

        nn::Result ConvertMovieMakerError(nn::Result src) NN_NOEXCEPT
        {
            if(!g_IsMovieMakerErrorConversionEnabled)
            {
                return src;
            }

            NN_RESULT_TRY(src)
                NN_RESULT_CATCH(nn::grc::ResultAlbumStorageFull)
                {
                    NN_RESULT_THROW(nn::album::ResultAlbumFull());
                }
                NN_RESULT_CATCH(nn::grc::ResultAlbumFileCountLimit)
                {
                    NN_RESULT_THROW(nn::album::ResultAlbumFileCountLimit());
                }
                NN_RESULT_CATCH(nn::grc::ResultAlbumFileSizeLimit)
                {
                    NN_RESULT_THROW(nn::album::ResultAlbumFileSizeLimit());
                }
                NN_RESULT_CATCH(nn::grc::ResultAlbumStorageError)
                {
                    NN_RESULT_THROW(nn::album::ResultSdcardNotAvailable());
                }
                NN_RESULT_CATCH(nn::grc::ResultAlbumInvalidParameter)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(src);
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW(nn::album::ResultAlbumError());
                }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }

        nn::os::SdkRecursiveMutex g_MovieMakerMutex;
        nn::util::optional<MovieMakerState> g_MovieMakerState = nn::util::nullopt;
    }

    size_t GetMovieMakerRequiredWorkMemorySize() NN_NOEXCEPT
    {
        return 96 * 1024 * 1024;
    }

    size_t GetMovieMakerRequiredWorkMemoryAlignment() NN_NOEXCEPT
    {
        return nn::os::MemoryPageSize;
    }

    bool IsMovieMakerInitialized() NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);

        return g_MovieMakerState != nn::util::nullopt;
    }

    bool IsMovieMakerRunning() NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);

        if(g_MovieMakerState == nn::util::nullopt)
        {
            return false;
        }

        return g_MovieMakerState->m_IsMovieMakerRunning;
    }

    namespace {
        nn::Result InitializeMovieMakerImpl(nn::sf::SharedPointer<nn::grcsrv::IMovieMaker>&& pMovieMakerProxy, uint64_t hLayer, bool isRecorderManaged) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::grc::InitializeForApplication(std::move(pMovieMakerProxy))
            );

            nn::grc::MovieLayer* pLayer = {};
            // NOTE:
            //   ここの Width と Height はレイヤに設定されるデフォルト値の初期値として使われる。
            // フレームバッファの解像度を指定する口がないグラフィクス API （ OpenGL のこと）の場合、
            // この値でフレームバッファが作成される。
            // それ以外（ NVN のこと）の場合にはこの値は参照されないので適当に LCD の解像度で設定しておく。
            //
            // OpenGL 対応する場合、ここで設定するよりは GetMovieMakerNativeWindow でアプリからサイズを受け取って
            // デフォルト値を上書きする方が筋が良い気がする。
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::grc::OpenMovieLayer(&pLayer, hLayer, 1280, 720)
            );

            g_MovieMakerState.emplace();
            MovieMakerState& state = *g_MovieMakerState;
            state.m_LayerHandle = hLayer;
            state.m_pLayer = pLayer;
            state.m_IsRecorderManaged = isRecorderManaged;
            state.m_IsMovieMakerRunning = false;

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::grc::InitializeFinishMovieFileReadyEvent(&state.m_FinishMovieFileReadyEvent, pLayer)
            );
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::grc::InitializeMovieAudioEncodeReadyEvent(&state.m_EncodeAudioReadyEvent, pLayer)
            );

            NN_RESULT_SUCCESS;
        }

        sf::SharedPointer<am::service::IMovieMaker> g_pAmMovieMaker;
    }

    nn::Result InitializeMovieMaker(void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
    {
    retry:
        std::unique_lock<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);

        NN_SDK_REQUIRES(!IsMovieMakerInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
        NN_SDK_REQUIRES_ALIGNED(pWorkMemory, GetMovieMakerRequiredWorkMemoryAlignment());
        NN_SDK_REQUIRES_GREATER_EQUAL(workMemorySize, GetMovieMakerRequiredWorkMemorySize());

        os::TransferMemory transferMemory(pWorkMemory, workMemorySize, os::MemoryPermission_None);
        decltype(g_pAmMovieMaker) pAmMovieMaker;
        auto result = am::GetApplicationFunctions()->CreateMovieMaker(&pAmMovieMaker, sf::NativeHandle(transferMemory.Detach(), true), workMemorySize);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(grc::ResultGrcBusy)
            {
                lock.unlock();
                os::SleepThread(TimeSpan::FromMilliSeconds(100));
                goto retry;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(ConvertMovieMakerError(result));
            }
        NN_RESULT_END_TRY

        nn::sf::SharedPointer<nn::grcsrv::IMovieMaker> pMovieMakerProxy;
        uint64_t hLayer = 0;
        NN_RESULT_DO(ConvertMovieMakerError(pAmMovieMaker->GetGrcMovieMaker(&pMovieMakerProxy)));
        NN_RESULT_DO(ConvertMovieMakerError(pAmMovieMaker->GetLayerHandle(&hLayer)));

        NN_RESULT_DO(ConvertMovieMakerError(InitializeMovieMakerImpl(std::move(pMovieMakerProxy), hLayer, true)));

        g_pAmMovieMaker = pAmMovieMaker;
        NN_RESULT_SUCCESS;
    }

    nn::Result InitializeMovieMakerInternal(nn::sf::SharedPointer<nn::grcsrv::IMovieMaker>&& pMovieMakerProxy, uint64_t hLayer) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        NN_SDK_REQUIRES(!IsMovieMakerInitialized());

        return InitializeMovieMakerImpl(std::move(pMovieMakerProxy), hLayer, false);
    }

    void FinalizeMovieMaker() NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        NN_SDK_REQUIRES(IsMovieMakerInitialized());

        if(g_MovieMakerState->m_IsMovieMakerRunning)
        {
            AbortMovieMaker();
        }

        nn::grc::FinalizeMovieAudioEncodeReadyEvent(&g_MovieMakerState->m_EncodeAudioReadyEvent, g_MovieMakerState->m_pLayer);
        nn::grc::FinalizeFinishMovieFileReadyEvent(&g_MovieMakerState->m_FinishMovieFileReadyEvent, g_MovieMakerState->m_pLayer);
        nn::grc::CloseMovieLayer(g_MovieMakerState->m_pLayer);

        if(g_MovieMakerState->m_IsRecorderManaged)
        {
            // TODO: AM 経由で grc を Finalize
        }

        g_MovieMakerState = nn::util::nullopt;

        nn::grc::FinalizeForApplication();

        g_pAmMovieMaker.Reset();
    }

    //------------------------------------------------------------------------------------

    nn::Result GetMovieMakerNativeWindow(nn::vi::NativeWindowHandle* pOutNativeWindow) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        NN_SDK_REQUIRES(IsMovieMakerInitialized());

        nn::vi::NativeWindowHandle hWindow = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::GetVideoNativeWindow(&hWindow, g_MovieMakerState->m_pLayer));

        *pOutNativeWindow = hWindow;
        NN_RESULT_SUCCESS;
    }

    nn::Result PrecheckToStartMovieMaker(uint64_t size) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        NN_SDK_REQUIRES(IsMovieMakerInitialized());
        NN_SDK_REQUIRES(!IsMovieMakerRunning());

        NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
        g_LibraryState.EnsureAlbumAvailable();

        NN_RESULT_DO( ConvertCapsrvResult(capsrv::PrecheckToCreateContentsForApplication(capsrv::AlbumFileContents_ExtraMovie, size)) );
        NN_RESULT_SUCCESS;
    }

    nn::Result StartMovieMaker(const MovieMakerMovieParameter& param) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        NN_SDK_REQUIRES(IsMovieMakerInitialized());
        NN_SDK_REQUIRES(!IsMovieMakerRunning());

        nn::grc::MovieMakerMovieParameter paramIn = {};
        std::memcpy(&paramIn, &param, sizeof(param));
        NN_SDK_REQUIRES(paramIn.CheckParameterValid(), "Invalid MovieMakerMovieParameter");

        NN_RESULT_DO(ConvertMovieMakerError(
            nn::grc::StartMovieFile(g_MovieMakerState->m_pLayer, paramIn)
        ));

        g_MovieMakerState->m_IsMovieMakerRunning = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result FinishMovieMaker() NN_NOEXCEPT
    {
        return FinishMovieMaker(nullptr, 0, nullptr, 0, ImageSize_1280x720);
    }

    nn::Result FinishMovieMaker(const void* pUserData, size_t userDataSize) NN_NOEXCEPT
    {
        return FinishMovieMaker(pUserData, userDataSize, nullptr, 0, ImageSize_1280x720);
    }

    nn::Result FinishMovieMaker(const void* pUserData, size_t userDataSize, const void* pImageData, size_t imageDataSize, ImageSize imageSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(userDataSize <= album::AlbumUserDataSizeMax);
        NN_SDK_REQUIRES(imageSize == album::ImageSize_1280x720);

        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        NN_SDK_REQUIRES(IsMovieMakerInitialized());
        NN_SDK_REQUIRES(IsMovieMakerRunning());

        nn::Result result;

        result = nn::grc::RequestFinishMovieFile(g_MovieMakerState->m_pLayer);
        if(result.IsFailure())
        {
            nn::grc::AbortMovieFile(g_MovieMakerState->m_pLayer);
            g_MovieMakerState->m_IsMovieMakerRunning = false;
            return ConvertMovieMakerError(result);
        }

        nn::os::WaitSystemEvent(&g_MovieMakerState->m_FinishMovieFileReadyEvent);

        // userDataSize == 0 ならアプリ任意データなし
        sf::InBuffer userData(reinterpret_cast<const char*>(pUserData), userDataSize);
        // imageDataSize == 0 ならサムネイル独自指定なし
        sf::InBuffer thumbnailImage(reinterpret_cast<const char*>(pImageData), imageDataSize);
        result = nn::grc::CompleteFinishMovieFile(g_MovieMakerState->m_pLayer, userData, thumbnailImage, imageSize);
        if(result.IsFailure())
        {
            nn::grc::AbortMovieFile(g_MovieMakerState->m_pLayer);
            g_MovieMakerState->m_IsMovieMakerRunning = false;
            return ConvertMovieMakerError(result);
        }

        g_MovieMakerState->m_IsMovieMakerRunning = false;
        NN_RESULT_SUCCESS;
    }

    void AbortMovieMaker() NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        if(!IsMovieMakerRunning())
        {
            return;
        }

        nn::grc::AbortMovieFile(g_MovieMakerState->m_pLayer);
        g_MovieMakerState->m_IsMovieMakerRunning = false;
    }

    nn::Result CheckMovieMakerError() NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
        NN_SDK_REQUIRES(IsMovieMakerInitialized());
        NN_SDK_REQUIRES(IsMovieMakerRunning());

        return ConvertMovieMakerError(nn::grc::CheckMovieFileError(g_MovieMakerState->m_pLayer));
    }

    nn::Result EncodeMovieMakerAudioSample(const void* buffer, size_t size) NN_NOEXCEPT
    {
        ptrdiff_t offset = 0;
        size_t remain = size;
        while(remain > 0)
        {
            nn::os::WaitSystemEvent(&g_MovieMakerState->m_EncodeAudioReadyEvent);

            size_t encodedSize = 0;
            {
                std::lock_guard<decltype(g_MovieMakerMutex)> lock(g_MovieMakerMutex);
                NN_SDK_REQUIRES(IsMovieMakerInitialized());
                NN_SDK_REQUIRES(IsMovieMakerRunning());

                NN_RESULT_DO(ConvertMovieMakerError(nn::grc::EncodeMovieAudioSample(
                    &encodedSize,
                    g_MovieMakerState->m_pLayer,
                    reinterpret_cast<const char*>(buffer) + offset,
                    remain
                )));
            }
            offset += encodedSize;
            remain -= encodedSize;
            //NN_SDK_LOG("audio encoded %lld bytes (size=%lld, remain %lld)\n", encodedSize, size, remain);
        }

        NN_RESULT_SUCCESS;
    }

    void SetMovieMakerErrorConversionEnabled(bool enabled) NN_NOEXCEPT
    {
        g_IsMovieMakerErrorConversionEnabled = enabled;
    }

    //-----------------------------------------------------------------

    NN_STATIC_ASSERT(sizeof(nn::album::MovieMakerMovieParameter) == sizeof(nn::grc::MovieMakerMovieParameter));
    NN_STATIC_ASSERT(NN_ALIGNOF(nn::album::MovieMakerMovieParameter) >= NN_ALIGNOF(nn::grc::MovieMakerMovieParameter));

    namespace {

        nn::grc::MovieMakerMovieParameter& GetData(nn::album::MovieMakerMovieParameter& value) NN_NOEXCEPT
        {
            return reinterpret_cast<nn::grc::MovieMakerMovieParameter&>(value);
        }

        const nn::grc::MovieMakerMovieParameter& GetData(const nn::album::MovieMakerMovieParameter& value) NN_NOEXCEPT
        {
            return reinterpret_cast<const nn::grc::MovieMakerMovieParameter&>(value);
        }
    }

    MovieMakerMovieParameter MovieMakerMovieParameter::GetDefaultValue() NN_NOEXCEPT
    {
        return MovieMakerMovieParameter();
    }

    MovieMakerMovieParameter::MovieMakerMovieParameter() NN_NOEXCEPT
    {
        GetData(*this) = nn::grc::MovieMakerMovieParameter::GetDefaultValue();
    }

    int MovieMakerMovieParameter::GetVideoBitRate() const NN_NOEXCEPT
    {
        return GetData(*this).videoBitRate;
    }

    void MovieMakerMovieParameter::SetVideoBitRate(int value) NN_NOEXCEPT
    {
        GetData(*this).videoBitRate = value;
    }

    int MovieMakerMovieParameter::GetVideoWidth() const NN_NOEXCEPT
    {
        return GetData(*this).videoWidth;
    }

    void MovieMakerMovieParameter::SetVideoWidth(int value) NN_NOEXCEPT
    {
        GetData(*this).videoWidth = value;
    }

    int MovieMakerMovieParameter::GetVideoHeight() const NN_NOEXCEPT
    {
        return GetData(*this).videoHeight;
    }

    void MovieMakerMovieParameter::SetVideoHeight(int value) NN_NOEXCEPT
    {
        GetData(*this).videoHeight = value;
    }

    int MovieMakerMovieParameter::GetVideoFrameRate() const NN_NOEXCEPT
    {
        return GetData(*this).videoFrameRate;
    }

    void MovieMakerMovieParameter::SetVideoFrameRate(int value) NN_NOEXCEPT
    {
        GetData(*this).videoFrameRate = value;
    }

    int MovieMakerMovieParameter::GetVideoKeyFrameInterval() const NN_NOEXCEPT
    {
        return GetData(*this).videoFrameCountBetweenIdr;
    }

    void MovieMakerMovieParameter::SetVideoKeyFrameInterval(int value) NN_NOEXCEPT
    {
        GetData(*this).videoFrameCountBetweenIdr = value;
    }

    NN_STATIC_ASSERT(static_cast<int>(album::ImageOrientation_None) == static_cast<int>(capsrv::ScreenShotOrientation_Default));
    NN_STATIC_ASSERT(static_cast<int>(album::ImageOrientation_Rotate90) == static_cast<int>(capsrv::ScreenShotOrientation_Rotate90));
    NN_STATIC_ASSERT(static_cast<int>(album::ImageOrientation_Rotate180) == static_cast<int>(capsrv::ScreenShotOrientation_Rotate180));
    NN_STATIC_ASSERT(static_cast<int>(album::ImageOrientation_Rotate270) == static_cast<int>(capsrv::ScreenShotOrientation_Rotate270));

    void MovieMakerMovieParameter::SetVideoImageOrientation(ImageOrientation orientation) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( orientation == ImageOrientation_None
                      || orientation == ImageOrientation_Rotate90
                      || orientation == ImageOrientation_Rotate180
                      || orientation == ImageOrientation_Rotate270 );

        GetData(*this).videoImageOrientation = orientation;
    }

    int MovieMakerMovieParameter::GetAudioBitRate() const NN_NOEXCEPT
    {
        return GetData(*this).audioBitRate;
    }

    void MovieMakerMovieParameter::SetAudioBitRate(int value) NN_NOEXCEPT
    {
        GetData(*this).audioBitRate = value;
    }

    int MovieMakerMovieParameter::GetAudioSampleRate() const NN_NOEXCEPT
    {
        return GetData(*this).audioSampleRate;
    }

    void MovieMakerMovieParameter::SetAudioSampleRate(int value) NN_NOEXCEPT
    {
        GetData(*this).audioSampleRate = value;
    }

    int MovieMakerMovieParameter::GetAudioChannelCount() const NN_NOEXCEPT
    {
        return GetData(*this).audioChannelCount;
    }

    void MovieMakerMovieParameter::SetAudioChannelCount(int value) NN_NOEXCEPT
    {
        GetData(*this).audioChannelCount = value;
    }

    nn::audio::SampleFormat MovieMakerMovieParameter::GetAudioSampleFormat() const NN_NOEXCEPT
    {
        return static_cast<nn::audio::SampleFormat>(GetData(*this).audioFormat);
    }

    void MovieMakerMovieParameter::SetAudioSampleFormat(nn::audio::SampleFormat value) NN_NOEXCEPT
    {
        GetData(*this).audioFormat = value;
    }

    //-----------------------------------------------------------------

}}

