﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/os.h>
#include <winext/cafe/mem/list.h>

#define OBJ_TO_LINK(list,obj)   ((MEMLink*)(((u32)(obj))+(list)->offset))

namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
  Name:         MEMInitList

  Description:  リスト構造体を初期化します。

  Arguments:    list:   リスト構造体へのポインタ。
                offset: リストに繋げたい構造体の中に存在する、Link型の
                        メンバ変数の構造体内でのオフセットを指定します。
                        stddef.hで定義されているoffsetofマクロを使用すると、
                        便利です。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMInitList( MEMList* list, u16 offset )
{
    //ASSERT( list != NULL );

    list->headObject = NULL;
    list->tailObject = NULL;
    list->numObjects = 0;
    list->offset     = offset;
}

/*---------------------------------------------------------------------------*
  Name:         SetFirstObject_                                      [static]

  Description:  最初のオブジェクトをリストに加えます。

  Arguments:    list:   リスト構造体へのポインタ。
                object: リストに繋げたいオブジェクトへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static void
SetFirstObject_( MEMList* list, void* object )
{
    MEMLink* link;

    //ASSERT( list   != NULL );
    //ASSERT( object != NULL );

    link = OBJ_TO_LINK(list, object);

    link->nextObject = NULL;
    link->prevObject = NULL;
    list->headObject = object;
    list->tailObject = object;
    list->numObjects++;
}

/*---------------------------------------------------------------------------*
  Name:         MEMAppendListObject

  Description:  オブジェクトをリストの最後に追加します。

  Arguments:    list:   リスト構造体へのポインタ。
                object: リストに繋げたいオブジェクトへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMAppendListObject( MEMList* list, void* object )
{
    //ASSERT( list   != NULL );
    //ASSERT( object != NULL );

    if (list->headObject == NULL)
    {
        // リストが空の時。
        SetFirstObject_(list, object);
    }
    else
    {
        MEMLink* link = OBJ_TO_LINK(list, object);

        link->prevObject = list->tailObject;
        link->nextObject = NULL;

        OBJ_TO_LINK(list, list->tailObject)->nextObject = object;
        list->tailObject = object;
        list->numObjects++;
    }
}

/*---------------------------------------------------------------------------*
  Name:         MEMPrependListObject

  Description:  オブジェクトをリストの先頭に挿入します。

  Arguments:    list:   リスト構造体へのポインタ。
                object: リストに繋げたいオブジェクトへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMPrependListObject( MEMList* list, void* object )
{
    //ASSERT( list   != NULL );
    //ASSERT( object != NULL );

    if (list->headObject == NULL)
    {
        // リストが空の時。
        SetFirstObject_(list, object);
    }
    else
    {
        MEMLink* link = OBJ_TO_LINK(list, object);

        link->prevObject = NULL;
        link->nextObject = list->headObject;

        OBJ_TO_LINK(list, list->headObject)->prevObject = object;
        list->headObject = object;
        list->numObjects++;
    }
}

/*---------------------------------------------------------------------------*
  Name:         MEMInsertListObject

  Description:  オブジェクトを指定された位置に挿入します。オブジェクトは、
                targetで指定されたオブジェクトの前に挿入されます。挿入先が指
                定されていない場合（targetがNULLの場合）、オブジェクトはリス
                トの最後に追加されます。

  Arguments:    list:   リスト構造体へのポインタ。
                target: 挿入したい位置にあるオブジェクトへのポインタ。
                object: リストに繋げたいオブジェクトへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMInsertListObject( MEMList* list, void* target, void* object )
{
    //ASSERT( list   != NULL );
    //ASSERT( object != NULL );

    if (target == NULL)
    {
        // targetが指定されていない場合は、AppendListObject()と同じ。
        MEMAppendListObject(list, object);
    }
    else if (target == list->headObject)
    {
        // targetがリストの先頭である場合はPrependListObject()と同じ。
        MEMPrependListObject(list, object);
    }
    else
    {
        MEMLink* link    = OBJ_TO_LINK(list, object);
        void* prevObj = OBJ_TO_LINK(list, target)->prevObject;
        MEMLink* prevLnk = OBJ_TO_LINK(list, prevObj);

        link->prevObject    = prevObj;
        link->nextObject    = target;
        prevLnk->nextObject = object;
        OBJ_TO_LINK(list, target)->prevObject = object;
        list->numObjects++;
    }
}

/*---------------------------------------------------------------------------*
  Name:         MEMRemoveListObject

  Description:  オブジェクトをリストから削除します。

  Arguments:    list:   リスト構造体へのポインタ。
                object: リストから削除したいオブジェクトへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMRemoveListObject( MEMList* list, void* object )
{
    MEMLink* link;

    //ASSERT( list   != NULL );
    //ASSERT( object != NULL );

    link = OBJ_TO_LINK(list, object);

    if (link->prevObject == NULL)
    {
        list->headObject = link->nextObject;
    }
    else
    {
        OBJ_TO_LINK(list, link->prevObject)->nextObject = link->nextObject;
    }
    if (link->nextObject == NULL)
    {
        list->tailObject = link->prevObject;
    }
    else
    {
        OBJ_TO_LINK(list, link->nextObject)->prevObject = link->prevObject;
    }
    link->prevObject = NULL;
    link->nextObject = NULL;
    list->numObjects--;
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetNextListObject

  Description:  objectで指定されたオブジェクトの次に繋がれているオブジェクト
                を返します。objectにNULLが指定されていた場合には、リストの先
                頭に繋がれているオブジェクトを返します。

  Arguments:    list:   リスト構造体へのポインタ。
                object: リスト中のオブジェクトへのポインタ。

  Returns:      指定されたオブジェクトの次のオブジェクトへのポインタを返しま
                す。もし、次のオブジェクトが無ければ、NULLを返します。
 *---------------------------------------------------------------------------*/
void*
MEMGetNextListObject( MEMList* list, void* object )
{
    //ASSERT( list != NULL );

    if (object == NULL)
    {
        return list->headObject;
    }
    return OBJ_TO_LINK(list, object)->nextObject;
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetPrevListObject

  Description:  objectで指定されたオブジェクトの前に繋がれているオブジェクト
                を返します。objectにNULLが指定されていた場合には、リストの後
                尾に繋がれているオブジェクトを返します。

  Arguments:    list:   リスト構造体へのポインタ。
                object: リスト中のオブジェクトへのポインタ。

  Returns:      指定されたオブジェクトの前のオブジェクトへのポインタを返しま
                す。もし、前のオブジェクトが無ければ、NULLを返します。
 *---------------------------------------------------------------------------*/
void*
MEMGetPrevListObject( MEMList* list, void* object )
{
    //ASSERT( list != NULL );

    if (object == NULL)
    {
        return list->tailObject;
    }
    return OBJ_TO_LINK(list, object)->prevObject;
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetNthListObject

  Description:  リストのＮ番目に繋がれているオブジェクトへのポインタを返しま
                す。先頭から順番にリストをたどる為、リストの後部に繋がれてい
                オブジェクト程、時間がかかります。

  Arguments:    index:  オブジェクトのインデックス。

  Returns:      オブジェクトへのポインタを返します。もし、指定されたインデッ
                クスのオブジェクトが無かった場合には、NULLが返ります。
 *---------------------------------------------------------------------------*/
void*
MEMGetNthListObject( MEMList* list, u16 index )
{
    int      count  = 0;
    MEMLink* object = NULL;

    //ASSERT( list != NULL );

    while ((object = (MEMLink*)MEMGetNextListObject(list, object)) != NULL)
    {
        if (index == count)
        {
            return object;
        }
        count++;
    }
    return NULL;
}

/*---------------------------------------------------------------------------*
  Name:         MEMCountListNode

  Description:  リストに繋がれているオブジェクトの数をカウントします。先頭か
                ら順番にリストをカウントする為、リストが長いと時間がかかりま
                す。

  Arguments:    list:   リスト構造体へのポインタ。

  Returns:      リストに繋がれているオブジェクトの数。
 *---------------------------------------------------------------------------*/
#if 0
int
MEMCountListNode(MEMList* list)
{
    int   count  = 0;
    Node* object = NULL;

    ASSERT( list != NULL );

    while ((object = GetNextListObject(list, object)) != NULL)
    {
        count++;
    }
    return count;
}
#endif

} // namespace winext
} // namespace internal
} // namespace nw
