﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ut/ut_FsFileStream.h>
#include <nw/assert.h>

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
  // TODO: NintendoSdk 対応後、このコメントを削除してください。
  using namespace nw::internal::winext;
#elif defined( NW_PLATFORM_ANDROID ) || defined(NW_PLATFORM_IOS)
  using namespace nw::internal::winext;
#endif

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
FsFileStream::FsFileStream()
: m_Available( false ),
  m_DuplicateFlag( false ),
  m_FilePos( 0 ),
  m_FileSize( 0 )
{
}

//---------------------------------------------------------------------------
FsFileStream::FsFileStream( const FsFileStream* fileStream )
: m_Available( true ),
  m_DuplicateFlag( true ),
  m_Client( fileStream->m_Client ),
  m_CmdBlock( fileStream->m_CmdBlock ),
  m_FileHandle( fileStream->m_FileHandle ),
  m_FileStat( fileStream->m_FileStat ),
  m_FilePos( 0 ),
  m_FileSize( fileStream->m_FileSize )
{
}

//---------------------------------------------------------------------------
bool
FsFileStream::Open( FSClient* client, FSCmdBlock* cmdBlock, const char* filePath, const char* mode )
{
    if ( m_Available )
    {
        Close();
    }

    FSStatus status = FSOpenFile(
        client,
        cmdBlock,
        filePath,
        mode,
        &m_FileHandle,
        FS_RET_ALL_ERROR
    );

    if ( status != FS_STATUS_OK )
    {
        NW_WARNING( status == FS_STATUS_OK, "FsFileStream::Open is Failed: %d\n", status );
        return false;
    }

    status = FSGetStatFile(
        client,
        cmdBlock,
        m_FileHandle,
        &m_FileStat,
        FS_RET_ALL_ERROR
    );

    if ( status != FS_STATUS_OK )
    {
        status = FSCloseFile(
            client,
            cmdBlock,
            m_FileHandle,
            FS_RET_ALL_ERROR
        );
        NW_ASSERT( status == FS_STATUS_OK );
        return false;
    }

    m_Client = client;
    m_CmdBlock = cmdBlock;
    m_FilePos = 0;
    m_FileSize = m_FileStat.size;
    m_Available = true;

    return true;
}

//---------------------------------------------------------------------------
void
FsFileStream::Close()
{
    if ( ! m_DuplicateFlag )
    {
        FSStatus status = FSCloseFile(
            m_Client,
            m_CmdBlock,
            m_FileHandle,
            FS_RET_ALL_ERROR
        );
        NW_ASSERT( status == FS_STATUS_OK );
    }

    m_Available = false;
}

//---------------------------------------------------------------------------
s32
FsFileStream::Read( void* buf, u32 length )
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanRead(), "Stream don't support Read function\n" );

    FSStatus status = FSReadFile(
        m_Client,
        m_CmdBlock,
        buf,
        1,
        length,
        m_FileHandle,
        0,
        FS_RET_ALL_ERROR
    );

    if ( status >= FS_STATUS_OK )
    {
        m_FilePos += status;
    }

    return status;
}

//---------------------------------------------------------------------------
bool
FsFileStream::ReadAsync(
    void*               buf,
    u32                 length,
    IoStreamCallback    callback,
    void*               arg
)
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanRead(), "Stream don't support Read function\n" );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );
    (void)buf;
    (void)length;
    (void)callback;
    (void)arg;
    return false;
}

//---------------------------------------------------------------------------
s32
FsFileStream:: Write( const void* buf, u32 length )
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanWrite(), "Stream don't support Write function\n" );

    FSStatus status = FSWriteFile(
        m_Client,
        m_CmdBlock,
        const_cast<void*>(buf),
        1,
        length,
        m_FileHandle,
        0,
        FS_RET_ALL_ERROR
    );

    if ( status >= FS_STATUS_OK )
    {
        m_FilePos += status;

        if ( m_FileSize < m_FilePos )
        {
            m_FileSize = m_FilePos;
        }
    }

    return status;
}

//---------------------------------------------------------------------------
bool
FsFileStream::WriteAsync(
    const void*         buf,
    u32                 length,
    IoStreamCallback    callback,
    void*               arg
)
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanWrite(), "Stream don't support Write function\n" );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );
    (void)buf;
    (void)length;
    (void)callback;
    (void)arg;
    return false;
}

//---------------------------------------------------------------------------
s32
FsFileStream::WaitAsync()
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );

    return -1;
}

//---------------------------------------------------------------------------
bool
FsFileStream::CanRead() const
{
    if ( ! m_Available ) { return false; }
#if 0
    return ( m_FileStat.permission & FS_STAT_PERM_READ ) ? true : false ;
#else
    return true;
#endif
}

//---------------------------------------------------------------------------
bool
FsFileStream::CanWrite() const
{
    if ( ! m_Available ) { return false; }
#if 0
    return ( m_FileStat.permission & FS_STAT_PERM_WRITE ) ? true : false ;
#else
    return true;
#endif
}

//---------------------------------------------------------------------------
u32
FsFileStream::GetSize() const
{
    NW_ASSERT( m_Available );

    return m_FileSize;
}

//---------------------------------------------------------------------------
bool
FsFileStream::Seek( s32 offset, u32 origin )
{
    NW_ASSERT( m_Available );

    FSFilePosition pos;

    switch( origin )
    {
    case FILE_STREAM_SEEK_BEGIN:
        pos = offset;
        break;
    case FILE_STREAM_SEEK_CURRENT:
        pos = m_FilePos + offset;
        break;
    case FILE_STREAM_SEEK_END:
        pos = m_FileSize - offset;
        break;
    default:
        NW_ASSERTMSG( false, "FsFileStream::Seek Invalid origin parameter." );
        return false;
    }

    FSStatus status = FSSetPosFile(
        m_Client,
        m_CmdBlock,
        m_FileHandle,
        pos,
        FS_RET_ALL_ERROR
    );

    if ( status != FS_STATUS_OK )
    {
        NW_WARNING( status == FS_STATUS_OK, "FSSetPosFile is Failed: status %d pos %d", status, pos );
        return false;
    }

    m_FilePos = pos;

    return true;
}

//---------------------------------------------------------------------------
void
FsFileStream::Cancel()
{
    NW_ASSERTMSG( CanCancel(), "Stream don't support Cancel function\n" );
}

//---------------------------------------------------------------------------
bool
FsFileStream::CancelAsync( IoStreamCallback callback, void* arg )
{
    NW_ASSERT( m_Available );
    NW_ASSERTMSG( CanCancel(), "Stream don't support Cancel function\n" );
    NW_ASSERTMSG( CanAsync(), "Stream don't support Aync function\n" );
    (void)callback;
    (void)arg;
    return false;

}

}   /* namespace ut */
}   /* namespace nw */

