﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/assert.h>
#include <nw/font/font_ResFontBase.h>
#include <nw/ut/ut_Inlines.h>

#include <nw/dev/dev_Profile.h>

typedef u32 TexName;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#include <gl/glew.h>

#ifdef _WIN32
#ifndef WIN32
#define WIN32
#endif
#endif
#pragma warning(push)
#pragma warning(disable:4100)
#pragma warning(disable:4063)
#pragma warning(disable:4005)
#include <nw/font/font_TextureTilingConverter.h>
#pragma warning(pop)
#include <sdk_ver.h>
#if CAFE_OS_SDK_VERSION < 20903
using namespace TexUtils;
#endif
#endif

namespace nw {
namespace font {

namespace {

enum
{
    CMD_TEX_NEAREST = 0,
    CMD_TEX_LINEAR  = 1
};

#define NW_FONT_COMMAND_TEX_WRAP_FILTER( magFilter, minFilter )             \
      ( (magFilter) << 1 | (minFilter) << 2 | ( 0/*isETC1*/ ? 2 : 0 ) << 4  \
    | 0/*wrapT*/ << 8 | 0/*wrapS*/ << 12                                    \
    | 0/*minFilter2*/ << 24 )

}   //     namespace


/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

ResFontBase::ResFontBase()
: Font()
, m_pResource(NULL)
, m_pFontInfo(NULL)
, m_pKerningTable(NULL)
{
    EnableLinearFilter(true, true);     // m_Filter
}

ResFontBase::~ResFontBase()
{
}



/* ------------------------------------------------------------------------
        構築/破棄
   ------------------------------------------------------------------------ */

void
ResFontBase::SetResourceBuffer(
    void*               pUserBuffer,
    FontInformation*    pFontInfo,
    FontKerningTable*   pKerningTable
)
{
    NW_ASSERT_VALID_POINTER(pUserBuffer);
    NW_ASSERT_VALID_POINTER(pFontInfo);
    NW_ASSERT(m_pResource == NULL);
    NW_ASSERT(m_pFontInfo == NULL);

    m_pResource = pUserBuffer;
    m_pFontInfo = pFontInfo;
    m_pKerningTable = pKerningTable;
}

void*
ResFontBase::RemoveResourceBuffer()
{
    // テクスチャ名の削除
    // m_pResourceが指す内容に依存するので
    // m_pResourceをクリアする前に行う。
    DeleteTextureNames();

    void* pUserData = m_pResource;

    m_pResource = NULL;
    m_pFontInfo = NULL;
    m_pKerningTable = NULL;

    return pUserData;
}




/* ------------------------------------------------------------------------
        フォント全体情報アクセサ
   ------------------------------------------------------------------------ */

int
ResFontBase::GetWidth() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->width;
}

int
ResFontBase::GetHeight() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->height;
}

int
ResFontBase::GetAscent() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->ascent;
}

int
ResFontBase::GetDescent() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->height - m_pFontInfo->ascent;
}

int
ResFontBase::GetBaselinePos() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->pGlyph->baselinePos;
}

int
ResFontBase::GetCellHeight() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->pGlyph->cellHeight;
}

int
ResFontBase::GetCellWidth() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->pGlyph->cellWidth;
}

int
ResFontBase::GetMaxCharWidth() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->pGlyph->maxCharWidth;
}

Font::Type
ResFontBase::GetType() const
{
    return TYPE_RESOURCE;
}

TexFmt
ResFontBase::GetTextureFormat() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return static_cast<TexFmt>(m_pFontInfo->pGlyph->sheetFormat & FONT_SHEET_FORMAT_MASK);
}

int
ResFontBase::GetLineFeed() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->linefeed;
}

const CharWidths
ResFontBase::GetDefaultCharWidths() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->defaultWidth;
}

void
ResFontBase::SetDefaultCharWidths( const CharWidths& widths )
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    NW_ASSERT_VALID_POINTER( &widths );
    m_pFontInfo->defaultWidth = widths;
}

bool
ResFontBase::SetAlternateChar( CharCode c )
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    GlyphIndex index = FindGlyphIndex(c);

    if (index != GLYPH_INDEX_NOT_FOUND)
    {
        m_pFontInfo->alterCharIndex = index;
        return true;
    }

    return false;
}

void
ResFontBase::SetLineFeed( int linefeed )
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    NW_FONT_MINMAX_ASSERT(linefeed, SCHAR_MIN, SCHAR_MAX);
    m_pFontInfo->linefeed = static_cast<s8>(linefeed);
}




/* ------------------------------------------------------------------------
        文字単体情報アクセサ
   ------------------------------------------------------------------------ */

int
ResFontBase::GetCharWidth( CharCode c ) const
{
    return GetCharWidths(c).charWidth;
}

const CharWidths
ResFontBase::GetCharWidths( CharCode c ) const
{
    GlyphIndex index = GetGlyphIndex(c);
    return GetCharWidthsFromIndex(index);
}

void
ResFontBase::GetGlyph( Glyph* glyph, CharCode c ) const
{
    GlyphIndex index = GetGlyphIndex(c);
    GetGlyphFromIndex(glyph, index);
}

bool
ResFontBase::HasGlyph( CharCode c ) const
{
    return ( GLYPH_INDEX_NOT_FOUND != FindGlyphIndex(c) );
}

int
ResFontBase::GetKerning(CharCode c0, CharCode c1) const
{
    if (m_pKerningTable == NULL || ( ! IsEnableKerning()))
    {
        return 0;
    }

    u32 firstWordNum = m_pKerningTable->firstWordNum;
    const KerningFirstTableElem* firstTable = m_pKerningTable->firstTable;

    const KerningSecondTable* secondTable = NULL;
    // firstTableからc0を二分探索
    {
        u32 middle;
        u32 start = 0;
        u32 end = firstWordNum;
        for(;;)
        {
            middle = ( start + end ) / 2;
            u16 firstWord = firstTable[ middle ].firstWord;
            if ( firstWord == c0 )
            {
                break;
            }
            else if ( firstWord < c0 )
            {
                if ( start == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                end = middle;
            }
        }

        u16 offset = firstTable[ middle ].offset;
        secondTable = reinterpret_cast<const KerningSecondTable*>(&reinterpret_cast<const u8*>(m_pKerningTable)[offset * 2]);
    }

    u32 secondWordNum = secondTable->secondWordNum;
    const KerningSecondTableElem* secondTableElems = secondTable->elems;

    // secondTableからc1を二分探索
    {
        u32 middle;
        u32 start = 0;
        u32 end = secondWordNum;
        for(;;)
        {
            middle = ( start + end ) / 2;
            u16 secondWord = secondTableElems[ middle ].secondWord;
            if ( secondWord == c1 )
            {
                // 見つかった。カーニングの値を返す
                return secondTableElems[ middle ].kerningValue;
            }
            else if ( secondWord < c1 )
            {
                if ( start == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                end = middle;
            }
        }
    }
}


/* ------------------------------------------------------------------------
        文字ストリーム
   ------------------------------------------------------------------------ */

CharacterCode
ResFontBase::GetCharacterCode() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);

    return static_cast<CharacterCode>(m_pFontInfo->characterCode);
}



/* ------------------------------------------------------------------------
        グリフインデックス
   ------------------------------------------------------------------------ */

ResFontBase::GlyphIndex
ResFontBase::GetGlyphIndex( CharCode c ) const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    GlyphIndex index = FindGlyphIndex(c);
    return (index != GLYPH_INDEX_NOT_FOUND) ? index: m_pFontInfo->alterCharIndex;
}

ResFontBase::GlyphIndex
ResFontBase::FindGlyphIndex( CharCode c ) const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);

    const FontCodeMap* pMap = m_pFontInfo->pMap;
    while (pMap != NULL)
    {
        if (pMap->ccodeBegin <= c && c <= pMap->ccodeEnd)
        {
            return FindGlyphIndex(pMap, c);
        }

        pMap = pMap->pNext;
    }

    return GLYPH_INDEX_NOT_FOUND;
}

ResFontBase::GlyphIndex
ResFontBase::FindGlyphIndex(
    const FontCodeMap*  pMap,
    CharCode            c
) const
{
    NW_ASSERT_VALID_POINTER(pMap);
    u16 index = GLYPH_INDEX_NOT_FOUND;

    switch (pMap->mappingMethod)
    {
    //-----------------------------------------------------------
    // インデックス = 文字コード - オフセット
    case FONT_MAPMETHOD_DIRECT:
        {
            u16 offset = pMap->mapInfo[0];
            index = static_cast<u16>(c - pMap->ccodeBegin + offset);
        }
        break;

    //-----------------------------------------------------------
    // インデックス = table[文字コード - 文字コードオフセット]
    case FONT_MAPMETHOD_TABLE:
        {
            const int table_index = c - pMap->ccodeBegin;

            index = pMap->mapInfo[table_index];
        }
        break;

    //-----------------------------------------------------------
    // インデックス = 二分探索(文字コード)
    case FONT_MAPMETHOD_SCAN:
        {
            const CMapInfoScan* const scanInfo
                = reinterpret_cast<const CMapInfoScan*>(pMap->mapInfo);
            const CMapScanEntry* first  = &(scanInfo->entries[0]);
            const CMapScanEntry* last   = &(scanInfo->entries[scanInfo->num - 1]);

            while( first <= last )
            {
                const CMapScanEntry* mid = first + (last - first) / 2;

                if( mid->ccode < c )
                {
                    first = mid + 1;
                }
                else if( c < mid->ccode )
                {
                    last = mid - 1;
                }
                else
                {
                    index = mid->index;
                    break;
                }
            }
        }
        break;

    //-----------------------------------------------------------
    // unknown
    default:
        NW_ASSERTMSG(false, "unknwon MAPMETHOD");
    }

    return index;
}

const CharWidths&
ResFontBase::GetCharWidthsFromIndex( GlyphIndex index ) const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    const FontWidth* pWidth;

    pWidth = m_pFontInfo->pWidth;

    while (pWidth != NULL)
    {
        if (pWidth->indexBegin <= index && index <= pWidth->indexEnd)
        {
            return GetCharWidthsFromIndex( pWidth, index );
        }

        pWidth = pWidth->pNext;
    }

    return m_pFontInfo->defaultWidth;
}

const CharWidths&
ResFontBase::GetCharWidthsFromIndex(
    const FontWidth*    pWidth,
    GlyphIndex          index
) const
{
    NW_ASSERT_VALID_POINTER(pWidth);
    return pWidth->widthTable[index - pWidth->indexBegin];
}

void
ResFontBase::GetGlyphFromIndex(
    Glyph*      glyph,
    GlyphIndex  index
) const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    const FontTextureGlyph& tg = *m_pFontInfo->pGlyph;

    const u32 cellsInASheet = internal::GetCellsInASheet(tg);
    const u32 sheetNo       = index       / cellsInASheet;
    const u32 offsetBytes   = sheetNo * tg.sheetSize;
    const void* pSheet      = tg.sheetImage + offsetBytes;

    glyph->pTexture         = pSheet;
    glyph->widths           = GetCharWidthsFromIndex(index);
    glyph->pTextureObject   = &m_TexObj;
    glyph->sheetIndex       = static_cast<u8>(sheetNo);
    SetGlyphMember(glyph, index, tg);
}

void
ResFontBase::SetGlyphMember(
    Glyph*                  glyph,
    GlyphIndex              index,
    const FontTextureGlyph& tg
)
{
    const u32 cellNo        = index       % internal::GetCellsInASheet(tg);
    const u32 cellUnitX     = cellNo      % tg.sheetRow;
    const u32 cellUnitY     = cellNo      / tg.sheetRow;
    const u32 cellPixelX    = cellUnitX   * (tg.cellWidth  + 1);
    const u32 cellPixelY    = cellUnitY   * (tg.cellHeight + 1);

    glyph->height       = tg.cellHeight;
    glyph->texFormat    = static_cast<TexFmt>(tg.sheetFormat & FONT_SHEET_FORMAT_MASK);
    glyph->texWidth     = tg.sheetWidth;
    glyph->texHeight    = tg.sheetHeight;
    glyph->cellX        = static_cast<u16>(cellPixelX + 1);
    glyph->cellY        = static_cast<u16>(cellPixelY + 1);
}


/* ------------------------------------------------------------------------
        テクスチャフィルタ
   ------------------------------------------------------------------------ */

void
ResFontBase::EnableLinearFilter(
    bool    atSmall,
    bool    atLarge
)
{
    // 同じレジスタに以下に設定される項目がありますが、フォントではサポートしていないため、
    // 常に固定値です。
    // ・テクスチャをラップして使用することはありません。
    // ・テクスチャをミップマップで使用することはありません。
    // ・テクスチャフォーマットとして ETC1(アルファ無し)はサポートしていません。
    // ・テクスチャユニット0がシャドウテクスチャが設定されることはありません。
    const int magFilter = atLarge ? CMD_TEX_LINEAR: CMD_TEX_NEAREST;
    const int minFilter = atSmall ? CMD_TEX_LINEAR: CMD_TEX_NEAREST;
    m_WrapFilter = static_cast<u32>(
        NW_FONT_COMMAND_TEX_WRAP_FILTER(
            magFilter,
            minFilter));

    if (! IsManaging(NULL))
    {
        // 一度管理しているテクスチャオブジェクトを破棄し、テクスチャオブジェクト名を再確保する。
        DeleteTextureNames();
        GenTextureNames();
    }
}

bool
ResFontBase::IsLinearFilterEnableAtSmall() const
{
    return 0 != (m_WrapFilter & (1 << 2));
}

bool
ResFontBase::IsLinearFilterEnableAtLarge() const
{
    return 0 != (m_WrapFilter & (1 << 1));
}

u32
ResFontBase::GetTextureWrapFilterValue() const
{
    return m_WrapFilter;
}

bool
ResFontBase::IsColorBlackWhiteInterpolationEnabled() const
{
    return m_TexObj.IsColorBlackWhiteInterpolationEnabled();
}

void
ResFontBase::SetColorBlackWhiteInterpolationEnabled(bool flag)
{
    m_TexObj.SetColorBlackWhiteInterpolationEnabled(flag);
}

bool
ResFontBase::IsBorderEffectEnabled() const
{
    return m_pFontInfo->fontType == FONT_TYPE_PACKED_TEXTURE;
}

int
ResFontBase::GetActiveSheetNum() const
{
    NW_ASSERT_VALID_POINTER(m_pFontInfo);
    return m_pFontInfo->pGlyph->sheetNum;
}


// シート毎のテクスチャオブジェクトの初期化
void
ResFontBase::GenTextureNames()
{
    internal::TextureObject* texObj = GetTextureObject();

    const FontTextureGlyph& tg = *m_pFontInfo->pGlyph;
    const int sheetNum = GetActiveSheetNum();
    // TextureObjectクラスのm_SheetNumがu8なので、シートの枚数は255までしか扱えない
    NW_ASSERT_MAX(sheetNum, 256);
    u32 offsetBytes = 0;
    const TexName texName = 0;
    const void *const pImage = tg.sheetImage + offsetBytes;
    const TexFmt format = static_cast<TexFmt>(tg.sheetFormat);
    texObj->Set(texName, pImage, format, tg.sheetWidth, tg.sheetHeight, static_cast<u8>(sheetNum), true);

    LoadTexture(texObj);
}

void
ResFontBase::DeleteTextureNames()
{
    internal::TextureObject* texObj = GetTextureObject();

    const TexName texName = texObj->GetName();
    if (texName != 0)
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        glDeleteTextures(1, &texName);
#endif
        texObj->SetName(0);  // テクスチャ名を未割り当て状態にしておく

#if defined(NW_DEBUG)
        texObj->Reset();
#endif
    }
}

namespace internal
{

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

void                    LoadTexture(
                            u16         texWidth,
                            u16         texHeight,
                            TexFmt      texFormat,
                            u16         texAttribute,
                            u8          SheetNum,
                            const void* pImage,
                            bool        isSmallLinearFilter,
                            bool        isLargeLinearFilter);

#elif defined(NW_PLATFORM_CAFE)

void InitTexture_(GX2Texture *texture, u32 width, u32 height,
    u32 depth, u32 numMips,
    GX2SurfaceFormat format, GX2SurfaceDim dim, GX2CompSel compSel)
{
    GX2_CHECK_ENUM_RANGE(format, GX2_SURFACE_FORMAT);
    GX2_CHECK_ENUM_RANGE(dim, GX2_SURFACE_DIM);

    // Set user surface values
    texture->surface.dim     = dim;
    texture->surface.width   = width;
    texture->surface.height  = height;
    texture->surface.depth   = depth;
    texture->surface.numMips = numMips;
    texture->surface.format  = format;

    // Set most common default surface values
    texture->surface.aa         = GX2_AA_MODE_1X; // AA textures not yet supported
    texture->surface.use        = GX2_SURFACE_USE_TEXTURE;
    texture->surface.tileMode   = GX2_TILE_MODE_DEFAULT;
    texture->surface.swizzle    = 0;

    // Set typical view values
    texture->viewFirstMip   = 0;
    texture->viewNumMips    = numMips;
    texture->viewFirstSlice = 0;
    texture->viewNumSlices  = depth;

    // Set default component selector
    texture->compSel = compSel;

    // Calculate the size, alignment, pitch, and mipOffset values
    GX2CalcSurfaceSizeAndAlignment(&texture->surface);

    // Now set up the registers based on those values
    GX2InitTextureRegs(texture);
}

#endif

}

void
ResFontBase::LoadTexture(const internal::TextureObject* pTexObj)
{
    NW_PROFILE("nw::font::ResFontBase::LoadTexture");
    NW_ASSERT_VALID_POINTER(pTexObj);
    bool doLoad = false;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    GLuint texName = 0;

    {
        texName = pTexObj->GetName();
        if (texName == 0)
        {
            glGenTextures(1, &texName);
            const_cast<internal::TextureObject*>(pTexObj)->SetName(texName);
            doLoad = true;
        }
        else
        {
            /// @todo 強制ロードの仕組みをどうするか？
//          doLoad = 0 != glyph.isSheetUpdated;          // 強制ロード
        }
    }

    if (doLoad)
    {
        glBindTexture(GL_TEXTURE_2D_ARRAY, texName);

        internal::LoadTexture(
            pTexObj->GetSize().width,
            pTexObj->GetSize().height,
            pTexObj->GetFormat(),
            pTexObj->GetFormatAttribute(),
            pTexObj->GetSheetNum(),
            reinterpret_cast<const void*>(pTexObj->GetImage()),
            false, false
            );

        NW_GL_ASSERT();
    }

#elif defined(NW_PLATFORM_CAFE)

    if (pTexObj->GetName() == 0)
    {
        const_cast<internal::TextureObject*>(pTexObj)->SetName(1);
        doLoad = true;
    }
    else
    {
        /// @todo 強制ロードの仕組みをどうするか？
        //doLoad = 0 != glyph.isSheetUpdated;          // 強制ロード
    }

    if (doLoad)
    {
        // GX2: テクスチャロード
        static const GX2SurfaceFormat s_TextureFormatTable[FONT_SHEET_FORMAT_QUANTITY] =
        {
            GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, // FONT_SHEET_FORMAT_RGBA8,
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_TC_R8_UNORM, // FONT_SHEET_FORMAT_A8
            GX2_SURFACE_FORMAT_T_BC1_UNORM, // FONT_SHEET_FORMAT_BC1
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_T_BC3_UNORM, // FONT_SHEET_FORMAT_BC3
            GX2_SURFACE_FORMAT_T_BC4_UNORM, // FONT_SHEET_FORMAT_BC4
            GX2_SURFACE_FORMAT_T_BC5_UNORM, // FONT_SHEET_FORMAT_BC5
            GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB, // FONT_SHEET_FORMAT_RGBA8_SRGB,
            GX2_SURFACE_FORMAT_T_BC1_SRGB, // FONT_SHEET_FORMAT_BC1_SRGB
            GX2_SURFACE_FORMAT_INVALID, // not support
            GX2_SURFACE_FORMAT_T_BC3_SRGB, // FONT_SHEET_FORMAT_BC3_SRGB
        };

        static const GX2CompSel s_TextureComponetSelectorTable[FONT_SHEET_FORMAT_QUANTITY] =
        {
            GX2_COMP_SEL_XYZW, // FONT_SHEET_FORMAT_RGBA8,
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // not support
            GX2_GET_COMP_SEL(GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_X_R), // FONT_SHEET_FORMAT_A8
            GX2_COMP_SEL_XYZW, // FONT_SHEET_FORMAT_BC1
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // FONT_SHEET_FORMAT_BC3
            GX2_GET_COMP_SEL(GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_C_1, GX2_COMPONENT_X_R), // FONT_SHEET_FORMAT_BC4
            GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G), // FONT_SHEET_FORMAT_BC5
            GX2_COMP_SEL_XYZW, // FONT_SHEET_FORMAT_RGBA8_SRGB,
            GX2_COMP_SEL_XYZW, // FONT_SHEET_FORMAT_BC1_SRGB
            GX2_COMP_SEL_XYZW, // not support
            GX2_COMP_SEL_XYZW, // FONT_SHEET_FORMAT_BC3_SRGB
        };

        NW_ASSERT_MAX(pTexObj->GetFormat(), FONT_SHEET_FORMAT_QUANTITY - 1);
        const GX2SurfaceFormat format = s_TextureFormatTable[pTexObj->GetFormat()];
        NW_ASSERT(format != GX2_SURFACE_FORMAT_INVALID);

        GX2Texture* texture = const_cast<GX2Texture*>(pTexObj->GetTexture());
        NW_ASSERT_NOT_NULL(texture);

        internal::InitTexture_(
            texture,
            pTexObj->GetSize().width,
            pTexObj->GetSize().height,
            pTexObj->GetSheetNum(),
            1,
            format,
            GX2_SURFACE_DIM_2D_ARRAY,
            s_TextureComponetSelectorTable[pTexObj->GetFormat()]);

        void* texAddr = reinterpret_cast<void*>(pTexObj->GetImage());
        NW_ASSERT_NOT_NULL(texAddr);
        NW_WARNING((reinterpret_cast<u32>(texAddr) % texture->surface.alignment) == 0, "Texture alignment is invalid!");
        GX2InitTexturePtrs(texture, texAddr, 0);
    }
#endif
}

//########################################
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
//########################################

namespace internal
{

struct TexSpec
{
    GX2SurfaceFormat gx2_format;
    s32 internalformat;
    s32 format;
    s32 type;
    s32 bit_by_pixel;
    bool compressed;
};

void SetupGLTextureSwzzling_(nw::font::TexFmt format)
{
    if (format == nw::font::FONT_SHEET_FORMAT_A8) {
        // A8の場合は、RGBの要素が全て1にになり、頂点カラーがそのまま出るようにする
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_R, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_G, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_B, GL_ONE);
    } else if (format == nw::font::FONT_SHEET_FORMAT_LA8) {
        // Cafeのフォーマット次第だが、現状ではAとIを逆転させて表示する
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_A, GL_RED);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_R, GL_ALPHA);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_G, GL_ALPHA);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_B, GL_ALPHA);
    } else if (format == nw::font::FONT_SHEET_FORMAT_LA4) {
        // 本来はLA4はそのまま出せるはずだが、ひとまず見られる絵が出るようにする
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_R, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_G, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_B, GL_ONE);
    } else if (format == nw::font::FONT_SHEET_FORMAT_BC4) {
        // R成分をアルファとして利用する
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_A, GL_RED);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_R, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_G, GL_ONE);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_B, GL_ONE);
    } else if (format == nw::font::FONT_SHEET_FORMAT_BC5) {
        // G成分をカラー(明度)として利用する
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_R, GL_GREEN);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_G, GL_GREEN);
        glTexParameteri(GL_TEXTURE_2D_ARRAY, GL_TEXTURE_SWIZZLE_B, GL_GREEN);
    }
}

void
LoadTexture(
    u16         texWidth,
    u16         texHeight,
    TexFmt      texFormat,
    u16         texAttribute,
    u8          sheetNum,
    const void* pImage,
    bool        /* isSmallLinearFilter */,
    bool        /* isLargeLinearFilter */
)
{
    static const TexSpec texSpec[FONT_SHEET_FORMAT_QUANTITY] =
    {
        { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GL_RGBA8, GL_RGBA, GL_UNSIGNED_BYTE, 32, false },   // FONT_SHEET_FORMAT_RGBA8
        { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GL_RGB8, GL_RGB, GL_UNSIGNED_BYTE, 24, false }, // FONT_SHEET_FORMAT_RGB8
        { GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM, GL_RGB5_A1, GL_RGBA, GL_UNSIGNED_SHORT_5_5_5_1, 16, false }, // FONT_SHEET_FORMAT_RGB5A1
        { GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM, GL_RGB, GL_RGB, GL_UNSIGNED_SHORT_5_6_5, 16, false },  // FONT_SHEET_FORMAT_RGB565
        { GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM, GL_RGBA4, GL_RGBA, GL_UNSIGNED_SHORT_4_4_4_4, 16, false },   // FONT_SHEET_FORMAT_RGBA4
        { GX2_SURFACE_FORMAT_TC_R8_G8_UNORM, GL_LUMINANCE8_ALPHA8, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, 16, false },   // FONT_SHEET_FORMAT_LA8
        { GX2_SURFACE_FORMAT_T_R4_G4_UNORM, GL_ALPHA8, GL_ALPHA, GL_UNSIGNED_BYTE, 8, false },  // FONT_SHEET_FORMAT_LA4
        { GX2_SURFACE_FORMAT_T_R4_G4_UNORM, GL_ALPHA4, GL_ALPHA, GL_UNSIGNED_BYTE, 4, false },  // FONT_SHEET_FORMAT_A4
        { GX2_SURFACE_FORMAT_TC_R8_UNORM, GL_ALPHA8, GL_ALPHA, GL_UNSIGNED_BYTE, 8, false },    // FONT_SHEET_FORMAT_A8
        { GX2_SURFACE_FORMAT_T_BC1_UNORM, GL_COMPRESSED_RGB_S3TC_DXT1_EXT, GL_RGBA, 0 /* N/A */, 4, true }, // FONT_SHEET_FORMAT_BC1
        { GX2_SURFACE_FORMAT_T_BC2_UNORM, GL_COMPRESSED_RGBA_S3TC_DXT3_EXT, GL_RGBA, 0 /* N/A */, 8, true },    // FONT_SHEET_FORMAT_BC2
        { GX2_SURFACE_FORMAT_T_BC3_UNORM, GL_COMPRESSED_RGBA_S3TC_DXT5_EXT, GL_RGBA, 0 /* N/A */, 8, true },    // FONT_SHEET_FORMAT_BC3
        { GX2_SURFACE_FORMAT_T_BC4_UNORM, GL_COMPRESSED_LUMINANCE_LATC1_EXT, GL_ALPHA8, 0 /* N/A */, 4, true }, // FONT_SHEET_FORMAT_BC4
        { GX2_SURFACE_FORMAT_T_BC5_UNORM, GL_COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT, GL_LUMINANCE8_ALPHA8, 0 /* N/A */, 8, true },    // FONT_SHEET_FORMAT_BC5
        { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB, GL_SRGB8_ALPHA8_EXT, GL_RGBA, GL_UNSIGNED_BYTE, 32, false },   // FONT_SHEET_FORMAT_RGBA8
        { GX2_SURFACE_FORMAT_T_BC1_SRGB, GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT, GL_RGBA, 0 /* N/A */, 4, true }, // FONT_SHEET_FORMAT_BC1
        { GX2_SURFACE_FORMAT_T_BC2_SRGB, GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT, GL_RGBA, 0 /* N/A */, 8, true },    // FONT_SHEET_FORMAT_BC2
        { GX2_SURFACE_FORMAT_T_BC3_SRGB, GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT, GL_RGBA, 0 /* N/A */, 8, true },    // FONT_SHEET_FORMAT_BC3
    };

    const TexSpec* spec = NULL;
    const GLint mipLevel = 0;
    if (texFormat < FONT_SHEET_FORMAT_QUANTITY)
    {
        spec = &texSpec[texFormat];
    }
    else
    {
        NW_ASSERTMSG(false,
            "CharWriter::LoadTexture() : Unknown font sheet format(=%d)",
            texFormat);
        return;
    }

    const u32 imageSize = texWidth * texHeight * sheetNum * spec->bit_by_pixel / 8;

    // タイリングされているテクスチャをGLで読めるリニアの形式にする
    if((texAttribute & FONT_SHEET_FORMAT_LINEAR_FLAG) == 0)
    {
        TC2Config config = {};
        config.gbTilingConfig = 0;
        config.gpu = GPU_Cafe;

        GX2Surface src_surface;
        src_surface.dim = GX2_SURFACE_DIM_2D_ARRAY;
        src_surface.width = texWidth;
        src_surface.height = texHeight;
        src_surface.depth = sheetNum;
        src_surface.numMips = 1;
        src_surface.format = spec->gx2_format;
        src_surface.aa = GX2_AA_MODE_1X;
        src_surface.use = GX2_SURFACE_USE_TEXTURE;
        src_surface.imageSize = imageSize;
        src_surface.imagePtr = const_cast<void*>(pImage);
        src_surface.mipSize = 0;
        src_surface.mipPtr = NULL;
        src_surface.tileMode = GX2_TILE_MODE_2D_TILED_THIN1;
        src_surface.swizzle = 0;
        src_surface.alignment = 1;
        src_surface.pitch = 128;

        GX2Surface dst_surface;

        // TODO : 明示的に関数をロードするように修正して、ビルド時＋実行時の texUtls 依存を無くす。
        TextureTilingConverter::Initialize(config);
        TextureTilingConverter::ConvertTiling(src_surface, dst_surface);

        // テクスチャを初期化
        if (spec->compressed)
        {
            glCompressedTexImage3D(GL_TEXTURE_2D_ARRAY, mipLevel, spec->internalformat, texWidth, texHeight, sheetNum, 0, src_surface.imageSize, dst_surface.imagePtr);
        }
        else
        {
            glTexImage3D(GL_TEXTURE_2D_ARRAY, mipLevel, spec->internalformat, texWidth, texHeight, sheetNum, 0, spec->format, spec->type, dst_surface.imagePtr);
        }
        NW_GL_ASSERT();

        // データはGL側が内部にコピーするので、タイリングを行う際に作ったsurfaceは削除してよい
        TextureTilingConverter::DestorySurface(dst_surface);
        TextureTilingConverter::Finalize();

        SetupGLTextureSwzzling_(texFormat);

    }else{

        // テクスチャを初期化
        if (spec->compressed)
        {
            glCompressedTexImage3D(GL_TEXTURE_2D_ARRAY, mipLevel, spec->internalformat, texWidth, texHeight, sheetNum, 0, imageSize, pImage);
        }
        else
        {
            glTexImage3D(GL_TEXTURE_2D_ARRAY, mipLevel, spec->internalformat, texWidth, texHeight, sheetNum, 0, spec->format, spec->type, pImage);
        }

        SetupGLTextureSwzzling_(texFormat);
    }
}

}

#endif

}   // namespace font
}   // namespace nw
