﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/control/dw_UIElementUtility.h>
#include <nw/dw/control/dw_ElementFinder.h>
#include <nw/dw/control/dw_HighestLevelFocusableElementFinder.h>

namespace nw {
namespace internal {
namespace dw {

u32 UIElementUtility::UnaryFunction::GetCurrentLevel()
{
    return m_CurrentLevel;
}

bool UIElementUtility::UnaryFunction::SetCurrentLevel(u32 value)
{
    if(m_CurrentLevel == value)
    {
        return false;
    }

    m_CurrentLevel = value;

    return OnCurrentLevelChanged();
}

//---------------------------------------------------------------------------

nw::math::Vector2 UIElementUtility::PositionToChildClient(const nw::math::Vector2 position, const UIElement& childElement)
{
    nw::math::Vector2 result(position);
    PositionToChildClient(&result, childElement);
    return result;
}

void UIElementUtility::PositionToChildClient(nw::math::Vector2* pPosition, const UIElement& childElement)
{
    NW_NULL_ASSERT(pPosition);
    pPosition->x -= childElement.GetTopLeft().x;
    pPosition->y -= childElement.GetTopLeft().y;
}

nw::math::Vector2 UIElementUtility::PositionToParentClient(const nw::math::Vector2 position, const UIElement& parentElement)
{
    nw::math::Vector2 result(position);
    PositionToParentClient(&result, parentElement);
    return result;
}

void UIElementUtility::PositionToParentClient(nw::math::Vector2* pPosition, const UIElement& parentElement)
{
    NW_NULL_ASSERT(pPosition);
    pPosition->x += parentElement.GetTopLeft().x;
    pPosition->y += parentElement.GetTopLeft().y;
}

bool UIElementUtility::HitTest(const nw::math::Vector2 position, const UIElement& element)
{
    nw::math::Vector2 topLeft = element.GetTopLeft();
    nw::math::Vector2 size    = element.GetFixedSize();

    return topLeft.x <= position.x && position.x < topLeft.x + size.x &&
           topLeft.y <= position.y && position.y < topLeft.y + size.y;
}


void UIElementUtility::ForChildElements(const UIElement& element, UnaryFunction& functor)
{
    UIElementList& contents = element.GetContents();

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pChildElement = contents[i];

        if(pChildElement == NULL)
        {
            continue;
        }

        if(functor(*pChildElement))
        {
            break;
        }
    }
}

void UIElementUtility::ForAllElements(const UIElement& element, UnaryFunction& functor)
{
    ForAllElementsInternal(element, functor, 1);
}

u32 UIElementUtility::IndexOf(const UIElement& targetElement, const UIElement& element)
{
    ElementFinder finder(element);
    return IndexOf(targetElement.GetContents(), finder);
}

u32 UIElementUtility::IndexOf(const UIElement& element, UnaryFunction& functor)
{
    return IndexOf(element.GetContents(), functor);
}

u32 UIElementUtility::IndexOf(const UIElementList& elements, const UIElement& element)
{
    ElementFinder finder(element);
    return IndexOf(elements, finder);
}

u32 UIElementUtility::IndexOf(const UIElementList& elements, UnaryFunction& functor)
{
    for(int i=0; i<elements.GetCount(); ++i)
    {
        UIElement* pChildElement = elements[i];

        if(pChildElement == NULL)
        {
            continue;
        }

        if(functor(*pChildElement))
        {
            return i;
        }
    }

    return INVALID_INDEX;
}

UIElement* UIElementUtility::GetFocusedElement(const UIElement& element)
{
    UIElement* pFocusEnterContent = element.GetLastFocusEnterContent();

    if(pFocusEnterContent == NULL || !pFocusEnterContent->GetIsContainsFocus())
    {
        return NULL;
    }

    if(pFocusEnterContent->GetIsFocused())
    {
        return pFocusEnterContent;
    }

    return GetFocusedElement(*pFocusEnterContent);
}

UIElement* UIElementUtility::GetEndFocusableElement(const UIElement& element)
{
    // LastFocusEnter 要素の子階層から見つかったら、それを優先します。
    // 見つからなかったら、兄弟の子階層からも検索します。
    UIElement* pTarget = element.GetLastFocusEnterContent();

    if(pTarget != NULL && pTarget->GetVisibility() == VISIBLE)
    {
        UIElement* pDescendantTarget = GetEndFocusableElement(*pTarget);

        if(pDescendantTarget != NULL)
        {
            return pDescendantTarget;
        }
    }

    HighestLevelFocusableElementFinder finder;
    ForAllElements(element, finder);

    if(finder.GetResult() != NULL)
    {
        return finder.GetResult();
    }

    // 子階層全体で見つからなかったら、自身を返します。
    return element.GetIsFocusable() ? const_cast<UIElement*>(&element) : NULL;
}

UIElement* UIElementUtility::GetPointerOverElement(const UIElement& element)
{
    UIElement* pPointerOverContent = element.GetPointerOverContent();

    if(pPointerOverContent == NULL || !pPointerOverContent->GetIsContainsPointerOver())
    {
        return NULL;
    }

    if(pPointerOverContent->GetIsPointerOver())
    {
        return pPointerOverContent;
    }

    return GetPointerOverElement(*pPointerOverContent);
}

#if defined(NW_DEBUG) || defined(NW_DEVELOP)

void UIElementUtility::PrintUIElementTree(const UIElement& element)
{
    static u32 level = 0;

    UIElementList& contents = element.GetContents();

    // インデントを出力します。
    for(u32 i=0; i<level; ++i)
    {
        printf(" ");
    }

    // "+" or "-" を出力します。
    u32 contentsCount = contents.GetCount();
    printf(contentsCount > 0 ? "+ " : "- ");

    // UI要素名を出力します。
    printf("%s", element.ToString());

    // フォーカスマークを出力します。
    if(element.GetIsContainsFocus())
    {
        printf(" [CF]");
    }

    // フォーカスマークを出力します。
    if(element.GetIsFocused())
    {
        printf(" [F]");
    }

    printf("\n");

    // 子要素を出力します。
    for(u32 i=0; i<contentsCount; ++i)
    {
        UIElement* pContent = contents[i];

        ++level;
        PrintUIElementTree(*pContent);
        --level;
    }
}

#endif

bool UIElementUtility::ForAllElementsInternal(const UIElement& element, UnaryFunction& functor, u32 level)
{
    // HACK : フォーカスを保持するコントロールのためにラストフォーカス要素を最優先で検索します。
    UIElement* pLastFocusedElement = element.GetLastFocusEnterContent();

    if(pLastFocusedElement != NULL)
    {
        if(functor.SetCurrentLevel(level))
        {
            return true;
        }

        if(functor(*pLastFocusedElement))
        {
            return true;
        }

        if(ForAllElementsInternal(*pLastFocusedElement, functor, level + 1))
        {
            return true;
        }
    }

    // ラストフォーカス要素以外は、インデックス順で検索します。
    UIElementList& contents = element.GetContents();

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pChildElement = contents[i];

        if(pChildElement == NULL ||
            pChildElement == pLastFocusedElement)
        {
            continue;
        }

        if(functor.SetCurrentLevel(level))
        {
            return true;
        }

        if(functor(*pChildElement))
        {
            return true;
        }

        if(ForAllElementsInternal(*pChildElement, functor, level + 1))
        {
            return true;
        }
    }

    return false;
}

} // dw
} // internal
} // nw
