﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/control/dw_UIElement.h>
#include <nw/dw/control/dw_UIElementUtility.h>
#include <nw/dw/control/dw_UIElementAligner.h>
#include <nw/dw/control/dw_FocusableElementFinder.h>
#include <nw/dw/control/dw_HighestLevelFocusableElementFinder.h>
#include <nw/dw/control/dw_FixedUIElementList.h>
#include <nw/dw/control/dw_UIElementListEventArgs.h>

// #define DEBUG_FOCUS
// #define DEBUG_POINTER

namespace nw {
namespace internal {
namespace dw {

const f32 UIElement::InvalidWidth  = -1;
const f32 UIElement::InvalidHeight = -1;

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
UIElement::UIElement() :
m_pParent(NULL),
m_TopLeft(0.f, 0.f),
m_TransformMatrix(nw::math::Matrix34::Identity()),
m_Size(0.f, 0.f),
m_MeasuredSize(0.f, 0.f),
m_FixedSize(0.f, 0.f),
m_MinimumSize(InvalidWidth, InvalidHeight),
m_MaximumSize(InvalidWidth, InvalidHeight),
m_ScrollOffset(0.f, 0.f),
m_Margin(2.f),
m_Padding(1.f),
m_Visibility(VISIBLE),
m_IsFocasable(false),
m_Measurement(MEASUREMENT_AUTO),
m_HorizontalAlignment(HORIZONTAL_STRETCH),
m_VerticalAlignment(VERTICAL_STRETCH),
m_Dock(DOCK_FILL),
m_BackgroundColor0(nw::ut::Color4f::X_TRANSPARENT()),
m_BackgroundColor1(nw::ut::Color4f::X_TRANSPARENT()),
m_BorderColor(nw::ut::Color4f::X_BLACK()),
m_IsBordered(false),
m_IsFocused(false),
m_IsContainsFocus(false),
m_pLastFocusEnterContent(NULL),
m_IsContainsPointerOver(false),
m_pPointerOverContent(NULL),
m_pContents(&UIELEMENTLIST_EMPTY),
m_ContentsChangedObserver(this, &UIElement::OnItemsChanged)
{
}
#if defined(_WIN32)
#pragma warning(pop) // warning: used in base member initializer list
#endif

UIElement* UIElement::GetParent() const
{
    return m_pParent;
}

const nw::math::Vector2 UIElement::GetTopLeft() const
{
    return m_TopLeft;
}

UIElement& UIElement::SetTopLeft(f32 x, f32 y)
{
    m_TopLeft.x = x;
    m_TopLeft.y = y;
    return *this;
}

UIElement& UIElement::SetTopLeft(const nw::math::Vector2& value)
{
    return SetTopLeft(value.x, value.y);
}

UIElement& UIElement::SetLeft(f32 value)
{
    m_TopLeft.x = value;
    return *this;
}

UIElement& UIElement::SetTop(f32 value)
{
    m_TopLeft.y = value;
    return *this;
}

const nw::math::Vector2 UIElement::GetSize() const
{
    return m_Size;
}

const f32 UIElement::GetWidth() const
{
    return m_Size.x;
}

const f32 UIElement::GetHeight() const
{
    return m_Size.y;
}

UIElement& UIElement::SetSize(f32 width, f32 height)
{
    m_Size.x = width;
    m_Size.y = height;

    return *this;
}

UIElement& UIElement::SetSize(const nw::math::Vector2& value)
{
    return SetSize(value.x, value.y);
}

const nw::math::Vector2& UIElement::GetMeasuredSize() const
{
    return m_MeasuredSize;
}

const f32 UIElement::GetMeasuredWidth() const
{
    return m_MeasuredSize.x;
}

const f32 UIElement::GetMeasuredHeight() const
{
    return m_MeasuredSize.y;
}

const nw::math::Vector2& UIElement::GetFixedSize() const
{
    return m_FixedSize;
}

const f32 UIElement::GetFixedWidth() const
{
    return m_FixedSize.x;
}

const f32 UIElement::GetFixedHeight() const
{
    return m_FixedSize.y;
}

UIElement& UIElement::SetWidth(f32 value)
{
    SetSize(value, GetHeight());
    return *this;
}

UIElement& UIElement::SetHeight(f32 value)
{
    SetSize(GetWidth(), value);
    return *this;
}

const nw::math::Vector2 UIElement::GetMinimumSize() const
{
    return m_MinimumSize;
}

const f32 UIElement::GetMinimumWidth() const
{
    return m_MinimumSize.x;
}

const f32 UIElement::GetMinimumHeight() const
{
    return m_MinimumSize.y;
}

UIElement& UIElement::SetMinimumSize(f32 width, f32 height)
{
    SetMinimumWidth(width);
    SetMinimumHeight(height);
    return *this;
}

UIElement& UIElement::SetMinimumSize(const nw::math::Vector2& value)
{
    return SetSize(value.x, value.y);
}

UIElement& UIElement::SetMinimumWidth(f32 value)
{
    NW_ASSERT(value >= InvalidWidth);
    m_MinimumSize.x = value;
    return *this;
}

UIElement& UIElement::SetMinimumHeight(f32 value)
{
    NW_ASSERT(value >= InvalidHeight);
    m_MinimumSize.y = value;

    // 正規化します。
    SetHeight(GetHeight());

    return *this;
}

const nw::math::Vector2 UIElement::GetMaximumSize() const
{
    return m_MaximumSize;
}

const f32 UIElement::GetMaximumWidth() const
{
    return m_MaximumSize.x;
}

const f32 UIElement::GetMaximumHeight() const
{
    return m_MaximumSize.y;
}

UIElement& UIElement::SetMaximumSize(f32 width, f32 height)
{
    SetMaximumWidth(width);
    SetMaximumHeight(height);
    return *this;
}

UIElement& UIElement::SetMaximumSize(const nw::math::Vector2& value)
{
    return SetSize(value.x, value.y);
}

UIElement& UIElement::SetMaximumWidth(f32 value)
{
    NW_ASSERT(value >= InvalidWidth);
    m_MaximumSize.x = value;
    return *this;
}

UIElement& UIElement::SetMaximumHeight(f32 value)
{
    NW_ASSERT(value >= InvalidHeight);
    m_MaximumSize.y = value;

    // 正規化します。
    SetHeight(GetHeight());

    return *this;
}

const Thickness UIElement::GetMargin() const
{
    return m_Margin;
}

UIElement& UIElement::SetMargin(const Thickness& value)
{
    m_Margin = value;
    return *this;
}

const Thickness UIElement::GetPadding() const
{
    return m_Padding;
}

UIElement& UIElement::SetPadding(const Thickness& value)
{
    m_Padding = value;
    return *this;
}

const nw::math::Vector2 UIElement::GetContentAreaTopLeft() const
{
    return nw::math::Vector2(m_Padding.left, m_Padding.top);
}

const nw::math::Vector2 UIElement::GetContentAreaSize() const
{
    if( m_FixedSize.IsZero() )
    {
        return nw::math::Vector2::Zero();
    }

    return nw::math::Vector2(
        m_FixedSize.x - m_Padding.GetWidth(),
        m_FixedSize.y - m_Padding.GetHeight());
}

Visibility UIElement::GetVisibility() const
{
    return m_Visibility;
}

UIElement& UIElement::SetVisibility(Visibility value)
{
    m_Visibility = value;
    return *this;
}

bool UIElement::GetIsFocusable() const
{
    return m_IsFocasable;
}

UIElement& UIElement::SetIsFocusable(bool value)
{
    m_IsFocasable = value;
    return *this;
}

bool UIElement::GetIsFocused() const
{
    return m_IsFocused;
}

bool UIElement::GetIsContainsFocus() const
{
    return m_IsContainsFocus;
}

UIElement* UIElement::GetLastFocusEnterContent() const
{
    return m_pLastFocusEnterContent;
}

bool UIElement::GetIsPointerOver() const
{
    return m_IsContainsPointerOver && m_pPointerOverContent == NULL;
}

bool UIElement::GetIsContainsPointerOver() const
{
    return m_IsContainsPointerOver;
}

UIElement* UIElement::GetPointerOverContent() const
{
    return m_pPointerOverContent;
}

Measurement UIElement::GetMeasurement() const
{
    return m_Measurement;
}

UIElement& UIElement::SetMeasurement(Measurement value)
{
    m_Measurement = value;
    return *this;
}

Dock UIElement::GetDock() const
{
    return m_Dock;
}

UIElement& UIElement::SetDock(Dock value)
{
    m_Dock = value;
    return *this;
}

HorizontalAlignment UIElement::GetHorizontalAlignment() const
{
    return m_HorizontalAlignment;
}

UIElement& UIElement::SetHorizontalAlignment(HorizontalAlignment value)
{
    m_HorizontalAlignment = value;
    return *this;
}

VerticalAlignment UIElement::GetVerticalAlignment() const
{
    return m_VerticalAlignment;
}

UIElement& UIElement::SetVerticalAlignment(VerticalAlignment value)
{
    m_VerticalAlignment = value;
    return *this;
}

const nw::ut::Color4f UIElement::GetBackgroundColor() const
{
    return m_BackgroundColor0;
}

UIElement& UIElement::SetBackgroundColor(const nw::ut::Color4f& value)
{
    m_BackgroundColor0 = value;
    m_BackgroundColor1 = value;
    return *this;
}

UIElement& UIElement::SetBackgroundColor(const nw::ut::Color4f& value0, const nw::ut::Color4f& value1)
{
    m_BackgroundColor0 = value0;
    m_BackgroundColor1 = value1;
    return *this;
}

const nw::ut::Color4f UIElement::GetBackgroundColor0() const
{
    return m_BackgroundColor0;
}

UIElement& UIElement::SetBackgroundColor0(const nw::ut::Color4f& value)
{
    m_BackgroundColor0 = value;
    return *this;
}

const nw::ut::Color4f UIElement::GetBackgroundColor1() const
{
    return m_BackgroundColor1;
}

UIElement& UIElement::SetBackgroundColor1(const nw::ut::Color4f& value)
{
    m_BackgroundColor1 = value;
    return *this;
}

const nw::ut::Color4f UIElement::GetBorderColor() const
{
    return m_BorderColor;
}

UIElement& UIElement::SetBorderColor(const nw::ut::Color4f& value)
{
    m_BorderColor = value;
    return *this;
}

bool UIElement::GetIsBordered() const
{
    return m_IsBordered;
}

UIElement& UIElement::SetIsBordered(bool value)
{
    m_IsBordered = value;
    return *this;
}

UIElementList& UIElement::GetContents()
{
    return *m_pContents;
}

UIElementList& UIElement::GetContents() const
{
    return *m_pContents;
}

UIElement& UIElement::SetContents(UIElementList& value)
{
    // コールバックを解除して削除イベント発行します。
    m_pContents->GetItemsChangedEvent() -= m_ContentsChangedObserver;

    for(int i=0; i<m_pContents->GetCount(); ++i)
    {
        UIElement* pItem = m_pContents->GetItem(i);

        if(pItem == NULL)
        {
            continue;
        }

        UIElementListEventArgs args(UIElementListEventArgs::REMOVE_ITEM, i, pItem);
        SynchronizeContentsChanged(args);
    }

    m_pContents = &value;

    // 挿入イベント発行してコールバックを登録します。
    for(int i=0; i<m_pContents->GetCount(); ++i)
    {
        UIElement* pItem = m_pContents->GetItem(i);

        if(pItem == NULL)
        {
            continue;
        }

        UIElementListEventArgs args(UIElementListEventArgs::INSERT_ITEM, i, pItem);
        SynchronizeContentsChanged(args);
    }

    m_pContents->GetItemsChangedEvent() += m_ContentsChangedObserver;

    return *this;
}

bool UIElement::SetFocus()
{
    if(m_IsFocused)
    {
        return true;
    }

    UIElement* pTargetElement = UIElementUtility::GetEndFocusableElement(*this);

    if(pTargetElement == NULL)
    {
        return false;
    }

    if(pTargetElement->GetIsFocused())
    {
        return true;
    }

    pTargetElement->FocusEnter(NULL);
    pTargetElement->SetFocusInternal();

    return true;
}

void UIElement::EnsureVisible()
{
    if(m_pParent != NULL)
    {
        m_pParent->EnsureVisibleInternal(*this);
    }
}

void UIElement::UpdateInputs(const Inputs& inputs)
{
    UpdatePointer(inputs);
    UpdatePointerInput(inputs);

    // フォーカス入力を更新する際には、ポインタ位置に依存しないので無効値を設定します。
    UpdateFocusedInput(Inputs(inputs).SetPointerPosition(0.f, 0.f));
}

const nw::math::Vector2& UIElement::Measure(UIRenderer& renderer)
{
    MeasureInternal(renderer);
    return GetMeasuredSize();
}

void UIElement::Update(UIElementTreeContext& context, UIRenderer& renderer, bool isMeasurement/*=true*/)
{
    if(isMeasurement)
    {
        MeasureInternal(renderer);
    }

    UpdateInternal(context);
}

void UIElement::Render(UIElementTreeContext& context, UIRenderer& renderer)
{
    const UIElement* pParentElement = context.PushUIElement(*this);

    UIElementRenderArgs& args = OnPrepareRender(context);

#if defined(DEBUG_POINTER) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
    if(GetIsPointerOver())
    {
        args.
            SetBorderColor(ut::Color4f::RED()).
            SetIsBordered(true);
    }
#endif

    // 自身の背景と境界線を描画します。
    if(!args.backgroundColor0.IsTransparent() ||
        !args.backgroundColor1.IsTransparent())
    {
        renderer.FillRectangle(
            &context,
            DrawRectangleArgs().
            SetTopLeft(nw::math::Vector2::Zero()).
            SetSize(GetFixedSize()).
            SetColor(args.backgroundColor0, args.backgroundColor1));
    }

    if(args.isBordered && !args.borderColor.IsTransparent())
    {
        renderer.DrawRectangle(
            &context,
            DrawRectangleArgs().
            SetTopLeft(nw::math::Vector2::Zero()).
            SetSize(GetFixedSize()).
            SetColor(args.borderColor));
    }

    // 自身を描画します。
    OnRender(context, renderer, args);

    PreRenderContents(context, renderer, args);

    // スクロールオフセットを適用して子要素を描画します。
    context.OffsetCurrentElementOrigin(-GetScrollOffset());
    RenderContents(context, renderer);
    context.OffsetCurrentElementOrigin(GetScrollOffset());

    PostRenderContents(context, renderer, args);

    context.PopUIElement(pParentElement);
}

void UIElement::SetMeasuredSize(UIElement& element, f32 width, f32 height)
{
    f32 newWidth  = width;
    f32 newHeight = height;

    if(element.m_MinimumSize.x != InvalidWidth)
    {
        newWidth = nw::ut::Max(element.m_MinimumSize.x, newWidth);
    }

    if(element.m_MaximumSize.x != InvalidWidth)
    {
        newWidth = nw::ut::Min(element.m_MaximumSize.x, newWidth);
    }

    if(element.m_MinimumSize.y != InvalidHeight)
    {
        newHeight = nw::ut::Max(element.m_MinimumSize.y, newHeight);
    }

    if(element.m_MaximumSize.y != InvalidHeight)
    {
        newHeight = nw::ut::Min(element.m_MaximumSize.y, newHeight);
    }

    element.m_MeasuredSize.x = newWidth;
    element.m_MeasuredSize.y = newHeight;
}

void UIElement::SetMeasuredSize(UIElement& element, const nw::math::Vector2 value)
{
    SetMeasuredSize(element, value.x, value.y);
}

const nw::math::Vector2& UIElement::GetScrollOffset() const
{
    return m_ScrollOffset;
}

void UIElement::SetScrollOffset(nw::math::Vector2 value)
{
    const nw::math::Vector2 contentSize = GetMeasuredSize();
    const nw::math::Vector2 size = GetFixedSize();

    f32 scrollWidth  = nw::ut::Max(0.f, contentSize.x - size.x);
    f32 scrollHeight = nw::ut::Max(0.f, contentSize.y - size.y);

    m_ScrollOffset.x = nw::ut::Clamp(value.x, 0.f, scrollWidth);
    m_ScrollOffset.y = nw::ut::Clamp(value.y, 0.f, scrollHeight);
}

const nw::math::Vector2 UIElement::OnMeasure(UIRenderer& renderer) const
{
    (void)renderer;

    if(m_pContents->GetCount() == 0)
    {
        return nw::math::Vector2::Zero();
    }

    UIElementList& contents = GetContents();

    f32 x = 0.f;
    f32 y = 0.f;

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pContent = contents[i];

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        const nw::math::Vector2 size = pContent->GetMeasuredSize();

        if(size.IsZero())
        {
            continue;
        }

        const Thickness margin = pContent->GetMargin();

        x = nw::ut::Max(x, size.x + margin.GetWidth());
        y = nw::ut::Max(y, size.y + margin.GetHeight());
    }

    return nw::math::Vector2(x, y);
}

UIElementRenderArgs& UIElement::OnPrepareRender(const UIElementTreeContext& context)
{
    (void)context;

    m_RenderArgs.
        SetBorderColor(m_BorderColor).
        SetBackgroundColor(m_BackgroundColor0, m_BackgroundColor1).
        SetIsBordered(m_IsBordered);

    return m_RenderArgs;
}

UIElement& UIElement::SetParent(UIElement* value)
{
    // 要素ツリーから削除される際には、フォーカスを削除し、マウス状態をリセットします。
    if(value == NULL)
    {
        UIElement* pFocusedElement = UIElementUtility::GetFocusedElement(*this);

        if(pFocusedElement != NULL)
        {
            pFocusedElement->KillFocusInternal();
            FocusLeave();
        }

        NW_ASSERT(!m_IsFocused);
        NW_ASSERT(!m_IsContainsFocus);

        if(m_IsContainsPointerOver)
        {
            PointerLeave();
        }

        NW_ASSERT(!GetIsPointerOver());
        NW_ASSERT(!m_IsContainsPointerOver);
    }

    m_pParent = value;

    return *this;
}

void UIElement::SynchronizeContentsChanged(UIElementListEventArgs& args)
{
    UIElement* pItem = args.GetItem();
    NW_ASSERT(pItem != NULL);

    // コンテンツ編集に同期して親要素へのポインタを設定します。
    switch(args.GetAction())
    {
    case UIElementListEventArgs::INSERT_ITEM:
        pItem->SetParent(this);
        break;

    case UIElementListEventArgs::REMOVE_ITEM:
        // フォーカスアイテムが削除された場合はフォーカスを解除します。
        if(pItem->GetIsFocused())
        {
            this->SetFocus();
        }

        if(m_pLastFocusEnterContent == pItem)
        {
            m_pLastFocusEnterContent = NULL;
        }

        pItem->SetParent(NULL);
        break;
    }

    // ラストフォーカスが設定されていない場合は自動設定します。
    if(m_pLastFocusEnterContent == NULL &&
        m_pContents->GetCount() > 0)
    {
        m_pLastFocusEnterContent = m_pContents->GetItem(0);
        NW_ASSERT(m_pLastFocusEnterContent);
    }

    OnContentsChanged(args);
}

nw::ut::Rect UIElement::GetVisibleRectangle() const
{
    return nw::ut::Rect(
        m_ScrollOffset.x,
        m_ScrollOffset.y,
        m_ScrollOffset.x + m_FixedSize.x,
        m_ScrollOffset.y + m_FixedSize.y);
}

void UIElement::EnsureVisibleInternal(UIElement& element)
{
    // 親階層から順に表示します。
    if(m_pParent != NULL)
    {
        m_pParent->EnsureVisibleInternal(*this);
    }

    OnEnsureVisible(element);
}

void UIElement::SetFocusInternal()
{
    if(!GetIsFocusable())
    {
        return;
    }

    if(m_IsFocused)
    {
        return;
    }

#if defined(DEBUG_FOCUS) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
    NW_LOG("[Focus] OnGotFocus : %s\n", ToString());
#endif

    m_IsFocused = true;
    OnGotFocus();
}

void UIElement::KillFocusInternal()
{
    if(!GetIsFocusable())
    {
        return;
    }

    if(!m_IsFocused)
    {
        return;
    }

#if defined(DEBUG_FOCUS) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
    NW_LOG("[Focus] OnLostFocus : %s\n", ToString());
#endif

    m_IsFocused = false;
    OnLostFocus();
}

void UIElement::FocusEnter(UIElement* pElement)
{
    // 子要素が SetFocus() された場合
    if(pElement != NULL)
    {
        // 自身がフォーカスを持つなら KillFocus しておきます。
        if(m_IsFocused)
        {
            KillFocusInternal();
        }

        UIElement* pFocusedElement = UIElementUtility::GetFocusedElement(*this);

        if(pFocusedElement != NULL)
        {
            NW_ASSERT(pFocusedElement->GetIsFocused());

            // 子孫要素がフォーカスを持つなら、ラストフォーカス要素が設定されているはずです。
            NW_ASSERT(m_pLastFocusEnterContent != NULL);
            NW_ASSERT(m_pLastFocusEnterContent->GetIsContainsFocus());

            pFocusedElement->KillFocusInternal();
            m_pLastFocusEnterContent->FocusLeave();
        }

        m_pLastFocusEnterContent = pElement;
    }
    // 自身が SetFocus() された場合
    else
    {
        // SetFocus() されたばかりなのでフォーカスは設定されていないはずです。
        NW_ASSERT(!m_IsFocused);
    }

    // 親階層から順に FocusEnter() します。
    // 既存のフォーカスを削除する必要があるので、
    // m_IsContainsFocus == true であっても親階層を評価する必要があります。
    if(m_pParent != NULL)
    {
        m_pParent->FocusEnter(this);
    }

    if(!m_IsContainsFocus)
    {
#if defined(DEBUG_FOCUS) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
        NW_LOG("[Focus] OnFocusEnter : %s\n", ToString());
#endif

        m_IsContainsFocus = true;
        OnFocusEnter(pElement);
    }
}

void UIElement::FocusLeave()
{
    if(!m_IsContainsFocus)
    {
        return;
    }

    // FocusLeave() をコールする前に KillFocusInternal() をコールする必要があります。
    NW_ASSERT(!m_IsFocused);

    // 子階層から順に FocusLeave() します。
    if(m_pLastFocusEnterContent != NULL)
    {
        m_pLastFocusEnterContent->FocusLeave();
    }

#if defined(DEBUG_FOCUS) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
    NW_LOG("[Focus] OnFocusLeave : %s\n", ToString());
#endif

    m_IsContainsFocus = false;
    OnFocusLeave();
}

void UIElement::UpdatePointer(const nw::internal::dw::Inputs& inputs)
{
    if(UIElementUtility::HitTest(inputs.GetPointerPosition(), *this))
    {
        const nw::math::Vector2 clientPosition =
            UIElementUtility::PositionToChildClient(inputs.GetPointerPosition(), *this);

        UpdatePointerImpl(Inputs(inputs).SetPointerPosition(clientPosition));
    }
    else
    {
        if(m_IsContainsPointerOver)
        {
            PointerLeave();
        }
    }
}

void UIElement::UpdatePointerImpl(const nw::internal::dw::Inputs& inputs)
{
    if(!m_IsContainsPointerOver)
    {
        PointerEnter(inputs);
    }

    const nw::math::Vector2 clientPosition =
        UIElementUtility::PositionToChildClient(inputs.GetPointerPosition(), *this);

    UIElement* pNewPointerOverContent = HitTestContents(clientPosition);

    if(m_pPointerOverContent != NULL &&
        pNewPointerOverContent != m_pPointerOverContent)
    {
        m_pPointerOverContent->PointerLeave();
    }

    Inputs clientPointerInput(inputs);
    clientPointerInput.SetPointerPosition(clientPosition);

    // 子要素上にマウスカーソルがある場合は、
    // 子要素の UpdatePointerImpl() に委譲し、自身の OnPointerOver() は呼び出しません。
    if(pNewPointerOverContent != NULL)
    {
        pNewPointerOverContent->UpdatePointerImpl(clientPointerInput);

        // 通常 UpdatePointerImpl() にて m_pPointerOverContent = pNewPointerOverContent に設定されますが、
        // マウス処理にて PointerOverContent が変更された場合は、ここで処理を中断します。
        if(!m_IsContainsPointerOver || m_pPointerOverContent != pNewPointerOverContent)
        {
            return;
        }

        NW_ASSERT(pNewPointerOverContent->GetIsContainsPointerOver());
    }
    else
    {
        PointerOver(clientPointerInput);
    }
}

void UIElement::UpdatePointerInput(const nw::internal::dw::Inputs& inputs)
{
    UIElement* pTargetElement = NULL;

    if(m_IsFocused)
    {
        pTargetElement = this;
    }
    else
    {
        pTargetElement = UIElementUtility::GetPointerOverElement(*this);
    }

    if(pTargetElement == NULL)
    {
        return;
    }

    pTargetElement->UpdatePointerInputImpl(inputs);
}

void UIElement::UpdatePointerInputImpl(const nw::internal::dw::Inputs& inputs)
{
    UIElement* pElement = this;
    nw::math::Vector2 clientPosition(inputs.GetPointerPosition());

    while(pElement != NULL)
    {
        if(pElement->OnUpdatePointerInput(Inputs(inputs).SetPointerPosition(clientPosition)))
        {
            break;
        }

        pElement = pElement->GetParent();

        if(pElement == NULL)
        {
            break;
        }

        UIElementUtility::PositionToParentClient(&clientPosition, *pElement);
    }
}

void UIElement::UpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    UIElement* pTargetElement = NULL;

    if(m_IsFocused)
    {
        pTargetElement = this;
    }
    else
    {
        pTargetElement = UIElementUtility::GetFocusedElement(*this);
    }

    if(pTargetElement == NULL)
    {
        return;
    }

    if(pTargetElement->PreviewUpdateFocusedInputImpl(inputs))
    {
        return;
    }

    pTargetElement->UpdateFocusedInputImpl(inputs);
}

bool UIElement::PreviewUpdateFocusedInputImpl(const nw::internal::dw::Inputs& inputs)
{
    if(m_pParent != NULL)
    {
        if(m_pParent->PreviewUpdateFocusedInputImpl(inputs))
        {
            return true;
        }
    }

    return OnPreviewUpdateFocusedInput(inputs);
}

void UIElement::UpdateFocusedInputImpl(const nw::internal::dw::Inputs& inputs)
{
    if(OnUpdateFocusedInput(inputs))
    {
        return;
    }

    if(m_pParent != NULL)
    {
        m_pParent->UpdateFocusedInputImpl(inputs);
    }
}

UIElement* UIElement::HitTestContents(const nw::math::Vector2& position) const
{
    const nw::math::Vector2& scrolledPosition = position + m_ScrollOffset;

    // インデックスの大きい子要素の方が Z 順位が高いので、後ろから HitTest します。
    for(s32 i = m_pContents->GetCount() - 1; i >= 0; --i)
    {
        UIElement* pChild = m_pContents->GetItem(i);

        if(pChild->GetVisibility() != VISIBLE)
        {
            continue;
        }

        if(UIElementUtility::HitTest(scrolledPosition, *pChild))
        {
            return pChild;
        }
    }

    return NULL;
}

void UIElement::PointerEnter(const nw::internal::dw::Inputs& inputs)
{
    if(m_IsContainsPointerOver)
    {
        NW_FATAL_ERROR("[UIElement] m_IsContainsPointerOver must not be false.\n");
        return;
    }

#if defined(DEBUG_POINTER) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
    NW_LOG("[Pointer] OnPointerEnter : %s\n", ToString());
#endif

    // 親の m_pPointerOverContent と、子の m_IsContainsPointerOver は必ず同期させます。
    m_IsContainsPointerOver = true;

    if(m_pParent != NULL)
    {
        m_pParent->m_pPointerOverContent = this;
    }

    OnPointerEnter(inputs);
}

void UIElement::PointerLeave()
{
    if(!m_IsContainsPointerOver)
    {
        NW_FATAL_ERROR("[UIElement] m_IsContainsPointerOver must not be true.\n");
        return;
    }

    if(m_pPointerOverContent != NULL)
    {
        m_pPointerOverContent->PointerLeave();

        // m_pPointerOverContent->PointerLeave() にて NULL に設定されるはずです。
        NW_ASSERT(m_pPointerOverContent == NULL);
    }

#if defined(DEBUG_POINTER) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
    NW_LOG("[Pointer] OnPointerLeave : %s\n", ToString());
#endif

    // 親の m_pPointerOverContent と、子の m_IsContainsPointerOver は必ず同期させます。
    m_IsContainsPointerOver = false;

    if(m_pParent != NULL)
    {
        m_pParent->m_pPointerOverContent = NULL;
    }

    OnPointerLeave();
}

void UIElement::PointerOver(const nw::internal::dw::Inputs& inputs)
{
    NW_ASSERT(GetIsPointerOver());

#if defined(DEBUG_POINTER) && (defined(NW_DEBUG) || defined(NW_DEVELOP))
    NW_LOG("[Pointer] PointerOver : %s\n", ToString());
#endif

    UIElement* pElement = this;
    nw::math::Vector2 clientPosition(inputs.GetPointerPosition());

    if(pElement != NULL)
    {
        pElement->OnPointerOver(Inputs(inputs).SetPointerPosition(clientPosition));
    }
}

void UIElement::MeasureInternal(UIRenderer& renderer)
{
    OnPrepareMeasure();

    MeasureContents(renderer);

    f32 measuredWidth  = m_Size.x;
    f32 measuredHeight = m_Size.y;

    if(m_Measurement != MEASUREMENT_MANUAL)
    {
        nw::math::Vector2 contentSize = OnMeasure(renderer);

        // コンテントサイズにパディングサイズを加算します。
        if(m_Measurement & MEASUREMENT_AUTO_WIDTH && contentSize.x > 0.f)
        {
            measuredWidth = contentSize.x + m_Padding.GetWidth();
        }

        if(m_Measurement & MEASUREMENT_AUTO_HEIGHT && contentSize.y > 0.f)
        {
            measuredHeight = contentSize.y + m_Padding.GetHeight();
        }
    }

    UIElement::SetMeasuredSize(*this, measuredWidth, measuredHeight);
}

void UIElement::UpdateInternal(UIElementTreeContext& context)
{
    const UIElement* pParentElement = context.PushUIElement(*this);

    UpdateAlignment(pParentElement);
    OnUpdate(context);
    UpdateContents(context);

    context.PopUIElement(pParentElement);
}

void UIElement::UpdateAlignment(const UIElement* pParentElement)
{
    f32 x      = m_TopLeft.x;
    f32 y      = m_TopLeft.y;
    f32 width  = m_MeasuredSize.x;
    f32 height = m_MeasuredSize.y;

    if(pParentElement != NULL)
    {
        if(pParentElement->CanAlignHorizontal(*this))
        {
            nw::ut::Rect horizontalAlignedRect = UIElementAligner::GetHorizontalAlignedRect(
                *this,
                pParentElement->GetContentAreaTopLeft().x,
                pParentElement->GetContentAreaSize().x);

            x     = horizontalAlignedRect.left;
            width = horizontalAlignedRect.GetWidth();
        }

        if(pParentElement->CanAlignVertical(*this))
        {
            nw::ut::Rect verticalAlignedRect = UIElementAligner::GetVerticalAlignedRect(
                *this,
                pParentElement->GetContentAreaTopLeft().y,
                pParentElement->GetContentAreaSize().y);

            y      = verticalAlignedRect.top;
            height = verticalAlignedRect.GetHeight();
        }
    }

    SetTopLeft(x, y);
    SetCaluclatedSize(width, height);
}

void UIElement::MeasureContents(UIRenderer& renderer)
{
    for(int i=0; i<m_pContents->GetCount(); ++i)
    {
        UIElement* pContent = m_pContents->GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        pContent->MeasureInternal(renderer);
    }
}

void UIElement::UpdateContents(UIElementTreeContext& context)
{
    for(int i=0; i<m_pContents->GetCount(); ++i)
    {
        UIElement* pContent = m_pContents->GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        pContent->UpdateInternal(context);
    }
}

void UIElement::RenderContents(UIElementTreeContext& context, UIRenderer& renderer) const
{
    for(int i=0; i<m_pContents->GetCount(); ++i)
    {
        UIElement* pContent = m_pContents->GetItem(i);

        if(pContent == NULL ||
            !ShouldRenderContent(*pContent))
        {
            continue;
        }

        pContent->Render(context, renderer);
    }
}

bool UIElement::ShouldRenderContent(UIElement& element) const
{
    NW_ASSERT(element.GetParent() == this);

    if(element.GetVisibility() != VISIBLE)
    {
        return false;
    }

    nw::ut::Rect visibleRect = GetVisibleRectangle();

    nw::math::Vector2 elementTopLeft = element.GetTopLeft();

    if(elementTopLeft.x < visibleRect.left &&
        !nw::ut::FloatEquals(elementTopLeft.x, visibleRect.left))
    {
        return false;
    }

    if(elementTopLeft.y < visibleRect.top &&
        !nw::ut::FloatEquals(elementTopLeft.y, visibleRect.top))
    {
        return false;
    }

    nw::math::Vector2 elementSize = element.GetFixedSize();
    f32 elementRight = elementTopLeft.x + elementSize.x;

    if(visibleRect.right < elementRight &&
        !nw::ut::FloatEquals(visibleRect.right, elementRight))
    {
        return false;
    }

    f32 elementBottom = elementTopLeft.y + elementSize.y;

    if(visibleRect.bottom < elementBottom &&
        !nw::ut::FloatEquals(visibleRect.bottom, elementBottom))
    {
        return false;
    }

    return true;
}

void UIElement::SetCaluclatedSize(f32 width, f32 height)
{
    f32 newWidth  = width;
    f32 newHeight = height;

    if(m_MinimumSize.x != InvalidWidth)
    {
        newWidth = nw::ut::Max(m_MinimumSize.x, newWidth);
    }

    if(m_MaximumSize.x != InvalidWidth)
    {
        newWidth = nw::ut::Min(m_MaximumSize.x, newWidth);
    }

    if(m_MinimumSize.y != InvalidHeight)
    {
        newHeight = nw::ut::Max(m_MinimumSize.y, newHeight);
    }

    if(m_MaximumSize.y != InvalidHeight)
    {
        newHeight = nw::ut::Min(m_MaximumSize.y, newHeight);
    }

    if(m_FixedSize.x == newWidth && m_FixedSize.y == newHeight)
    {
        return;
    }

    m_FixedSize.x = newWidth;
    m_FixedSize.y = newHeight;

    ValidateScrollOffset();
}

void UIElement::ValidateScrollOffset()
{
    SetScrollOffset(m_ScrollOffset);
}

} // dw
} // internal
} // nw
