﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/control/dw_StackPanel.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace internal {
namespace dw {

const f32 StackPanel::SCROLLBAR_SIZE = 3.f;   // HACK : ★スクロールバーの幅（暫定実装）
const nw::ut::Color4f StackPanel::SCROLLBAR_COLOR = NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.52f, 0.64f, 0.80f, 1.f));

StackPanel::StackPanel() :
m_Orientation(VERTICAL)
{
    SetIsFocusable(false);
    SetMargin(Thickness(0.f));
    SetPadding(Thickness(0.f));
}

Orientation StackPanel::GetOrientation() const
{
    return m_Orientation;
}

void StackPanel::SetOrientation(Orientation value)
{
    m_Orientation = value;
}

s32 StackPanel::GetWheelScrollDelta(s32 lines) const
{
    if(GetContents().GetCount() == 0)
    {
        return Base::GetWheelScrollDelta(lines);
    }

    return static_cast<s32>(-GetContents().GetItem(0)->GetFixedHeight()) * lines;
}

void StackPanel::OnEnsureVisible(UIElement& element)
{
    // HACK : ★スクロールオフセットを計算します。（暫定実装）
    const nw::math::Vector2 scrollOffset = GetScrollOffset();
    const nw::math::Vector2 elementTopLeft = element.GetTopLeft();
    const nw::math::Vector2 elementSize = element.GetFixedSize();
    const nw::math::Vector2 size = GetFixedSize();

    f32 deltaLeft   = elementTopLeft.x - scrollOffset.x;
    f32 deltaTop    = elementTopLeft.y - scrollOffset.y;
    f32 deltaRight  = (elementTopLeft.x + elementSize.x) - size.x - scrollOffset.x;
    f32 deltaBottom = (elementTopLeft.y + elementSize.y) - size.y - scrollOffset.y;

    nw::math::Vector2 newScrollOffset(scrollOffset);

    if(deltaLeft < 0)
    {
        newScrollOffset.x += deltaLeft;
    }
    else if(deltaRight > 0)
    {
        newScrollOffset.x += deltaRight;
    }

    if(deltaTop < 0)
    {
        newScrollOffset.y += deltaTop;
    }
    else if(deltaBottom > 0)
    {
        newScrollOffset.y += deltaBottom;
    }

    SetScrollOffset(newScrollOffset);
}

bool StackPanel::OnUpdatePointerInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Mouse& mouse = *inputs.GetMouse();

    s32 wheelDelta = mouse.GetWheel();

    if(wheelDelta != 0)
    {
        nw::math::Vector2 scrollOffset = GetScrollOffset();
        const nw::math::Vector2 measuredSize = GetMeasuredSize();
        const nw::math::Vector2 size = GetFixedSize();

        scrollOffset.y += GetWheelScrollDelta(wheelDelta);
        scrollOffset.y = nw::ut::Min(scrollOffset.y, measuredSize.y - size.y);
        scrollOffset.y = nw::ut::Max(scrollOffset.y, 0.f);

        SetScrollOffset(scrollOffset);

        return true;
    }

    return false;
}

const nw::math::Vector2 StackPanel::OnMeasure(UIRenderer& renderer) const
{
    (void)renderer;

    switch(m_Orientation)
    {
    case HORIZONTAL:
        return MeasureByHorizontal();

    case VERTICAL:
        return MeasureByVertical();

    default:
        NW_ASSERT(false);
        return nw::math::Vector2::Zero();
    }
}

void StackPanel::OnUpdate(const UIElementTreeContext& context)
{
    (void)context;

    switch(m_Orientation)
    {
    case HORIZONTAL:
        UpdateContentsByHorizontal();
        break;

    case VERTICAL:
        UpdateContentsByVertical();
        break;

    default:
        NW_ASSERT(false);
    }
}

void StackPanel::PreRenderContents(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
{
    (void)context;
    (void)renderer;
    (void)args;

    // TODO : ★Scissor 設定
}

void StackPanel::PostRenderContents(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
{
    (void)context;
    (void)renderer;
    (void)args;

    // TODO : ★Scissor 設定解除

    // HACK : ★スクロールバーを描画します。（暫定実装）
    switch(m_Orientation)
    {
    case HORIZONTAL:
        RenderScrollbarByHorizontal(context, renderer);
        break;

    case VERTICAL:
        RenderScrollbarByVertical(context, renderer);
        break;

    default:
        NW_ASSERT(false);
        break;
    }
}

bool StackPanel::CanAlignHorizontal(const UIElement& element) const
{
    (void)element;

    // 横方向 StackPanel の場合は、横方向のアライメントを無視します。
    return m_Orientation != HORIZONTAL;
}

bool StackPanel::CanAlignVertical(const UIElement& element) const
{
    (void)element;

    // 縦方向 StackPanel の場合は、縦方向のアライメントを無視します。
    return m_Orientation != VERTICAL;
}

nw::math::Vector2 StackPanel::MeasureByHorizontal() const
{
    const UIElementList& contents = GetContents();

    // StackPanel ではパディングを無視します。
    float x = 0;
    float y = 0;

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pContent = contents.GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        const nw::math::Vector2 size = pContent->GetMeasuredSize();

        if(size.IsZero())
        {
            continue;
        }

        const Thickness margin = pContent->GetMargin();

        x += size.x + margin.GetWidth();
        y = nw::ut::Max(y, size.y + margin.GetHeight());
    }

    return nw::math::Vector2(x, y);
}

nw::math::Vector2 StackPanel::MeasureByVertical() const
{
    const UIElementList& contents = GetContents();

    // StackPanel ではパディングを無視します。
    float x = 0;
    float y = 0;

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pContent = contents.GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        const nw::math::Vector2 size = pContent->GetMeasuredSize();
        const Thickness margin = pContent->GetMargin();

        x = nw::ut::Max(x, size.x + margin.GetWidth());
        y += size.y + margin.GetHeight();
    }

    return nw::math::Vector2(x, y);
}

void StackPanel::UpdateContentsByHorizontal()
{
    // StackPanel ではパディングを無視します。
    const nw::math::Vector2 topLeft(0.f, 0.f);
    const UIElementList& contents = GetContents();

    float currentX = topLeft.x;

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pContent = contents.GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        const Thickness margin = pContent->GetMargin();

        pContent->SetTopLeft(
            nw::math::Vector2(currentX + margin.left, topLeft.y + margin.top));

        currentX += pContent->GetFixedSize().x + margin.GetWidth();
    }
}

void StackPanel::UpdateContentsByVertical()
{
    // StackPanel ではパディングを無視します。
    const nw::math::Vector2 topLeft(0.f, 0.f);
    const UIElementList& contents = GetContents();

    float currentY = topLeft.y;

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pContent = contents.GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        const Thickness margin = pContent->GetMargin();

        pContent->SetTopLeft(
            nw::math::Vector2(topLeft.x + margin.left, currentY + margin.top));

        currentY += pContent->GetFixedSize().y + margin.GetHeight();
    }
}

void StackPanel::RenderScrollbarByHorizontal(const UIElementTreeContext& context, UIRenderer& renderer) const
{
    const nw::math::Vector2 measuredSize = GetMeasuredSize();
    const nw::math::Vector2 size = GetFixedSize();

    if(measuredSize.x <= size.x)
    {
        return;
    }

    const nw::math::Vector2& scrollOffset = GetScrollOffset();

    f32 left  = GetFixedWidth() * scrollOffset.x / measuredSize.x;
    f32 width = GetFixedWidth() * size.x / measuredSize.x;

    renderer.FillRectangle(
        &context,
        DrawRectangleArgs().
        SetTopLeft(left, size.y - SCROLLBAR_SIZE).
        SetSize(width, SCROLLBAR_SIZE).
        SetColor(SCROLLBAR_COLOR));
}

void StackPanel::RenderScrollbarByVertical(const UIElementTreeContext& context, UIRenderer& renderer) const
{
    const nw::math::Vector2 measuredSize = GetMeasuredSize();
    const nw::math::Vector2 size = GetFixedSize();

    if(measuredSize.y <= size.y)
    {
        return;
    }

    const nw::math::Vector2& scrollOffset = GetScrollOffset();

    f32 top    = GetFixedHeight() * scrollOffset.y / measuredSize.y;
    f32 height = GetFixedHeight() * size.y / measuredSize.y;

    renderer.FillRectangle(
        &context,
        DrawRectangleArgs().
        SetTopLeft(size.x - SCROLLBAR_SIZE, top).
        SetSize(SCROLLBAR_SIZE, height).
        SetColor(SCROLLBAR_COLOR));
}

} // dw
} // internal
} // nw
