﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_Profile.h>

#if defined(NW_DEV_ENABLED) && defined(NW_PROFILE_ENABLED)

#include <nw/dev/dev_Utility.h>
#include <nw/ut/ut_String.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_Flag.h>
#include <nw/ut/ut_Foreach.h>

#ifdef NW_PERFORMANCE_COUNTER_ENABLED
#include <nn/os.h>
#include <nn/dbg/dbg_PerformanceCounterSelect.h>
#endif

namespace nw
{
namespace dev
{

ProfileCenter* ProfileCenter::m_Instance = NULL;
void* ProfileManager::m_NodeMemory = NULL;

//----------------------------------------
void
ProfileReport::DumpReport(u32 mode, ExportFormat exportFormat, float accuracy)
{
    // 名前のないレポートはスキップします。
    if (std::strcmp(name, "") == 0) { return; }

    float elapsedTime = TickToTime(this->elapsedTime, accuracy);
    float maxElapsedTime = TickToTime(this->maxElapsedTime, accuracy);
    float minElapsedTime = TickToTime(this->minElapsedTime, accuracy);
    float averageTime = elapsedTime / static_cast<float>(this->callCount);
    float frame = 10.0f / 60.0f;

    const int MAX_LENGTH = 256;
    char formatString[MAX_LENGTH];
    formatString[0] = '\0';

    char output[256];
    output[0] = '\0';

    if (ut::CheckFlag(mode, TIMER))
    {
        char* format = "|%9.3f|%6d|%7.3f|   %5.2f|%7.3f|%7.3f|";
        ut::snprintf(formatString, MAX_LENGTH, MAX_LENGTH - 1, format,
            elapsedTime,
            callCount,
            averageTime, averageTime / frame,
            maxElapsedTime,
            minElapsedTime);
        ut::strncat(output, MAX_LENGTH, formatString, std::strlen(formatString) + 1);
    }

#ifdef NW_PERFORMANCE_COUNTER_ENABLED
    u64   iCacheMiss = (iCacheMiss / callCount);
    u64   dCacheReadMiss = (dCacheReadMiss / callCount);
    u64   dCacheWriteMiss = (dCacheWriteMiss / callCount);

    if (ut::CheckFlag(mode, I_CACHE_MISS))
    {
        char* format = "%6llu|";
        ut::snprintf(formatString, MAX_LENGTH, MAX_LENGTH - 1, format, iCacheMiss);
        ut::strncat(output, MAX_LENGTH, formatString, std::strlen(formatString) + 1);
    }

    if (ut::CheckFlag(mode, D_CACHE_READ_MISS))
    {
        char* format = "%6llu|";
        ut::snprintf(formatString, MAX_LENGTH, MAX_LENGTH - 1, format, dCacheReadMiss);
        ut::strncat(output, MAX_LENGTH, formatString, std::strlen(formatString) + 1);
    }

    if (ut::CheckFlag(mode, D_CACHE_WRITE_MISS))
    {
        char* format = "%6llu|";
        ut::snprintf(formatString, MAX_LENGTH, MAX_LENGTH - 1, format, dCacheWriteMiss);
        ut::strncat(output, MAX_LENGTH, formatString, std::strlen(formatString) + 1);
    }
#endif

    ut::strncat(output, MAX_LENGTH, this->name, std::strlen(this->name) + 1);
    char* str = "|\n";
    ut::strncat(output, MAX_LENGTH, str, std::strlen(str) + 1);

    if (exportFormat == CSV)
    {
        std::replace(output, output + std::strlen(output), '|', ',');
    }

    NW_DEV_LOG(output);
}

//----------------------------------------
void
ProfileNode::DumpHierarchicalReports(u32 mode, ExportFormat exportFormat, float accuracy)
{
    m_Report.DumpReport(mode, exportFormat, accuracy);

    ProfileNode::ChildrenRange children = GetChildren();
    for (ProfileNode::ChildrenList::iterator child = children.first;
        child != children.second; ++child)
    {
        (*child)->DumpHierarchicalReports(mode, exportFormat, accuracy);
    }
}

//----------------------------------------
void
ProfileNode::TabulateReport(ProfileReportSet& reports)
{
    // 名前のないレポートはスキップします。
    if (std::strcmp(m_Report.name, "") != 0)
    {
        ProfileReportSet::iterator found = reports.find(m_Report);
        if (found != reports.end())
        {
            ProfileReport& report = const_cast<ProfileReport&>(*found);
            report.Combine(m_Report);
        }
        else
        {
            reports.insert(m_Report);
        }
    }

    ProfileNode::ChildrenRange children = GetChildren();
    for (ProfileNode::ChildrenList::iterator child = children.first;
        child != children.second; ++child)
    {
        (*child)->TabulateReport(reports);
    }
}

//----------------------------------------
ProfileManager*
ProfileManager::CreateProfileManager(
    ProfileManager::Description& description,
    ut::IAllocator* allocator)
{
    size_t nodeMemoryCapacity = sizeof(ProfileNode) * (description.maxReport * 2);
    m_NodeMemory = allocator->Alloc(nodeMemoryCapacity);
    ProfileNode::Allocator nodeAllocator(m_NodeMemory, nodeMemoryCapacity);

    void* managerMemory = allocator->Alloc(sizeof(ProfileManager));
    ProfileManager* profileManager = new(managerMemory) ProfileManager(nodeAllocator, description);

    return profileManager;
}

//----------------------------------------
void
ProfileManager::Destroy(ut::IAllocator* allocator)
{
#if defined(NW_PLATFORM_WIN32)
    nn::lmem::HeapHandle heap = this->m_RootNode.GetAllocator().GetHeap();
#elif defined(NW_PLATFORM_CAFE)
    MEMHeapHandle heap = this->m_RootNode.GetAllocator().GetHeap();
#endif

    void* managerMemory = this;
    this->~ProfileManager();
    allocator->Free(managerMemory);


#if defined(NW_PLATFORM_WIN32)
    nn::lmem::DestroyExpHeap(heap);
    allocator->Free(m_NodeMemory);
#elif defined(NW_PLATFORM_CAFE)
    void* nodesMemory = MEMDestroyExpHeap(heap);
    allocator->Free(nodesMemory);
#endif
    m_NodeMemory = NULL;
}

//----------------------------------------
ProfileManager::ProfileManager(
    ProfileNode::Allocator allocator,
    const ProfileManager::Description& description)
: m_Description(description)
, m_RootNode(allocator, "", NULL)
, m_CurrentNode(&m_RootNode)
{
}

//----------------------------------------
ProfileManager::~ProfileManager()
{
}

//----------------------------------------
void
ProfileManager::ClearReports()
{
    m_RootNode.Clear();
    m_CurrentNode = &m_RootNode;
}

//----------------------------------------
void
ProfileManager::DumpReports()
{
    NW_DEV_LOG("\n");

    float accuracy = 0.0f;
    switch (this->m_Description.accuracy)
    {
        case MILI_SECOND:
        {
            accuracy = 1000000.0f;
            NW_DEV_LOG("Time: [ms]\n");
        }
        break;
        case MICRO_SECOND:
        {
            accuracy = 1000.0f;
            NW_DEV_LOG("Time: [us]\n");
        }
        break;
        default:
        {
            NW_FATAL_ERROR("Unsupported accuracy.");
        }
        break;
    }

    const int MAX_LENGTH = 256;
    char caption[MAX_LENGTH];
    caption[0] = '\0';
    char form[MAX_LENGTH];
    form[0] = '\0';
    if (ut::CheckFlag(m_Description.mode, TIMER))
    {
        char* str = "|    total|  call|average|average%%|    max|    min|";
        ut::strncat(caption, MAX_LENGTH, str, std::strlen(str) + 1);
        char* fstr ="|RIGHT:   |RIGHT:|RIGHT: |RIGHT:  |RIGHT: |RIGHT: |";
        ut::strncat(form, MAX_LENGTH, fstr, std::strlen(fstr) + 1);
    }

#ifdef NW_PERFORMANCE_COUNTER_ENABLED
    if (ut::CheckFlag(m_Description.mode, I_CACHE_MISS))
    {
        char* str = " IMiss|";
        ut::strncat(caption, MAX_LENGTH, str, std::strlen(str) + 1);
        char* fstr = "RIGHT:|";
        ut::strncat(form, MAX_LENGTH, fstr, std::strlen(fstr) + 1);
    }

    if (ut::CheckFlag(m_Description.mode, D_CACHE_READ_MISS))
    {
        char* str = "DRMiss|";
        ut::strncat(caption, MAX_LENGTH, str, std::strlen(str) + 1);
        char* fstr = "RIGHT:|";
        ut::strncat(form, MAX_LENGTH, fstr, std::strlen(fstr) + 1);
    }

    if (ut::CheckFlag(m_Description.mode, D_CACHE_WRITE_MISS))
    {
        char* str = "DWMiss|";
        ut::strncat(caption, MAX_LENGTH, str, std::strlen(str) + 1);
        char* fstr = "RIGHT:|";
        ut::strncat(form, MAX_LENGTH, fstr, std::strlen(fstr) + 1);
    }
#endif

    if (this->m_Description.exportFormat == CSV)
    {
        std::replace(caption, caption + std::strlen(caption), '|', ',');
        char* str = "name,\n";
        ut::strncat(caption, MAX_LENGTH, str, std::strlen(str) + 1);

        // caption 出力
        NW_DEV_LOG(caption);
    }
    else
    {
        char* str = "name|h\n";
        ut::strncat(caption, MAX_LENGTH, str, std::strlen(str) + 1);
        char* fstr = "LEFT:|c\n";
        ut::strncat(form, MAX_LENGTH, fstr, std::strlen(fstr) + 1);

        // caption 出力
        NW_DEV_LOG(caption);
        NW_DEV_LOG(form);
    }

#if defined(NW_PROFILE_DUMP_HIERARCHICAL_REPORTS_ENABLED)
    m_RootNode.DumpHierarchicalReports(m_Description.mode, m_Description.exportFormat, accuracy);
#else
    ProfileReportSet::allocator_type reportsAllocator(m_RootNode.GetAllocator());
    ProfileReportSet reports(CompareProfileReport(), reportsAllocator);
    m_RootNode.TabulateReport(reports);

    ProfileReportSet::iterator reportEnd = reports.end();
    for (ProfileReportSet::iterator report = reports.begin(); report != reportEnd; ++report)
    {
        const_cast<ProfileReport&>(*report).DumpReport(m_Description.mode, m_Description.exportFormat, accuracy);
    }
#endif
}

//----------------------------------------
void
ProfileCenter::Initialize(int maxReport, ut::IAllocator* allocator)
{
    NW_ASSERT(m_Instance == NULL);

#ifdef NW_PERFORMANCE_COUNTER_ENABLED
    // プログラムカウンタ初期化
    nn::dbg::MPCore::AcquirePMCControl();
#endif

    ProfileManager::Description description;
    description.maxReport = maxReport;
    description.exportFormat = CHART;
    description.accuracy = MILI_SECOND;
    description.mode = TIMER | I_CACHE_MISS | D_CACHE_READ_MISS;
    ProfileManager* profileManager = ProfileManager::CreateProfileManager(description, (allocator));

    void* memory = allocator->Alloc(sizeof(ProfileCenter));
    NW_ASSERT_NOT_NULL(memory);
    m_Instance = new(memory) ProfileCenter();
    ProfileCenter::SetProfileManager(profileManager);
}

//----------------------------------------
void
ProfileCenter::Finalize(ut::IAllocator* allocator)
{
    ProfileManager* profileManager = ProfileCenter::GetProfileManager();
    if (profileManager)
    {
        profileManager->Destroy(allocator);
        ProfileCenter::SetProfileManager(NULL);
    }

    ut::SafeFree(m_Instance, allocator);

#ifdef NW_PERFORMANCE_COUNTER_ENABLED
    // プログラムカウンタ解放
    nn::dbg::MPCore::ReleasePMCControl();
#endif
}

} // namespace nw::dev
} // namespace nw

#endif
