﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_LoadMeter.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/gx2.h>
#endif

namespace nw
{
namespace dev
{

//------------------------------------------------------------------------------
/* ctor */
LoadMeterBase::LoadMeterBase( Section* buffer, s32 sectionNum, const char* name, nw::ut::Color4u8 color )
 : m_Color( color ),
   m_CurrentBuffer( 0 ),
   m_TopSection( -1 ),
   m_OverNum( 0 )
{
    m_SectionArray[0].Wrap( &buffer[0], sectionNum );
    m_SectionArray[1].Wrap( &buffer[sectionNum], sectionNum );
    m_SectionNum[0] = 0;
    m_SectionNum[1] = 0;
    if( name ) this->SetName( name );
}

//------------------------------------------------------------------------------
void
LoadMeterBase::SetBeginMeasure( nw::ut::Tick tick )
{
    SetBeginMeasure( tick, m_Color );
}

//------------------------------------------------------------------------------
void
LoadMeterBase::SetBeginMeasure( nw::ut::Tick tick, const nw::ut::Color4u8& color )
{
    AddSection( tick, color );
}

//------------------------------------------------------------------------------
void
LoadMeterBase::SetEndMeasure( nw::ut::Tick tick )
{
    m_FinalEnd[ m_CurrentBuffer ] = tick;

    if ( m_OverNum > 0 )
    {
        return;
    }

    Section* section = &m_SectionArray[m_CurrentBuffer].at( m_TopSection );
    if ( static_cast<s64>( tick - section->begin ) < 0 )
    {
        tick = section->begin;
    }

    EndSection( tick );

    m_TopSection =  section->parent;
}

//------------------------------------------------------------------------------
nw::ut::Tick
LoadMeterBase::GetLastTotalSpan() const
{
    nw::ut::Tick tick;

    for( s32 i = 0; i < m_SectionNum[ 1 - m_CurrentBuffer ]; ++i )
    {
        const Section* section = &m_SectionArray[ 1 - m_CurrentBuffer ].at( i );

        // 親要素のみカウントします。
        if ( section->parent == -1 )
        {
            tick += ( section->end - section->begin );
        }
    }

    return tick;
}

//------------------------------------------------------------------------------
void
LoadMeterBase::AddSection( nw::ut::Tick tick, const nw::ut::Color4u8& color )
{
    if ( m_SectionArray[m_CurrentBuffer].size() <= m_SectionNum[ m_CurrentBuffer ] )
    {
        ++m_OverNum;
        return;
    }

    Section* section = &m_SectionArray[m_CurrentBuffer].at( m_SectionNum[ m_CurrentBuffer ] );
    section->begin = section->end = tick;
    section->color = color;
    section->parent = m_TopSection;

    m_TopSection = m_SectionNum[ m_CurrentBuffer ];

    ++m_SectionNum[ m_CurrentBuffer ];
}

//------------------------------------------------------------------------------
void
LoadMeterBase::EndSection( nw::ut::Tick tick )
{
    Section* section = &m_SectionArray[m_CurrentBuffer].at( m_TopSection );
    section->end = tick;
}

//------------------------------------------------------------------------------
void
LoadMeterBase::OnEndFrame()
{
    NW_ASSERT( m_TopSection == -1 );
    NW_ASSERTMSG( m_OverNum == 0, "Section is not enough, Please add %d or more sections.", m_OverNum );

    // 前のフレームの結果を入れ替える.
    m_CurrentBuffer = 1 - m_CurrentBuffer;

    m_TopSection = -1;
    m_OverNum = 0;
    m_SectionNum[ m_CurrentBuffer ] = 0;
}

} // namespace dev
} // namespace nw

