﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_FileDevice.h>
#include <nw/dev/dev_FileDeviceManager.h>
#include <nw/ut/ut_String.h>

#if defined(NW_USE_NINTENDO_SDK)
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#endif
#endif

#if defined( NW_PLATFORM_WIN32 )
  using namespace nw::internal::winext;
#endif

namespace nw
{
namespace dev
{

#if defined(NW_USE_NINTENDO_SDK)
namespace internal
{
namespace
{

//!
//! @brief nn::fs のパス形式への変換を提供します。
//!
class PathUtil
{
public:
    //---------------------------------------------------------------------------
    //! @brief パスを文字列バッファに書き込みます。
    //!
    //! @details
    //! パスは nn::fs の形式に変換されます。
    //!
    //! dst の内容は上書きされます。
    //!
    //! @param[out] dst 文字列バッファです。
    //! @param[in] path 追加するパスです。
    //!
    //! @return dst を返します。
    //---------------------------------------------------------------------------
    static ut::BufferedSafeString&
    SetPath(ut::BufferedSafeString& dst, const char* path)
    {
        NW_NULL_ASSERT(path);

        dst.clear();

        if (!path)
        {
            return dst;
        }

        // nn::fs のパス形式 c:/dir/file.ext に変換します。

        if (IsDirectorySeparator(path[0]) && ::isalpha(path[1]) && IsDirectorySeparator(path[2]))
        {
            // /c を c: に変換します。
            dst.append(path[1]);
            dst.append(':');
            path = path + 2;
        }

        size_t length = dst.getBufferSize();
        for (size_t i = 0; i < length && path[i] != '\0'; ++i)
        {
            switch (char c = path[i])
            {
            case '\\':
                dst.append('/');
                break;

            default:
                dst.append(c);
                break;
            }
        }

        return dst;
    }

    //---------------------------------------------------------------------------
    //! @brief パスを文字列バッファの末尾に追加します。
    //!
    //! @details
    //! パスは nn::fs の形式に変換されます。
    //!
    //! @param[out] dst 文字列バッファです。
    //! @param[in] path 追加するパスです。
    //!
    //! @return dst を返します。
    //---------------------------------------------------------------------------
    static ut::BufferedSafeString&
    AppendPath(ut::BufferedSafeString& dst, const char* path)
    {
        NW_NULL_ASSERT(path);

        if (!path)
        {
            return dst;
        }

        if (!IsDirectorySeparator(path[0]))
        {
            dst.append('/');
        }

        size_t length = dst.getBufferSize();
        for (size_t i = 0; i < length && path[i] != '\0'; ++i)
        {
            switch (char c = path[i])
            {
            case '\\':
                dst.append('/');
                break;

            default:
                dst.append(c);
                break;
            }
        }

        return dst;
    }

    //---------------------------------------------------------------------------
    //! @brief パスを連結し文字列バッファに書き込みます。
    //!
    //! @details
    //! path1 の末尾に path2 を連結し dst に格納します。 path2 が絶対パスのときは path1 は使われません。
    //!
    //! パスは nn::fs の形式に変換されます。
    //!
    //! @param[out] dst 連結されたパスが格納されます。
    //! @param[in] path1 連結する最初のパスです。
    //! @param[in] path2 連結する次のパスです。
    //!
    //! @return dst を返します。
    //---------------------------------------------------------------------------
    static ut::BufferedSafeString&
    JoinPath(ut::BufferedSafeString& dst, const char* path1, const char* path2)
    {
        NW_NULL_ASSERT(path1);
        NW_NULL_ASSERT(path2);

        dst.clear();

        if (!path1 || !path2)
        {
            return dst;
        }

        if (IsDirectorySeparator(path2[0]) || (::isalpha(path2[0]) && path2[1] == ':'))
        {
            // path2 が絶対パスの場合。
            SetPath(dst, path2);
        }
        else
        {
            SetPath(dst, path1);
            AppendPath(dst, path2);
        }

        return dst;
    }

private:
    static bool IsDirectorySeparator(int c)
    {
        return c == '/' || c == '\\';
    }
};

} // namespace {anonymous}
} // namespace internal
#endif // NW_USE_NINTENDO_SDK

struct FileDevice::FileHandleInner
{
#if defined(NW_USE_NINTENDO_SDK)
    nn::fs::FileHandle m_Handle;
    size_t m_Position;
#else
    FSFileHandle m_Handle;
    FSFilePosition m_Position;
#endif
};

struct FileDevice::DirHandleInner
{
#if defined(NW_USE_NINTENDO_SDK)
    nn::fs::DirectoryHandle m_Handle;
#else
    FSDirHandle m_Handle;
#endif
};

//------------------------------------------------------------------------------
FileHandle::~FileHandle()
{
    if(m_Device)
    {
        m_Device->Close(this);
    }
}

//------------------------------------------------------------------------------
bool
FileHandle::Close()
{
    if ( m_Device )
    {
        return m_Device->Close(this);
    }
    NW_ERR( "handle not opened" );
    return false;
}

//------------------------------------------------------------------------------
u32
FileHandle::Read(u8* buf, u32 size)
{
    if ( m_Device )
    {
        return m_Device->Read(this, buf, size);
    }
    NW_ERR( "handle not opened" );
    return 0;
}

//------------------------------------------------------------------------------
bool
FileHandle::TryRead(u32* readSize, u8* buf, u32 size)
{
    if ( m_Device )
    {
        return m_Device->TryRead(readSize, this, buf, size);
    }
    NW_ERR( "handle not opened" );
    return false;
}

//------------------------------------------------------------------------------
bool
FileHandle::Seek(s32 offset, SeekOrigin origin)
{
    if ( m_Device )
    {
        return m_Device->Seek(this, offset, origin);
    }
    NW_ERR( "handle not opened" );
    return false;
}

//------------------------------------------------------------------------------
bool
FileHandle::TrySeek(s32 offset, SeekOrigin origin)
{
    if ( m_Device )
    {
        return m_Device->TrySeek(this, offset, origin);
    }
    NW_ERR( "handle not opened" );
    return false;
}

//------------------------------------------------------------------------------
u32
FileHandle::GetCurrentSeekPos()
{
    if ( m_Device )
    {
        return m_Device->GetCurrentSeekPos(this);
    }
    NW_ERR( "handle not opened" );
    return 0;
}

//------------------------------------------------------------------------------
bool
FileHandle::TryGetCurrentSeekPos(u32* pos)
{
    if (m_Device)
    {
        return m_Device->TryGetCurrentSeekPos(pos, this);
    }
    NW_ERR( "handle not opened" );
    return false;
}

//------------------------------------------------------------------------------
u32
FileHandle::GetFileSize()
{
    if (m_Device)
    {
        return m_Device->GetFileSize(this);
    }
    NW_ERR( "handle not opened" );
    return 0;
}

//------------------------------------------------------------------------------
bool
FileHandle::TryGetFileSize(u32* size)
{
    if ( m_Device )
    {
        return m_Device->TryGetFileSize(size, this);
    }
    NW_ERR( "handle not opened" );
    return false;
}


//------------------------------------------------------------------------------
bool
DirectoryHandle::Close()
{
    if ( m_Device )
    {
        return m_Device->CloseDirectory(this);
    }
    NW_ERR("handle not opened");
    return false;
}

//------------------------------------------------------------------------------
u32
DirectoryHandle::Read(DirectoryEntry entry[], u32 num)
{
    if ( m_Device )
    {
        return m_Device->ReadDirectory(this, entry, num);
    }
    NW_ERR("handle not opened");
    return 0;
}

//------------------------------------------------------------------------------
bool
DirectoryHandle::TryRead(u32* readNum, DirectoryEntry entry[], u32 num)
{
    if ( m_Device )
    {
        return m_Device->TryReadDirectory(readNum, this, entry, num);
    }
    NW_ERR("handle not opened");
    return false;
}


//------------------------------------------------------------------------------
#if defined(NW_USE_NINTENDO_SDK)
FileDevice::FileDevice()
    : m_LastRawError(0)
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    ut::FixedSafeString<m_CWDPath.array_size> tmp;

    DWORD retval = GetEnvironmentVariableA("CAFE_CONTENT_DIR", tmp.getBuffer(), tmp.getBufferSize());

    if (retval == 0)
    {
        retval = GetCurrentDirectoryA(tmp.getBufferSize(), tmp.getBuffer());
    }

    NW_ASSERT_MINMAXLT(retval, 1, static_cast<DWORD>(tmp.getBufferSize()));
    if (0 < retval && retval < static_cast<DWORD>(tmp.getBufferSize()))
    {
        internal::PathUtil::SetPath(m_CWDPath, tmp.c_str());
    }
    else
    {
        m_CWDPath = "";
    }
#else
#error "not implemented"
#endif
}
#else
FileDevice::FileDevice()
: m_CWDPath( nw::ut::SafeString(FS_CONTENT_DIR) ),
  m_LastRawError( 0 )
{}
#endif

#if defined(NW_USE_NINTENDO_SDK)
//------------------------------------------------------------------------------
FileDevice::FileDevice( const char* CWDPath )
: m_LastRawError( 0 )
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    ut::FixedSafeString<m_CWDPath.array_size> tmp;
    GetCurrentDirectoryA(tmp.getBufferSize(), tmp.getBuffer());
    if (CWDPath)
    {
        internal::PathUtil::JoinPath(m_CWDPath, tmp.c_str(), CWDPath);
    }
    else
    {
        internal::PathUtil::SetPath(m_CWDPath, tmp.c_str());
    }
#else
#error "not implemented"
#endif
}
#else
//------------------------------------------------------------------------------
FileDevice::FileDevice( const char* CWDPath )
: m_CWDPath( nw::ut::SafeString(CWDPath) ),
  m_LastRawError( 0 )
{}
#endif

//------------------------------------------------------------------------------
u8*
FileDevice::TryLoad(LoadArg& arg)
{
    return DoLoad(arg);
}

//------------------------------------------------------------------------------
FileDevice*
FileDevice::TryOpen(
    FileHandle* handle,
    const char* fileName,
    FileOpenFlag flag
)
{
    NW_ASSERT_NOT_NULL( handle );

    FileDevice* device = DoOpen(handle, fileName, flag);

    SetHandleBaseFileDevice(handle, device);
    return device;
}

//------------------------------------------------------------------------------
bool
FileDevice::Close(FileHandle* handle)
{
    NW_ASSERT_NOT_NULL( handle );

    bool success = DoClose(handle);
    if ( success )
    {
        SetHandleBaseFileDevice(handle, NULL);
    }

    return success;
}

//------------------------------------------------------------------------------
bool
FileDevice::TryRead(u32* readSize, FileHandle* handle, u8* buf, u32 size)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( buf );

    return DoRead(readSize, handle, buf, size);
}

//------------------------------------------------------------------------------
bool
FileDevice::TrySeek(FileHandle* handle, s32 offset, FileHandle::SeekOrigin origin)
{
    NW_ASSERT_NOT_NULL( handle );

    return DoSeek(handle, offset, origin);
}

//------------------------------------------------------------------------------
bool
FileDevice::TryGetCurrentSeekPos(u32* pos, FileHandle* handle)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( pos );

    return DoGetCurrentSeekPos(pos, handle);
}

//------------------------------------------------------------------------------
bool
FileDevice::TryGetFileSize(u32* size, FileHandle* handle)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( size );

    return DoGetFileSize(size, handle);
}

//------------------------------------------------------------------------------
FileDevice*
FileDevice::TryOpenDirectory(DirectoryHandle* handle, const char* dirName)
{
    NW_ASSERT_NOT_NULL( handle );

    FileDevice* device = DoOpenDirectory(handle, dirName);
    SetHandleBaseFileDevice(handle, device);
    return device;
}

//------------------------------------------------------------------------------
bool
FileDevice::CloseDirectory(DirectoryHandle* handle)
{
    NW_ASSERT_NOT_NULL( handle );

    bool success = DoCloseDirectory(handle);

    if( success )
    {
        SetHandleBaseFileDevice(handle, NULL);
    }

    return success;
}

//------------------------------------------------------------------------------
bool
FileDevice::TryReadDirectory(u32* readNum, DirectoryHandle* handle, DirectoryEntry entry[], u32 num)
{
    NW_ASSERT_NOT_NULL( handle );

    u32 tmpReadNum = 0;
    bool success = DoReadDirectory(&tmpReadNum, handle, entry, num);

    if( readNum )
    {
        *readNum = tmpReadNum;
    }

    if ( tmpReadNum > num )
    {
        NW_ERR( "buffer overflow" );
        return false;
    }

    return success;
}

//------------------------------------------------------------------------------
bool
FileDevice::SetCurrentDir(const char* path)
{
#if defined(NW_USE_NINTENDO_SDK)
  #if defined(NN_BUILD_CONFIG_OS_WIN32)
    return SetCurrentDirectoryA(path) != 0;
  #else
    #error "not implemented"
  #endif
#else
    FSClient* client = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    FSStatus status = FSChangeDir( client, cmdBlock, path, FS_RET_ALL_ERROR );
    return (status == FS_STATUS_OK);
#endif
}

//------------------------------------------------------------------------------
u8*
FileDevice::DoLoad(LoadArg& arg)
{
    // オープン
    FileHandle handle;

    FileDevice* dev = TryOpen(
        &handle,
        arg.path,
        FileDevice::FILE_OPEN_FLAG_READ_ONLY
        );

    if ( !dev )
    {
        return NULL;
    }

    // ファイルサイズ
    u32 buffer_size = arg.bufferSize;
    if ( buffer_size == 0 )
    {
        u32 file_size = 0;
        bool success = TryGetFileSize(&file_size, &handle);

        if (!success)
        {
            return NULL;
        }

        NW_ASSERT( file_size != 0 );

        // 少なくとも FileDevice::BUFFER_MIN_ALIGNMENT の倍数サイズにする
        u32 roundup_size = nw::ut::RoundUp( file_size, FileDevice::BUFFER_MIN_ALIGNMENT );
        buffer_size = roundup_size;
    }

    // メモリ確保
    u8* buffer = arg.buffer;
    bool need_unload = false;
    if ( !buffer )
    {
        // 少なくとも BUFFER_MIN_ALIGNMENT バイトアライメントする
        buffer = static_cast<u8*>( arg.allocator->Alloc( buffer_size, nw::ut::Max<s32>( arg.alignment, BUFFER_MIN_ALIGNMENT ) ) );

        // Unload で delete する
        need_unload = true;
    }

    // 読み込み
    u32 readSize = 0;
    bool success = TryRead(&readSize, &handle, buffer, buffer_size);
    if ( !success )
    {
        if ( need_unload )
        {
            nw::ut::SafeFree( buffer, arg.allocator );
        }
        return NULL;
    }

    // クローズ
    success = Close(&handle);
    if ( !success )
    {
        if ( need_unload )
        {
            nw::ut::SafeFree( buffer, arg.allocator );
        }
        return NULL;
    }

    // 成功したので、LoadArg に情報を返す
    arg.readSize = readSize;
    arg.roundupSize = buffer_size;
    arg.needUnload = need_unload;

    return buffer;
}

//------------------------------------------------------------------------------
FileDevice*
FileDevice::DoOpen( FileHandle* handle, const char* fileName, FileOpenFlag flag )
{
#if defined(NW_USE_NINTENDO_SDK)
    FileHandleInner* handleInner = GetFileHandleInner(handle);

    ut::FixedSafeString<m_CWDPath.array_size> buf;
    internal::PathUtil::JoinPath(buf, m_CWDPath.c_str(), fileName);
    const char* filePath = buf.c_str();

    nn::Result result;
    switch (flag)
    {
    default:
        NW_ERR("illegal open flag[%d]", flag);
        NN_FALL_THROUGH;

    case FILE_OPEN_FLAG_READ_ONLY:  //!< 読み込みのみ
        result = nn::fs::OpenFile(&(handleInner->m_Handle), filePath, nn::fs::OpenMode_Read);
        NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::OpenFile", filePath, result.GetDescription());
        break;

    case FILE_OPEN_FLAG_WRITE_ONLY: //!< 書き込みのみ（ファイルがあっても上書き。ない場合はcFileOpenFlag_Createと同じ動作）
        {
            nn::fs::OpenMode openMode = nn::fs::OpenMode_Write;
            result = nn::fs::OpenFile(&(handleInner->m_Handle), filePath, openMode);
            if (result.IsSuccess())
            {
                result = nn::fs::SetFileSize(handleInner->m_Handle, 0 /*size*/);
                NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::SetFileSize", filePath, result.GetDescription());
            }
            else if (nn::fs::ResultPathNotFound::Includes(result))
            {
                result = nn::fs::CreateFile(filePath, 0 /*size*/);
                NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::CreateFile", filePath, result.GetDescription());
                if (result.IsSuccess())
                {
                    result = nn::fs::OpenFile(&(handleInner->m_Handle), filePath, openMode);
                    NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::OpenFile", filePath, result.GetDescription());
                }
            }
            else
            {
                NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::OpenFile", filePath, result.GetDescription());
            }
        }
        break;

    case FILE_OPEN_FLAG_READ_WRITE: //!< 読み書き
        {
            nn::fs::OpenMode openMode = nn::fs::OpenMode(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
            result = nn::fs::OpenFile(&(handleInner->m_Handle), filePath, openMode);
            NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::OpenFile", filePath, result.GetDescription());
        }
        break;

    case FILE_OPEN_FLAG_CREATE:     //!< 書き込み（すでにファイルがあるとエラー）
        {
            nn::fs::OpenMode openMode = nn::fs::OpenMode_Write;
            result = nn::fs::OpenFile(&(handleInner->m_Handle), filePath, openMode);
            if (result.IsSuccess())
            {
                // すでにファイルがあるときはエラー
                nn::fs::CloseFile(handleInner->m_Handle);
                return NULL;
            }
            else if (nn::fs::ResultPathNotFound::Includes(result))
            {
                result = nn::fs::CreateFile(filePath, 0 /*size*/);
                NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::CreateFile", filePath, result.GetDescription());
                if (result.IsSuccess())
                {
                    result = nn::fs::OpenFile(&(handleInner->m_Handle), filePath, openMode);
                    NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::OpenFile", filePath, result.GetDescription());
                }
            }
            else
            {
                NW_WARNING(result.IsSuccess(), "%s failed: %s : %d\n", "nn::fs::OpenFile", filePath, result.GetDescription());
            }
        }
        break;
    };

    handleInner->m_Position = 0;

    if (result.IsFailure())
    {
        m_LastRawError = result.GetDescription();
        return NULL;
    }

    return this;
#else
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();

    FileHandleInner* handle_inner = GetFileHandleInner(handle);

    const char* flag_str;
    switch (flag)
    {
    case FILE_OPEN_FLAG_READ_ONLY:  //!< 読み込みのみ
        flag_str = "r";
        break;

    case FILE_OPEN_FLAG_WRITE_ONLY: //!< 書き込みのみ（ファイルがあっても上書き。ない場合はcFileOpenFlag_Createと同じ動作）
        flag_str = "w";
        break;

    case FILE_OPEN_FLAG_READ_WRITE: //!< 読み書き
        flag_str = "r+";
        break;

    case FILE_OPEN_FLAG_CREATE:     //!< 書き込み（すでにファイルがあるとエラー）
        flag_str = "w+";
        break;

    default:
        NW_ERR( "illegal open flag[%d]", flag );
        flag_str = "r";
        break;
    };

    static const u32 FORMAT_BUFFER_SIZE = 256;
    char buf[FORMAT_BUFFER_SIZE];
    const char* filePath = fileName;

    if ( fileName[0] != '/' )
    {
        nw::ut::snprintf( buf, FORMAT_BUFFER_SIZE, FORMAT_BUFFER_SIZE - 1, "%s/%s", m_CWDPath.c_str(), fileName );
        filePath = buf;
    }

    FSStatus status = FSOpenFile( client, cmdBlock, filePath, flag_str, &handle_inner->m_Handle, FS_RET_ALL_ERROR );
    handle_inner->m_Position = 0;

    if (status != FS_STATUS_OK)
    {
        NW_WARNING( false, "FSOpenFile failed: %s : %d\n", filePath, status );
        handle_inner->m_Handle = 0;
        m_LastRawError = status;
        return NULL;
    }

    return this;
#endif
}

//------------------------------------------------------------------------------
bool
FileDevice::DoClose(FileHandle* handle)
{
#if defined(NW_USE_NINTENDO_SDK)
    FileHandleInner* handleInner = GetFileHandleInner(handle);

    nn::fs::CloseFile(handleInner->m_Handle);

    return true;
#else
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    FileHandleInner* handle_inner = GetFileHandleInner(handle);

    FSStatus status = FSCloseFile(client, cmdBlock, handle_inner->m_Handle, FS_RET_ALL_ERROR);

    if (status != FS_STATUS_OK)
    {
        NW_WARNING( false, "FSCloseFile failed: %d\n", status );
        m_LastRawError = status;
        return false;
    }

    return true;
#endif
}

//------------------------------------------------------------------------------
bool
FileDevice::DoRead(u32* readSize, FileHandle* handle, u8* buf, u32 size)
{
#if defined(NW_USE_NINTENDO_SDK)
    nn::Result result;
    s64 fileSize;
    FileHandleInner* handleInner = GetFileHandleInner(handle);

    result = nn::fs::GetFileSize(&fileSize, handleInner->m_Handle);
    if (result.IsFailure())
    {
        NW_WARNING(false, "nn::fs::GetFileSize() failed: %d\n", result.GetDescription());
        m_LastRawError = result.GetDescription();

        if (readSize)
        {
            *readSize = 0;
        }

        return false;
    }

    if (fileSize <= handleInner->m_Position)
    {
        if (readSize)
        {
            *readSize = 0;
        }

        return true;
    }

    size_t readableSize = ut::Min(size_t(fileSize - handleInner->m_Position), size_t(size));
    result = nn::fs::ReadFile(handleInner->m_Handle, handleInner->m_Position, buf, readableSize);
    if (result.IsFailure())
    {
        NW_WARNING(false, "nn::fs::ReadFile() failed: %d\n", result.GetDescription());
        m_LastRawError = result.GetDescription();
        if (readSize)
        {
            *readSize = 0;
        }
        return false;
    }

    handleInner->m_Position += readableSize;

    if (readSize)
    {
        *readSize = readableSize;
    }

    return true;
#else
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    FileHandleInner* handle_inner = GetFileHandleInner(handle);

    s32 result = FSReadFile(client, cmdBlock, buf, 1, size, handle_inner->m_Handle, 0, FS_RET_ALL_ERROR);

    if (result >= FS_STATUS_OK)
    {
        handle_inner->m_Position += result;

        if (readSize)
        {
            *readSize = result;
        }

        return true;
    }
    else
    {
        NW_WARNING( false, "FSReadFile failed: %d\n", result );
        m_LastRawError = result;
        return false;
    }
#endif
}

//------------------------------------------------------------------------------
bool
FileDevice::DoSeek(FileHandle* handle, s32 offset, FileHandle::SeekOrigin origin)
{
#if defined(NW_USE_NINTENDO_SDK)
    FileHandleInner* handleInner = GetFileHandleInner(handle);

    switch (origin)
    {
    case FileHandle::SEEK_ORIGIN_BEGIN:
        break;

    case FileHandle::SEEK_ORIGIN_CURRENT:
        offset += handleInner->m_Position;
        break;

    case FileHandle::SEEK_ORIGIN_END:
        NW_ASSERT(offset <= 0);
        {
            u32 size = 0;
            if (DoGetFileSize(&size, handle))
            {
                offset = size + offset;
            }
            else
            {
                return false;
            }
        }
        break;

    default:
        return false;
    }

    handleInner->m_Position = offset;
    return true;
#else
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    FileHandleInner* handle_inner = GetFileHandleInner(handle);

    switch (origin)
    {
    case FileHandle::SEEK_ORIGIN_BEGIN:
        break;

    case FileHandle::SEEK_ORIGIN_CURRENT:
        offset += handle_inner->m_Position;
        break;

    case FileHandle::SEEK_ORIGIN_END:
        NW_ASSERT(offset <= 0);
        {
            u32 size = 0;
            if ( DoGetFileSize(&size, handle) )
            {
                offset = size + offset;
            }
            else
            {
                return false;
            }
        }
        break;

    default:
        return false;
    }

    FSStatus status = FSSetPosFile( client, cmdBlock, handle_inner->m_Handle, offset, FS_RET_ALL_ERROR );

    if (status == FS_STATUS_OK)
    {
        handle_inner->m_Position = offset;
        return true;
    }
    else
    {
        NW_WARNING( false, "FSSetPosFile failed: %d\n", status );
        m_LastRawError = status;
        return false;
    }
#endif
}

//------------------------------------------------------------------------------
bool
FileDevice::DoGetCurrentSeekPos(u32* pos, FileHandle* handle)
{
    FileHandleInner* handle_inner = GetFileHandleInner(handle);
    *pos = handle_inner->m_Position;
    return true;
}

//------------------------------------------------------------------------------
bool
FileDevice::DoGetFileSize(u32* size, FileHandle* handle)
{
#if defined(NW_USE_NINTENDO_SDK)
    nn::Result result;
    FileHandleInner* handleInner = GetFileHandleInner(handle);

    int64_t outValue;
    result = nn::fs::GetFileSize(&outValue, handleInner->m_Handle);

    if (result.IsFailure())
    {
        NW_WARNING(false, "nn::fs::GetFileSize() failed: %d\n", result.GetDescription());
        m_LastRawError = result.GetDescription();
        return false;
    }

    if (UINT_MAX <= outValue)
    {
        NW_WARNING(false, "nn::fs::GetFileSize() too big: %ld\n", outValue);
        return false;
    }

    if (size)
    {
        *size = static_cast<u32>(outValue);
    }

    return true;
#else
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    FileHandleInner* handle_inner = GetFileHandleInner(handle);

    FSStat stat;
    FSStatus status = FSGetStatFile( client, cmdBlock, handle_inner->m_Handle, &stat, FS_RET_ALL_ERROR );

    if (status != FS_STATUS_OK)
    {
        NW_WARNING( false, "FSStatFile failed: %d\n", status );
        m_LastRawError = status;
        return false;
    }
    else
    {
        *size = stat.size;
        return true;
    }
#endif
}

//------------------------------------------------------------------------------
FileDevice*
FileDevice::DoOpenDirectory(DirectoryHandle* handle, const char* dirName)
{
#if defined(NW_USE_NINTENDO_SDK)
    nn::Result result;
    DirHandleInner* handleInner = GetDirHandleInner(handle);

    ut::FixedSafeString<m_CWDPath.array_size> buf;
    internal::PathUtil::JoinPath(buf, m_CWDPath.c_str(), dirName);

    result = nn::fs::OpenDirectory(&handleInner->m_Handle, buf.c_str(), nn::fs::OpenDirectoryMode());

    if (result.IsFailure())
    {
        NW_WARNING(false, "nn::fs::OpenDirectory() failed: %d\n", result.GetDescription());
        m_LastRawError = result.GetDescription();
        return NULL;
    }

    return this;
#else
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    DirHandleInner* dirHandle = GetDirHandleInner(handle);

    static const u32 FORMAT_BUFFER_SIZE = 256;
    char buf[FORMAT_BUFFER_SIZE];
    nw::ut::snprintf( buf, FORMAT_BUFFER_SIZE, FORMAT_BUFFER_SIZE - 1, "%s/%s", m_CWDPath.c_str(), dirName );

    FSStatus status = FSOpenDir( client, cmdBlock, buf, &dirHandle->m_Handle, FS_RET_ALL_ERROR );

    if (status != FS_STATUS_OK)
    {
        NW_WARNING( false, "FSOpenDir failed: %d\n", status );
        m_LastRawError = status;
        return NULL;
    }

    return this;
#endif
}
//------------------------------------------------------------------------------
bool
FileDevice::DoCloseDirectory(DirectoryHandle* handle)
{
#if defined(NW_USE_NINTENDO_SDK)
    DirHandleInner* handleInner = GetDirHandleInner(handle);

    nn::fs::CloseDirectory(handleInner->m_Handle);

    return true;
#else
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    DirHandleInner* dirHandle = GetDirHandleInner(handle);

    FSStatus status = FSCloseDir(client, cmdBlock, dirHandle->m_Handle, FS_RET_ALL_ERROR);

    if (status != FS_STATUS_OK)
    {
        NW_WARNING( false, "FSCloseDir failed: %d\n", status );
        m_LastRawError = status;
        return false;
    }

    return true;
#endif
}
//------------------------------------------------------------------------------
bool
FileDevice::DoReadDirectory(u32* readNum, DirectoryHandle* handle, DirectoryEntry entry[], u32 num)
{
#if defined(NW_USE_NINTENDO_SDK)
    DirHandleInner* handleInner = GetDirHandleInner(handle);

    u32 i;
    for (i = 0; i < num; ++i)
    {
        nn::fs::DirectoryEntry dirEntry[1];
        int64_t numRead = 0;

        nn::Result result = nn::fs::ReadDirectory(&numRead, dirEntry, handleInner->m_Handle, 1);

        if (result.IsFailure())
        {
            NW_WARNING(false, "nn::fs::ReadDirectory() failed: %d\n", result.GetDescription());
            m_LastRawError = result.GetDescription();
            return false;
        }

        if (numRead < 1)
        {
            break;
        }

        entry[i].name = ut::SafeString(dirEntry[0].name);
        entry[i].isDirectory = (dirEntry[0].directoryEntryType == nn::fs::DirectoryEntryType_Directory);
    }

    if (readNum)
    {
        *readNum = i;
    }

    return true;
#else
    FSDirEntry dirEntry;
    FSClient* client     = FileDeviceManager::GetInstance()->GetFSClient();
    FSCmdBlock* cmdBlock = FileDeviceManager::GetInstance()->GetFSCmdBlock();
    DirHandleInner* dirHandle = GetDirHandleInner(handle);

    for ( u32 i = 0; i < num; ++i )
    {
        FSStatus status = FSReadDir(client, cmdBlock, dirHandle->m_Handle, &dirEntry, FS_RET_ALL_ERROR);

        if (status != FS_STATUS_OK)
        {
            if (readNum)
            {
                *readNum = i;
            }

            if (status == FS_STATUS_END)
            {
                // 最後のファイルまで読み込んだ
                return true;
            }
            else
            {
                NW_WARNING( false, "FSReadDir failed: %d\n", status );
                m_LastRawError = status;
                return false;
            }
        }

        entry[i].name = nw::ut::SafeString(dirEntry.name);
        entry[i].isDirectory = ( (dirEntry.stat.flag & FS_STAT_FLAG_IS_DIRECTORY) != 0 );
    }

    if (readNum)
    {
        *readNum = num;
    }

    return true;
#endif
}

//------------------------------------------------------------------------------
FileDevice::FileHandleInner*
FileDevice::GetFileHandleInner(FileHandle* handle)
{
    NW_COMPILER_ASSERT( sizeof(FileHandleInner) <= HANDLE_BYTES_MAX );

    return reinterpret_cast<FileHandleInner*>(&GetHandleBaseHandleBuffer(handle)[0]);
}

//------------------------------------------------------------------------------
FileDevice::DirHandleInner*
FileDevice::GetDirHandleInner(DirectoryHandle* handle)
{
    NW_COMPILER_ASSERT( sizeof(DirHandleInner) <= HANDLE_BYTES_MAX );

    return reinterpret_cast<DirHandleInner*>(&GetHandleBaseHandleBuffer(handle)[0]);
}

} // namespace dev
} // namespace nw

