﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/cafe/dev_PrimitiveRendererCafe.h>
#include <nw/ut/ut_Memory.h>
#include <nw/dev/dev_FileDeviceManager.h>
#include <cafe/gfd.h>

namespace nw
{
namespace dev
{

const s32 MAX_SPHERE_X = 16;
const s32 MAX_SPHERE_Y = 8;
const s32 MAX_DISK_DIVISION = 32;

//--------------------------------------------------------------------------
PrimitiveRendererCafe::PrimitiveRendererCafe( nw::ut::IAllocator* /*allocator*/ )
      : m_VertexShader( NULL ),
        m_PixelShader( NULL ),
        m_VertexProgram( NULL ),
        m_PixelProgram( NULL ),
        m_FetchShaderBufPtr( NULL ),
        m_ViewMtx( NULL ),
        m_ProjMtx( NULL ),
        m_Allocator( NULL ),
        m_ParamWVPOffset( 0 ),
        m_ParamUserOffset( 0 ),
        m_ParamRateOffset( 0 ),
        m_ParamColor0Offset( 0 ),
        m_ParamColor1Offset( 0 ),
        m_ParamUVSrcOffset( 0 ),
        m_ParamUVSizeOffset( 0 ),
        m_ParamTexLocation( 0 ),
        m_AttrVertexLocation( 0 ),
        m_AttrTexCoord0Location( 0 ),
        m_AttrColorRateLocation( 0 ),
        m_TriangleIndexBuf( NULL ),
        m_QuadVertexBuf( NULL ),
        m_QuadIndexBuf( NULL ),
        m_BoxIndexBuf( NULL ),
        m_LineVertexBuf( NULL ),
        m_LineIndexBuf( NULL ),
        m_CubeVertexBuf( NULL ),
        m_CubeIndexBuf( NULL ),
        m_WireCubeVertexBuf( NULL ),
        m_WireCubeIndexBuf( NULL ),
        m_SphereSVertexBuf( NULL ),
        m_SphereSIndexBuf( NULL ),
        m_SphereLVertexBuf( NULL ),
        m_SphereLIndexBuf( NULL ),
        m_DiskSVertexBuf( NULL ),
        m_DiskLVertexBuf( NULL ),
        m_CircleSIndexBuf( NULL ),
        m_CircleLIndexBuf( NULL )
{
}

//--------------------------------------------------------------------------
PrimitiveRendererCafe::~PrimitiveRendererCafe()
{
    if ( m_Allocator )
    {
        nw::ut::SafeFree( m_VertexShader, m_Allocator );
        nw::ut::SafeFree( m_PixelShader, m_Allocator );
        if ( m_VertexProgram ) GX2NotifyMemFree( m_VertexProgram );
        if ( m_PixelProgram ) GX2NotifyMemFree( m_PixelProgram );
        nw::ut::SafeFree( m_VertexProgram, m_Allocator );
        nw::ut::SafeFree( m_PixelProgram, m_Allocator );
        nw::ut::SafeFree( m_FetchShaderBufPtr, m_Allocator );
        nw::ut::SafeFree( m_TriangleIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_QuadVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_QuadIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_BoxIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_LineVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_LineIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_CubeVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_CubeIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_WireCubeVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_WireCubeIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_SphereSVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_SphereSIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_SphereLVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_SphereLIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_DiskSVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_DiskLVertexBuf, m_Allocator );
        nw::ut::SafeFree( m_CircleSIndexBuf, m_Allocator );
        nw::ut::SafeFree( m_CircleLIndexBuf, m_Allocator );
    }
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::InitializeFromBinaryImpl(
    nw::ut::IAllocator* allocator,
    const void* binaryData,
    u32 binarySize
)
{
    bool result;
    NW_UNUSED_VARIABLE( result ); // Release では使用しない。

    m_Allocator = allocator;

    // VertexShader、PixelShader のセットアップ
    result = this->ReadVertexShader( allocator, &m_VertexShader, 0, const_cast<void*>(binaryData) );
    NW_ASSERT( result );
    result = this->ReadPixelShader( allocator, &m_PixelShader, 0, const_cast<void*>(binaryData) );
    NW_ASSERT( result );

    m_ParamWVPOffset = static_cast<u32>( GX2GetVertexUniformVarOffset( m_VertexShader, "wvp" ) );
    NW_ASSERT( m_ParamWVPOffset != GX2_UNIFORM_VAR_INVALID_OFFSET );
    m_ParamUserOffset = static_cast<u32>( GX2GetVertexUniformVarOffset( m_VertexShader, "user" ) );
    NW_ASSERT( m_ParamUserOffset != GX2_UNIFORM_VAR_INVALID_OFFSET );
    m_ParamColor0Offset = static_cast<u32>( GX2GetVertexUniformVarOffset( m_VertexShader, "color0" ) );
    NW_ASSERT( m_ParamColor0Offset != GX2_UNIFORM_VAR_INVALID_OFFSET );
    m_ParamColor1Offset = static_cast<u32>( GX2GetVertexUniformVarOffset( m_VertexShader, "color1" ) );
    NW_ASSERT( m_ParamColor1Offset != GX2_UNIFORM_VAR_INVALID_OFFSET );
    m_ParamUVSrcOffset = static_cast<u32>( GX2GetVertexUniformVarOffset( m_VertexShader, "uv_src" ) );
    NW_ASSERT( m_ParamUVSrcOffset != GX2_UNIFORM_VAR_INVALID_OFFSET );
    m_ParamUVSizeOffset = static_cast<u32>( GX2GetVertexUniformVarOffset( m_VertexShader, "uv_size" ) );
    NW_ASSERT( m_ParamUVSizeOffset != GX2_UNIFORM_VAR_INVALID_OFFSET );
    m_ParamRateOffset = static_cast<u32>( GX2GetPixelUniformVarOffset( m_PixelShader, "rate" ) );
    NW_ASSERT( m_ParamRateOffset != GX2_UNIFORM_VAR_INVALID_OFFSET );
    m_ParamTexLocation = static_cast<u32>( GX2GetPixelSamplerVarLocation( m_PixelShader, "texture0" ) );
    NW_ASSERT( m_ParamTexLocation != static_cast<u32>(-1) );

    m_AttrVertexLocation = static_cast<u32>(GX2GetVertexAttribVarLocation( m_VertexShader, "Vertex" ));
    NW_ASSERT( m_AttrVertexLocation != static_cast<u32>(-1) );
    m_AttrTexCoord0Location = static_cast<u32>( GX2GetVertexAttribVarLocation( m_VertexShader, "TexCoord0" ) );
    NW_ASSERT( m_AttrTexCoord0Location != static_cast<u32>(-1) );
    m_AttrColorRateLocation = static_cast<u32>(GX2GetVertexAttribVarLocation( m_VertexShader, "ColorRate" ));
    NW_ASSERT( m_AttrColorRateLocation != static_cast<u32>(-1) );

    // Vertex Attribute の設定
    // Vertex
    GX2InitAttribStream( &m_Attributes[0], m_AttrVertexLocation, 0, 0, GX2_ATTRIB_FORMAT_32_32_32_FLOAT );
    // TexCoord0
    GX2InitAttribStream( &m_Attributes[1], m_AttrTexCoord0Location, 0, sizeof( float ) * 3, GX2_ATTRIB_FORMAT_32_32_FLOAT );
    // ColorRate
    GX2InitAttribStream( &m_Attributes[2], m_AttrColorRateLocation, 0, sizeof( float ) * 5, GX2_ATTRIB_FORMAT_32_32_32_32_FLOAT );

    // FetchShader のセットアップ
    m_FetchShaderBufPtr = allocator->Alloc( GX2CalcFetchShaderSize( ATTRIBUTE_NUM ), GX2_SHADER_ALIGNMENT );
    GX2InitFetchShader( &m_FetchShader, m_FetchShaderBufPtr, ATTRIBUTE_NUM, m_Attributes );
    GX2Invalidate( GX2_INVALIDATE_CPU_SHADER, m_FetchShaderBufPtr, GX2CalcFetchShaderSize( ATTRIBUTE_NUM ) );

    // バッファ生成
    {
        // Triangle
        static const u16 idx[ 3 ] = { 0, 2, 1 };
        m_TriangleIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( idx ), GX2_INDEX_BUFFER_ALIGNMENT ) );
        nw::ut::MemCpy( m_TriangleIndexBuf, idx, sizeof( idx ) );

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_TriangleIndexBuf, sizeof( u16 ) * 3 );
    }

    {
        // Quad
        m_QuadVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * 4, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_QuadIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * 6, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetQuadVertex( m_QuadVertexBuf, m_QuadIndexBuf );

        // GL と GX2 では uv の原点が異なるので、テクスチャ座標を修正する。今のところテクスチャを貼ることができるのは Quad だけなので、
        // この処理は Quad でのみ行えばよい。
        m_QuadVertexBuf[0].uv.Set( 0.f, 0.f );
        m_QuadVertexBuf[1].uv.Set( 1.f, 0.f );
        m_QuadVertexBuf[2].uv.Set( 0.f, 1.f );
        m_QuadVertexBuf[3].uv.Set( 1.f, 1.f );

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_QuadVertexBuf, sizeof( Vertex ) * 4 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_QuadIndexBuf, sizeof( u16 ) * 6 );
    }

    {
        // Box
        static const u16 idx[ 4 ] = { 0, 1, 3, 2 };
        m_BoxIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( idx ), GX2_INDEX_BUFFER_ALIGNMENT ) );
        nw::ut::MemCpy( m_BoxIndexBuf, idx, sizeof( idx ) );

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_BoxIndexBuf, sizeof( u16 ) * 4 );
    }

    {
        // Line
        m_LineVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * 4, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_LineIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * 6, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetLineVertex( m_LineVertexBuf, m_LineIndexBuf );

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_LineVertexBuf, sizeof( Vertex ) * 4 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_LineIndexBuf, sizeof( u16 ) * 6 );
    }

    {
        // Cube
        m_CubeVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * 8, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_CubeIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * 3 * 2 * 6, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetCubeVertex( m_CubeVertexBuf, m_CubeIndexBuf );

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_CubeVertexBuf, sizeof(Vertex) * 8 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_CubeIndexBuf, sizeof(u16) * 3 * 2 * 6 );
    }

    {
        // WireCube
        m_WireCubeVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * 8, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_WireCubeIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * 17, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetWireCubeVertex( m_WireCubeVertexBuf, m_WireCubeIndexBuf );

        GX2Invalidate(GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_WireCubeVertexBuf, sizeof( Vertex ) * 8 );
        GX2Invalidate(GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_WireCubeIndexBuf, sizeof( u16 ) * 17 );
    }

    {
        // SphereS
        s32 vertexNum = internal::CalcSphereVertexNum( MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 );
        s32 indexNum = internal::CalcSphereIndexNum( MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 );

        m_SphereSVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * vertexNum, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_SphereSIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * indexNum, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetSphereVertex( m_SphereSVertexBuf, m_SphereSIndexBuf, MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 );

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_SphereSVertexBuf, sizeof( Vertex ) * vertexNum );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_SphereSIndexBuf, sizeof( u16 ) * indexNum );
    }

    {
        // SphereL
        s32 vertexNum = internal::CalcSphereVertexNum( MAX_SPHERE_X, MAX_SPHERE_Y );
        s32 indexNum = internal::CalcSphereIndexNum( MAX_SPHERE_X, MAX_SPHERE_Y );

        m_SphereLVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * vertexNum, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_SphereLIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * indexNum, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetSphereVertex( m_SphereLVertexBuf, m_SphereLIndexBuf, MAX_SPHERE_X, MAX_SPHERE_Y );

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_SphereLVertexBuf, sizeof( Vertex ) * vertexNum );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_SphereLIndexBuf, sizeof( u16 ) * indexNum );
    }

    {
        // CircleS
        s32 vertexNum = internal::CalcDiskVertexNum( MAX_DISK_DIVISION / 2 );
        m_DiskSVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * vertexNum, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_CircleSIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * MAX_DISK_DIVISION / 2, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetDiskVertex( m_DiskSVertexBuf, NULL, MAX_DISK_DIVISION / 2 );

        for ( s32 i = 0; i < MAX_DISK_DIVISION / 2; ++i )
        {
            m_CircleSIndexBuf[ i ] = i;
        }

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_DiskSVertexBuf, sizeof( Vertex ) * vertexNum );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_CircleSIndexBuf, sizeof( u16 ) * MAX_DISK_DIVISION / 2 );
    }

    {
        // CircleL
        s32 vertexNum = internal::CalcDiskVertexNum( MAX_DISK_DIVISION );
        m_DiskLVertexBuf = reinterpret_cast<Vertex*>( allocator->Alloc( sizeof( Vertex ) * vertexNum, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_CircleLIndexBuf = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * MAX_DISK_DIVISION, GX2_INDEX_BUFFER_ALIGNMENT ) );

        internal::SetDiskVertex( m_DiskLVertexBuf, NULL, MAX_DISK_DIVISION );

        for ( s32 i = 0; i < MAX_DISK_DIVISION; ++i )
        {
            m_CircleLIndexBuf[ i ] = i;
        }

        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_DiskLVertexBuf, sizeof( Vertex ) * vertexNum );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_CircleLIndexBuf, sizeof( u16 ) * MAX_DISK_DIVISION );
    }

    // サンプラーオブジェクトのセットアップ
    GX2InitSampler( &m_DrawQuadSampler, GX2_TEX_CLAMP_CLAMP, GX2_TEX_XY_FILTER_BILINEAR );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::InitializeImpl( nw::ut::IAllocator* allocator, const char* shaderPath )
{
    u32 size = 0;
    void* binary = NULL;

    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path      = shaderPath;
        loadArg.allocator = allocator;
        loadArg.alignment = PPC_IO_BUFFER_ALIGN;

        binary = fileSystem->Load( loadArg );
        NW_ASSERT( loadArg.readSize > 0 );
    }

    InitializeFromBinaryImpl( allocator, binary, size );

    allocator->Free( binary );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::SetViewportImpl( const s32 x, const s32 y, const s32 width, const s32 height )
{
    GX2SetViewport( x, -y, width, height, 0.f, 1.f );
    GX2SetScissor( static_cast<u32>( x ), static_cast<u32>( y ), static_cast<u32>( width ), static_cast<u32>( height ) );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::BeginImpl()
{
    // WVP 行列生成
    nw::math::MTX44 wvp;
    const nw::math::MTX44 viewMtx( *m_ViewMtx );
    nw::math::MTX44Mult( &wvp, m_ProjMtx, &viewMtx );

    GX2SetShaders( &m_FetchShader, m_VertexShader, m_PixelShader );

    // 初期値設定
    GX2SetVertexUniformReg( m_ParamWVPOffset, 4 * 4, wvp );
    GX2SetVertexUniformReg( m_ParamUserOffset, 4 * 4, nw::math::MTX44::Identity() );

    // 線の幅を設定
    // TODO: cafe 指定した座標と線が出る位置に、他のプラットフォームと違いがあるため、それを吸収するために一時的に
    //       線幅を2にしておく。将来的には描画側で吸収する。
    GX2SetLineWidth( 2.0f );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::EndImpl()
{
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawTriangleImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color
)
{
    DrawTriangles( modelMtx, color, color, m_QuadVertexBuf, 4, m_TriangleIndexBuf, 3 );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawTriangleImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color,
    const DrawBuffer& buf
)
{
    DrawTriangles( modelMtx, color, color, buf.vertex, buf.vertexNum, buf.index, buf.indexNum );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawQuadImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight
)
{
    GX2SetVertexUniformReg( m_ParamUVSrcOffset, 4, nw::math::VEC4::Zero() );
    GX2SetVertexUniformReg( m_ParamUVSizeOffset, 4, nw::math::VEC4( 1.f, 1.f, 1.f, 1.f ) );

    DrawTriangles( modelMtx, colorLeft, colorRight, m_QuadVertexBuf, 4, m_QuadIndexBuf, 6 );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawQuadImpl(
    const nw::math::MTX34& modelMtx,
    const nw::gfnd::Texture& texture,
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight,
    const nw::math::VEC2& uvSrc,
    const nw::math::VEC2& uvSize
)
{
    const nw::gfnd::TextureCafeGX2* texureCafeGx2 = nw::ut::DynamicCast<const nw::gfnd::TextureCafeGX2*>( &texture );
    NW_ASSERT_NOT_NULL( texureCafeGx2 );

    GX2SetVertexUniformReg( m_ParamUVSrcOffset, 4, nw::math::VEC4( uvSrc.x, uvSrc.y, 0.f, 0.f ) );
    GX2SetVertexUniformReg( m_ParamUVSizeOffset, 4, nw::math::VEC4( uvSize.x, uvSize.y, 0.f, 0.f ) );

    DrawTriangles( modelMtx, colorLeft, colorRight, m_QuadVertexBuf, 4, m_QuadIndexBuf, 6, texureCafeGx2->GetGX2Texture() );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawBoxImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight
)
{
    DrawLines( modelMtx, colorLeft, colorRight, GX2_PRIMITIVE_LINE_LOOP, m_QuadVertexBuf, 4, m_BoxIndexBuf, 4 );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawCubeImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1
)
{
    DrawTriangles( modelMtx, color0, color1, m_CubeVertexBuf, 8, m_CubeIndexBuf, 3 * 2 * 6 );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawWireCubeImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1
)
{
    DrawLines( modelMtx, color0, color1, GX2_PRIMITIVE_LINE_LOOP, m_WireCubeVertexBuf, 8, m_WireCubeIndexBuf, 17 );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawLineImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1,
    const f32 lineWidth /* = 1.f */
)
{
    DrawLines( modelMtx, color0, color1, GX2_PRIMITIVE_LINES, m_LineVertexBuf, 2, m_LineIndexBuf, 2, lineWidth );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawLineImpl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color,
    const DrawBuffer& buf,
    const f32 lineWidth /* = 1.f */
)
{
    DrawLines( modelMtx, color, color, GX2_PRIMITIVE_LINES, buf.vertex, buf.vertexNum, buf.index, buf.indexNum, lineWidth );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawSphere4x8Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    DrawTriangles(
        modelMtx,
        north,
        south,
        m_SphereSVertexBuf,
        internal::CalcSphereVertexNum( MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 ),
        m_SphereSIndexBuf,
        internal::CalcSphereIndexNum( MAX_SPHERE_X / 2, MAX_SPHERE_Y / 2 )
    );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawSphere8x16Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    DrawTriangles(
        modelMtx,
        north,
        south,
        m_SphereLVertexBuf,
        internal::CalcSphereVertexNum( MAX_SPHERE_X, MAX_SPHERE_Y ),
        m_SphereLIndexBuf,
        internal::CalcSphereIndexNum( MAX_SPHERE_X, MAX_SPHERE_Y )
    );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawCircle16Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& edge
)
{
    DrawLines(
        modelMtx,
        edge,
        edge,
        GX2_PRIMITIVE_LINE_LOOP,
        m_DiskSVertexBuf,
        internal::CalcDiskVertexNum( MAX_DISK_DIVISION / 2 ),
        m_CircleSIndexBuf,
        MAX_DISK_DIVISION / 2
    );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawCircle32Impl(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& edge
)
{
    DrawLines(
        modelMtx,
        edge,
        edge,
        GX2_PRIMITIVE_LINE_LOOP,
        m_DiskLVertexBuf,
        internal::CalcDiskVertexNum( MAX_DISK_DIVISION ),
        m_CircleLIndexBuf,
        MAX_DISK_DIVISION
    );
}



//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawTriangles(
     const nw::math::MTX34& modelMtx,
     const nw::ut::Color4u8& color0,
     const nw::ut::Color4u8& color1,
     Vertex* vertex,
     u32 vertexNum,
     u16* index,
     u32 indexNum,
     const GX2Texture* texture /* = NULL */
)
{
    GX2SetVertexUniformReg( m_ParamUserOffset, 3 * 4, modelMtx );
    GX2SetVertexUniformReg( m_ParamColor0Offset, 4, static_cast<nw::ut::FloatColor>( color0 ) );
    GX2SetVertexUniformReg( m_ParamColor1Offset, 4, static_cast<nw::ut::FloatColor>( color1 ) );

    // 設定するユニフォームが float 型でも、GX2SetPixelUniformReg の仕様として vec4 を与える必要がある。
    if (texture)
    {
        GX2SetPixelUniformReg( m_ParamRateOffset, 4, nw::math::VEC4( 1.f, 0.f, 0.f, 0.f ) );
        GX2SetPixelTexture( texture, m_ParamTexLocation);
        GX2SetPixelSampler( &m_DrawQuadSampler, m_ParamTexLocation);
    }
    else
    {
        GX2SetPixelUniformReg( m_ParamRateOffset, 4, nw::math::VEC4::Zero() );
    }

    GX2SetAttribBuffer(
        0,
        sizeof( Vertex ) * vertexNum,
        sizeof( Vertex ),
        vertex
    );

    GX2DrawIndexed(
        GX2_PRIMITIVE_TRIANGLES,
        indexNum,
        GX2_INDEX_FORMAT_U16,
        index
    );
}

//--------------------------------------------------------------------------
void
PrimitiveRendererCafe::DrawLines(
    const nw::math::MTX34& modelMtx,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1,
    GX2PrimitiveType lineType,
    Vertex* vertex,
    u32 vertexNum,
    u16* index,
    u32 indexNum,
    f32 lineWidth /* = 1.f */
)
{
    GX2SetLineWidth( lineWidth );

    GX2SetVertexUniformReg( m_ParamUserOffset, 3 * 4, modelMtx );
    GX2SetVertexUniformReg( m_ParamColor0Offset, 4, static_cast<nw::ut::FloatColor>( color0 ) );
    GX2SetVertexUniformReg( m_ParamColor1Offset, 4, static_cast<nw::ut::FloatColor>( color1 ) );

    // 本来は float 一つに 0 を入れればよいが、GX2SetPixelUniformReg の仕様として vec4 を与える必要がある。
    GX2SetPixelUniformReg( m_ParamRateOffset, 4, nw::math::VEC4::Zero() );

    GX2SetAttribBuffer(
        0,
        sizeof( Vertex ) * vertexNum,
        sizeof( Vertex ),
        vertex
    );

    GX2DrawIndexed(
        lineType,
        indexNum,
        GX2_INDEX_FORMAT_U16,
        index
    );
}

//--------------------------------------------------------------------------
bool
PrimitiveRendererCafe::ReadVertexShader(
    nw::ut::IAllocator* allocator,
    GX2VertexShader**   ppShader,
    u32                 index,
    const void*         pData
)
{
    NW_ASSERT_NOT_NULL( pData );
    NW_ASSERT_NOT_NULL( ppShader );

    // インデックスの範囲チェック。
    if ( index >= GFDGetVertexShaderCount( pData ) )
    {
        return false;
    }

    // メモリサイズを計算。
    u32 headerSize  = GFDGetVertexShaderHeaderSize( index, pData );
    u32 programSize = GFDGetVertexShaderProgramSize( index, pData );

    if ( ! headerSize || ! programSize )
    {
        return false;
    }

    // シェーダヘッダのアロケート
    GX2VertexShader* pHeader = (GX2VertexShader*)( allocator->Alloc(headerSize, PPC_IO_BUFFER_ALIGN) );

    // シェーダプログラムのアロケート( 256 アライメント )
    m_VertexProgram = allocator->Alloc( programSize, GX2_SHADER_ALIGNMENT );
    GX2NotifyMemAlloc( m_VertexProgram, programSize, GX2_SHADER_ALIGNMENT );

    // シェーダファイルから、ヘッダとプログラムを取得。
    u32 ret = GFDGetVertexShader( pHeader, m_VertexProgram, index, pData );

    if ( ret )
    {
        // GPU からアクセスするデータのキャッシュフラッシュ
        GX2Invalidate( GX2_INVALIDATE_CPU_SHADER,
                       pHeader->shaderPtr,
                       pHeader->shaderSize );

        *ppShader = pHeader;
    }
    else
    {
        NW_WARNING(false, "Invalid Vertex Shader :%d", ret);

        // エラーの場合のメモリ解放
        if ( pHeader )
        {
            allocator->Free( pHeader );
        }

        if ( m_VertexProgram )
        {
            GX2NotifyMemFree( m_VertexProgram );
            nw::ut::SafeFree( m_VertexProgram, allocator );
        }
    }

    return ret != FALSE;
}

//--------------------------------------------------------------------------
bool
PrimitiveRendererCafe::ReadPixelShader(
    nw::ut::IAllocator* allocator,
    GX2PixelShader**    ppShader,
    u32                 index,
    const void*         pData
)
{
    NW_ASSERT_NOT_NULL( pData );
    NW_ASSERT_NOT_NULL( ppShader );

    // インデックスの範囲チェック。
    if ( index >= GFDGetPixelShaderCount(pData) )
    {
        return false;
    }

    // メモリサイズを計算。
    u32 headerSize = GFDGetPixelShaderHeaderSize( index, pData );
    u32 programSize = GFDGetPixelShaderProgramSize( index, pData );

    if ( ! headerSize || ! programSize )
    {
        return false;
    }

    // シェーダヘッダのアロケート
    GX2PixelShader* pHeader = (GX2PixelShader*)allocator->Alloc( headerSize, PPC_IO_BUFFER_ALIGN );

    // シェーダプログラムのアロケート( 256 アライメント )
    m_PixelProgram = allocator->Alloc( programSize, GX2_SHADER_ALIGNMENT );
    GX2NotifyMemAlloc( m_PixelProgram, programSize, GX2_SHADER_ALIGNMENT );

    // シェーダファイルから、ヘッダとプログラムを取得。
    u32 ret = GFDGetPixelShader( pHeader, m_PixelProgram, index, pData );

    if ( ret )
    {
        // GPU からアクセスするデータのキャッシュフラッシュ
        GX2Invalidate( GX2_INVALIDATE_CPU_SHADER,
                       pHeader->shaderPtr,
                       pHeader->shaderSize);

        *ppShader = pHeader;
    }
    else
    {
        NW_WARNING(false, "Invalid Pixel Shader :%d", ret);

        // エラーの場合のメモリ解放
        if ( pHeader )
        {
            allocator->Free( pHeader );
        }

        if ( m_PixelProgram )
        {
            GX2NotifyMemFree( m_PixelProgram );
            nw::ut::SafeFree( m_PixelProgram, allocator );
        }
    }

    return ret != FALSE;
}


} // namespace dev
} // namespace nw
