﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/sndspy_SpySession.h>

#include <nw/snd/spy/fnd/basis/sndspyfnd_Memory.h>
#include <nw/snd/spy/fnd/hio/sndspyfnd_HioChannel.h>
#include <nw/snd/spy/fnd/string/sndspyfnd_String.h>
#include <nw/snd/spy/protocol/sndspy_Packet.h>
#include <nw/snd/spy/sndspy_SpyData.h>

#if defined(NW_DEBUG)
#define STATE_DEBUG_ENABLED
#define COM_DEBUG_ENABLED
#endif

namespace nw {
namespace snd {
namespace spy {
namespace internal {

//----------------------------------------------------------
SpySession::SpySession() :
m_State(STATE_NOT_INITIALIZED),
m_StateChangedCallback(NULL),
m_StateChangedCallbackParam(NULL),
m_SyncChannelBuffer(NULL),
m_SyncChannelBufferLength(0)
{
}

//----------------------------------------------------------
void
SpySession::Initialize(void* buffer, u32 bufferLength)
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    if(m_State >= STATE_INITIALIZED)
    {
        return;
    }

    // バッファサイズチェック
    NW_ASSERT(bufferLength >= GetRequiredMemorySize());

    m_SyncChannelBuffer = nw::snd::spy::internal::fnd::RoundUp(buffer, nw::snd::spy::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
    m_SyncChannelBufferLength = bufferLength - nw::snd::spy::internal::fnd::GetOffsetFromPtr(buffer, m_SyncChannelBuffer);

    SetState(STATE_INITIALIZED);
}

//----------------------------------------------------------
void
SpySession::Finalize()
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    if(m_State <= STATE_NOT_INITIALIZED)
    {
        return;
    }

    Close();

    m_SyncChannelBuffer = NULL;
    m_SyncChannelBufferLength = 0;

    SetState(STATE_NOT_INITIALIZED);
}

//----------------------------------------------------------
size_t
SpySession::GetRequiredMemorySize()
{
    u32 maxPacketLength = sizeof(InitializePacket);
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(InitializeReplyPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(FinalizePacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(FinalizeReplyPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(PingPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(PongPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(SetOutputDirPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(SetOutputDirReplyPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(NotifyDataReadPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(NotifyDataReadReplyPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(QueryDataInfoPacket)));
    maxPacketLength = nw::snd::spy::internal::fnd::Max(maxPacketLength, static_cast<u32>(sizeof(QueryDataInfoReplyPacket)));

    return
        nw::snd::spy::internal::fnd::RoundUp(maxPacketLength, nw::snd::spy::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
}

//----------------------------------------------------------
bool
SpySession::Open(PortType syncPort, PortType dataPort)
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    NW_ASSERTMSG(IsInitialized(), "SpySession is not initialized.\n");

    if(m_State >= STATE_OPENING)
    {
        return true;
    }

    SetState(STATE_OPENING);

    if(!OpenSyncChannel(syncPort))
    {
        Close();
        return false;
    }

    if(!m_DataChannel.Open(dataPort))
    {
        Close();
        return false;
    }

    SetState(STATE_OPENED);

    return true;
}

//----------------------------------------------------------
void
SpySession::Close()
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    NW_ASSERTMSG(IsInitialized(), "SpySession is not initialized.\n");

    if(m_State < STATE_OPENING)
    {
        return;
    }

    Disconnect();

    m_DataChannel.Close();
    CloseSyncChannel();

    SetState(STATE_INITIALIZED);
}

//----------------------------------------------------------
bool
SpySession::Connect()
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    if(IsConnected())
    {
        return true;
    }

    if(m_State < STATE_OPENED)
    {
        return false;
    }

    SetState(STATE_CONNECTING);

    m_StateLock.Leave();

    if(!m_SyncChannel.Connect())
    {
        m_StateLock.Enter();

        Disconnect();
        return false;
    }

    if(!m_DataChannel.Connect())
    {
        m_StateLock.Enter();

        Disconnect();
        return false;
    }

    m_StateLock.Enter();

    // Sync, Data チャンネルの Connect() 中に別スレッドでセッションが Close() 等された場合。
    if(m_State != STATE_CONNECTING)
    {
        return false;
    }

    SetState(STATE_PREPARING);

    return true;
}

//----------------------------------------------------------
void
SpySession::Disconnect()
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    if(m_State < STATE_CONNECTING)
    {
        return;
    }

    SetState(STATE_DISCONNECTING);

    m_DataChannel.Disconnect();
    m_SyncChannel.Disconnect();

    SetState(STATE_OPENED);
}

//----------------------------------------------------------
bool
SpySession::ProcessSyncPacket(IMessageHandler* msgHandler)
{
    if(!IsConnected())
    {
        return false;
    }

    if(!m_SyncChannel.IsConnected() || !m_DataChannel.IsConnected())
    {
        Close();
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();

    if(packetHeader == NULL)
    {
        // 次のパケットヘッダを読む。
        NW_ASSERT(m_SyncPacketReader.CanReadHeader());
        m_SyncPacketReader.ReadPacketHeader();

        packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();

        if(packetHeader == NULL)
        {
            return false;
        }
    }

    bool completeProcess = false;

    if(IsPreparing())
    {
        // 準備中は、初期化パケットしか受け付けない。
        // それ以外のパケットはスキップする。
        switch (static_cast<u32>(packetHeader->id))
        {
        case PacketID::ID_INITIALIZE:
            completeProcess = OnInitialize(msgHandler);
            break;

        default:
            completeProcess = m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize());
            break;
        }
    }
    else
    {
        switch (static_cast<u32>(packetHeader->id))
        {
        // IsPrepared() 状態で処理しないパケットだが、ツールの状態次第で受信することがあるので、スルーしておく。
        case PacketID::ID_INITIALIZE:
            completeProcess = m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize());
            break;

        case PacketID::ID_PING:
            completeProcess = OnPing();
            break;

        case PacketID::ID_SELECT_DATA_ID:
            completeProcess = OnSelectDataID(msgHandler);
            break;

        case PacketID::ID_SET_OUTPUT_DIR:
            completeProcess = OnSetOutputDir(msgHandler);
            break;

        case PacketID::ID_NOTIFY_DATA_READ:
            completeProcess = OnNotifyDataRead(msgHandler);
            break;

        case PacketID::ID_FINALIZE:
            completeProcess = OnFinalize(msgHandler);
            break;

        case PacketID::ID_QUERY_DATA_INFO:
            completeProcess = OnQueryDataInfo(msgHandler);
            break;

        default:
            NW_FATAL_ERROR("unknown packet '%08x'\n", u32(packetHeader->id));
            break;
        }
    }

    if(!completeProcess)
    {
        return true;
    }

    // パケット処理が完了した場合は、GetCurrentPacketHeader() で NULL が返るように Next() する
    if(m_SyncPacketReader.GetState() == SpyPacketReader::STATE_READ_BODY)
    {
        m_SyncPacketReader.Next();
    }

    return true;
}

//----------------------------------------------------------
u32
SpySession::SendData(const void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);

    if(!m_DataChannel.IsConnected())
    {
        return 0;
    }

    return m_DataChannel.Write(buffer, length);
}

//----------------------------------------------------------
void
SpySession::SetState(State value)
{
    if(m_State == value)
    {
        return;
    }

#ifdef STATE_DEBUG_ENABLED
    NW_LOG("[SpySession] State = %s --> %s\n", StateToString(m_State), StateToString(value));
#endif

    m_State = value;

    if(m_StateChangedCallback != NULL)
    {
        m_StateChangedCallback(m_StateChangedCallbackParam);
    }
}

//----------------------------------------------------------
bool
SpySession::OpenSyncChannel(PortType port)
{
    if(!m_SyncChannel.Open(port))
    {
        return false;
    }

    m_SyncPacketReader.Initialize(&m_SyncChannel, m_SyncChannelBuffer);

    return true;
}

//----------------------------------------------------------
void
SpySession::CloseSyncChannel()
{
    m_SyncPacketReader.Finalize();
    m_SyncChannel.Close();
}

//----------------------------------------------------------
void*
SpySession::GetPacketBodyBuffer()
{
    if(m_SyncChannelBuffer == NULL) { return NULL; }

    if(m_SyncPacketReader.GetState() < SpyPacketReader::STATE_READ_HEADER)
    {
        return NULL;
    }

    return nw::snd::spy::internal::fnd::AddOffsetToPtr(m_SyncChannelBuffer, sizeof(PacketHeader));
}

//----------------------------------------------------------
bool
SpySession::OnInitialize(IMessageHandler* msgHandler)
{
    // 通信準備中でない場合は、処理しない
    if (!IsPreparing())
    {
        return true;
    }

    NW_ASSERT_NOT_NULL(msgHandler);
    NW_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::STATE_READ_HEADER);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NW_ASSERT_NOT_NULL(packetHeader);
    NW_ASSERT(packetHeader->bodyLength >= sizeof(InitializePacket::Body));

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpySession] OnInitialize()\n");
#endif

    if(msgHandler != NULL)
    {
        msgHandler->OnInitializeSession();
    }

    InitializeReplyPacket* replyPacket = new(m_SyncChannelBuffer) InitializeReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(InitializeReplyPacket));

    SetState(STATE_PREPARED);

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnSetOutputDir(IMessageHandler* msgHandler)
{
    // 通信準備ができていない場合は、処理しない
    if (!IsPrepared())
    {
        return true;
    }

    NW_ASSERT_NOT_NULL(msgHandler);
    NW_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::STATE_READ_HEADER);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NW_ASSERT_NOT_NULL(packetHeader);
    NW_ASSERT(packetHeader->bodyLength >= offsetof(SetOutputDirPacket::Body, path));

    SetOutputDirPacket::Body* body = reinterpret_cast<SetOutputDirPacket::Body*>(GetPacketBodyBuffer());

    // パス文字列がパケットに収まっているか。
    if (body != NULL && !(body->length > 0 &&
        offsetof(SetOutputDirPacket::Body, length) + body->length <= packetHeader->bodyLength))
    {
        NW_FATAL_ERROR("invalid SetOutputDirPacket (%d).", body->length);
        body = NULL;
    }

    // パス文字列はヌル終端されているか。
    if (body != NULL && !(body->path[body->length - 1] == '\0'))
    {
        NW_FATAL_ERROR("SetOutputDirPacket::path string is not null-terminated.");
        body = NULL;
    }

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpySession] OnSetOutputDir(path = '%s').\n", body? body->path : "<invalid>");
#endif

    if(msgHandler != NULL)
    {
        msgHandler->OnSetOutputDirPath(body? body->path : NULL);
    }

    SetOutputDirReplyPacket* replyPacket = new(m_SyncChannelBuffer) SetOutputDirReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(SetOutputDirReplyPacket));

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnSelectDataID(IMessageHandler* msgHandler)
{
    // 通信準備ができていない場合は、処理しない
    if (!IsPrepared())
    {
        return true;
    }

    NW_ASSERT_NOT_NULL(msgHandler);
    NW_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::STATE_READ_HEADER);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NW_ASSERT_NOT_NULL(packetHeader);
    NW_ASSERT(packetHeader->bodyLength >= offsetof(SelectDataIDPacket::Body, selectionFlags));

    SelectDataIDPacket::Body* body = reinterpret_cast<SelectDataIDPacket::Body*>(GetPacketBodyBuffer());
    NW_ASSERT(packetHeader->bodyLength == offsetof(SelectDataIDPacket::Body, selectionFlags) + sizeof(u32) * body->selectionFlagsLength);

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpySession] OnSelectDataID() : selectionFlagsLength = %d.\n", body->selectionFlagsLength);
    for (u32 i = 0; i < body->selectionFlagsLength; ++i)
    {
        NW_LOG("  flags[%d] = %08x\n", i, body->selectionFlags[i]);
    }
#endif

    if(msgHandler != NULL)
    {
        msgHandler->OnSelectDataID(body->selectionFlagsLength, body->selectionFlags);
    }

    SelectDataIDReplyPacket* replyPacket = new(m_SyncChannelBuffer) SelectDataIDReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(SelectDataIDReplyPacket));

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnNotifyDataRead(IMessageHandler* msgHandler)
{
    // 通信準備ができていない場合は、処理しない
    if (!IsPrepared())
    {
        return true;
    }

    NW_ASSERT_NOT_NULL(msgHandler);
    NW_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::STATE_READ_HEADER);

    if(!m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NW_ASSERT_NOT_NULL(packetHeader);
    NW_ASSERT(packetHeader->bodyLength >= sizeof(NotifyDataReadPacket::Body));

    NotifyDataReadPacket::Body* body = reinterpret_cast<NotifyDataReadPacket::Body*>(GetPacketBodyBuffer());

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpySession] OnNotifyDataRead() : fileNo=%d.\n", body->fileNo);
#endif

    if(msgHandler != NULL)
    {
        msgHandler->OnDataRead(body->fileNo);
    }

    NotifyDataReadReplyPacket* replyPacket = new(m_SyncChannelBuffer) NotifyDataReadReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(NotifyDataReadReplyPacket));

    return true;
}


//----------------------------------------------------------
bool
SpySession::OnFinalize(IMessageHandler* msgHandler)
{
    // 通信準備ができていない場合は、処理しない
    if (!IsPrepared())
    {
        return true;
    }

    NW_ASSERT_NOT_NULL(msgHandler);
    NW_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::STATE_READ_HEADER);

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpySession] OnFinalize().\n");
#endif

    if(msgHandler != NULL)
    {
        msgHandler->OnFinalizeSession();
    }

    FinalizeReplyPacket* replyPacket = new(m_SyncChannelBuffer) FinalizeReplyPacket();
    m_SyncChannel.Write(replyPacket, sizeof(FinalizeReplyPacket));

    // 一度 Close() しないと再接続できないプラットフォームがあるため、
    // Disconnect() ではなく Close() する。
    Close();

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnPing()
{
    // 通信準備ができていない場合は、処理しない
    if (!IsPrepared())
    {
        return true;
    }

    NW_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::STATE_READ_HEADER);

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpySession] OnPing().\n");
#endif

    PongPacket* replyPacket = new(m_SyncChannelBuffer) PongPacket();

    if(m_SyncChannel.Write(replyPacket, sizeof(PongPacket)) != sizeof(PongPacket))
    {
        Disconnect();
    }

    return true;
}

//----------------------------------------------------------
bool
SpySession::OnQueryDataInfo(IMessageHandler* msgHandler)
{
    // 通信準備ができていない場合は、処理しない
    if (!IsPrepared())
    {
        return true;
    }

    NW_ASSERT_NOT_NULL(msgHandler);
    NW_ASSERT(m_SyncPacketReader.GetState() >= SpyPacketReader::STATE_READ_HEADER);

    if (m_SyncPacketReader.GetState() == SpyPacketReader::STATE_READ_HEADER &&
        !m_SyncPacketReader.ReadPacketBody(GetPacketBodyBuffer(), GetPacketBodyBufferSize()))
    {
        return false;
    }

    const PacketHeader* packetHeader = m_SyncPacketReader.GetCurrentPacketHeader();
    NW_ASSERT_NOT_NULL(packetHeader);

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpySession] OnQueryDataInfo()\n");
#endif

    QueryDataInfoReplyPacket* replyPacket = new(m_SyncChannelBuffer) QueryDataInfoReplyPacket();

    const SpyDataInfo* dataInfo = NULL;

    if(msgHandler != NULL)
    {
        // NULLが返った場合はデータタイプの終了を表します。
        dataInfo = msgHandler->OnQueryDataInfo();
    }

    if (dataInfo != NULL)
    {
        static const char PADDING[4] = { '\0', '\0', '\0', '\0' };

        const char* dataName = dataInfo->GetDataName();
        size_t dataNameLength = strlen(dataName);
        size_t dataNameLengthPadded = nw::snd::spy::internal::fnd::RoundUp(dataNameLength, 4);

        replyPacket->body.dataVersion = dataInfo->GetDataVersion();
        replyPacket->body.dataId = static_cast<u16>(dataInfo->GetDataID());
        replyPacket->body.dataNameLength = static_cast<u16>(dataNameLength);
        replyPacket->SetPacketSize();
        m_SyncChannel.Write(replyPacket, sizeof(QueryDataInfoReplyPacket));
        m_SyncChannel.Write(dataName, dataNameLength);
        if (dataNameLength != dataNameLengthPadded)
        {
            m_SyncChannel.Write(PADDING, dataNameLengthPadded - dataNameLength);
        }
    }
    else
    {
        replyPacket->body.dataVersion = 0;
        replyPacket->body.dataId = DATA_ID_INVALID;
        replyPacket->body.dataNameLength = 0;
        m_SyncChannel.Write(replyPacket, sizeof(QueryDataInfoReplyPacket));
    }

    return true;
}

//----------------------------------------------------------
const char*
SpySession::StateToString(State value)
{
#if defined(STATE_DEBUG_ENABLED)
    static const char* strings[] = {
        "STATE_NOT_INITIALIZED",
        "STATE_INITIALIZED",
        "STATE_OPENING",
        "STATE_OPENED",
        "STATE_DISCONNECTING",
        "STATE_CONNECTING",
        "STATE_PREPARING",
        "STATE_PREPARED"
    };

    return strings[value];
#else
    (void)value;
    return NULL;
#endif
}

} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
