﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/fnd/hio/sndspyfnd_HioChannel.h>

#include <WinSock2.h>

namespace nw {
namespace snd {
namespace spy {
namespace internal {
namespace fnd {

//----------------------------------------------------------
HioChannel::HioChannel() :
m_Port(INVALID_PORT),
m_Socket(INVALID_SOCKET),
m_ClientSocket(INVALID_SOCKET)
{
}

//----------------------------------------------------------
bool
HioChannel::Open(const PortType port)
{
    m_Socket = socket(AF_INET, SOCK_STREAM, 0);

    if(m_Socket == INVALID_SOCKET)
    {
        return false;
    }

    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    addr.sin_addr.S_un.S_addr = INADDR_ANY;

    m_Port = port;

    if(bind(m_Socket, reinterpret_cast<const struct sockaddr*>(&addr), sizeof(struct sockaddr_in)) == SOCKET_ERROR)
    {
        Close();
        return false;
    }

    if(listen(m_Socket, 1) == SOCKET_ERROR)
    {
        Close();
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
HioChannel::Close()
{
    if(m_ClientSocket != INVALID_SOCKET)
    {
        closesocket(m_ClientSocket);
        m_ClientSocket = INVALID_SOCKET;
    }

    if(m_Socket != INVALID_SOCKET)
    {
        closesocket(m_Socket);
        m_Socket = INVALID_SOCKET;
    }

    m_Port = INVALID_PORT;
}

//----------------------------------------------------------
bool
HioChannel::Connect()
{
    if(!IsOpened())
    {
        NW_ASSERTMSG(false, "HioChannel is not opened.");
        return false;
    }

    if(IsConnected())
    {
        return true;
    }

    struct sockaddr_in client;
    s32 addrlen = sizeof(client);

    m_ClientSocket = accept(m_Socket, reinterpret_cast<struct sockaddr*>(&client), &addrlen);

    if(m_ClientSocket == INVALID_SOCKET)
    {
        Disconnect();
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
HioChannel::Disconnect()
{
    if(m_ClientSocket != INVALID_SOCKET)
    {
        closesocket(m_ClientSocket);
        m_ClientSocket = INVALID_SOCKET;
    }
}

//----------------------------------------------------------
bool
HioChannel::IsOpened() const
{
    return m_Port != INVALID_PORT;
}

//----------------------------------------------------------
bool
HioChannel::IsConnected() const
{
    return IsOpened() && m_ClientSocket != INVALID_SOCKET;
}

//----------------------------------------------------------
HioChannel::ConstPortType
HioChannel::GetPort() const
{
    return m_Port;
}

//----------------------------------------------------------
u32
HioChannel::Read(void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);

    if(!IsOpened())
    {
        return 0;
    }

    if(!IsConnected())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    char* current = reinterpret_cast<char*>(buffer);
    u32   readLength = 0;

    while(readLength < length)
    {
        int recvLength = recv(m_ClientSocket, current, length - readLength, 0);

        // ブロッキングモードの場合、0 以下を返したら切断されたとみなす
        if(recvLength <= 0)
        {
            Disconnect();
            break;
        }

        current += recvLength;
        readLength += recvLength;
    }

    return readLength;
}

//----------------------------------------------------------
u32
HioChannel::Write(const void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);

    if(!IsOpened())
    {
        return 0;
    }

    if(!IsConnected())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    const char* current = reinterpret_cast<const char*>(buffer);
    u32         writtenLength = 0;

    while(writtenLength < length)
    {
        int sendLength = send(m_ClientSocket, current, length - writtenLength, 0);

        // ブロッキングモードの場合、0 以下を返したら切断されたとみなす
        if(sendLength <= 0)
        {
            Disconnect();
            break;
        }

        current += sendLength;
        writtenLength += sendLength;
    }

    return writtenLength;
}

} // namespace nw::snd::spy::internal::fnd
} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
