﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundArchiveFileEditor.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_SoundArchivePlayer.h>
#include <nw/snd/fnd/io/sndfnd_LegacyStreamAdaptor.h>
#include <nw/snd/edit/sndedit_SoundArchiveEditController.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

#include <nw/snd/fnd/string/sndfnd_String.h>


#if defined(NW_PLATFORM_WIN32)
namespace {
    void ReplaceFilePath(const nw::snd::edit::internal::ResSoundInfo* soundInfo, const char* originalPath, char* replacedPath)
    {
        const nw::snd::edit::internal::ResStreamSoundInfo* streamSoundInfo =
            reinterpret_cast<const nw::snd::edit::internal::ResStreamSoundInfo*>(soundInfo);
        NW_ASSERT_NOT_NULL(streamSoundInfo);
        if (streamSoundInfo->streamSoundParam.streamFileType == nw::snd::SoundArchive::STREAM_FILE_TYPE_ADTS){
            s32 fileNameStartIndex =
                nw::snd::internal::fnd::String::LastIndexOf(
                    originalPath,
                    '\\',
                    nw::snd::internal::FILE_PATH_MAX);

            if (fileNameStartIndex != nw::snd::internal::fnd::String::INDEX_NOT_FOUND)
            {
                fileNameStartIndex++;
                s32 extensionStartIndex =
                    nw::snd::internal::fnd::String::LastIndexOf(
                        &originalPath[fileNameStartIndex],
                        '.',
                        nw::snd::internal::FILE_PATH_MAX - fileNameStartIndex);

                if (extensionStartIndex != nw::snd::internal::fnd::String::INDEX_NOT_FOUND)
                {
                    s32 fileNameLength = extensionStartIndex;

                    char parentDir[nw::snd::internal::FILE_PATH_MAX];
                    nw::ut::strncpy(parentDir, nw::snd::internal::FILE_PATH_MAX, originalPath, fileNameStartIndex - 1);

                    char fileName[nw::snd::internal::FILE_PATH_MAX];
                    nw::ut::strncpy(fileName, nw::snd::internal::FILE_PATH_MAX, &originalPath[fileNameStartIndex], fileNameLength);

                    char resultPath[nw::snd::internal::FILE_PATH_MAX];
                    nw::ut::snprintf(resultPath, nw::snd::internal::FILE_PATH_MAX, "%s\\_forPC\\%s.bfstm", parentDir, fileName);

                    size_t resultPathLength = strnlen(resultPath, nw::snd::internal::FILE_PATH_MAX);

                    nw::ut::strncpy(replacedPath, nw::snd::internal::FILE_PATH_MAX, resultPath, resultPathLength);
                }
            }
        }
        else
        {
            nw::ut::strncpy(replacedPath, nw::snd::internal::FILE_PATH_MAX, originalPath, nw::snd::internal::FILE_PATH_MAX);
        }
    }
}
#endif

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//----------------------------------------------------------
SoundArchiveFileEditor::SoundArchiveFileEditor() :
m_EditController(NULL),
m_SoundArchivePlayer(NULL)
{
    m_SoundArchiveHook.Initialize(*this);
}

//----------------------------------------------------------
Result
SoundArchiveFileEditor::Initialize(
    SoundArchiveEditController* editController,
    SoundArchivePlayer* soundArchivePlayer)
{
    if(editController == NULL ||
        soundArchivePlayer == NULL)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }

    m_EditController = editController;
    m_SoundArchivePlayer = soundArchivePlayer;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Finalize()
{
    m_EditController = NULL;
    m_SoundArchivePlayer = NULL;
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Start()
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveFileEditor is not initialized.");
    m_SoundArchivePlayer->detail_SetSoundArchiveFilesHook(&m_SoundArchiveHook);
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Stop()
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveFileEditor is not initialized.");
    m_SoundArchivePlayer->detail_SetSoundArchiveFilesHook(NULL);
}

//----------------------------------------------------------
bool
SoundArchiveFileEditor::Hook::IsTargetItemImpl(const char* itemLabel)
{
    NW_ASSERT_NOT_NULL(itemLabel);
    NW_ASSERT(*itemLabel != '\0');

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_EditController->GetItemCacheState(itemLabel) == CACHE_STATE_CACHED;
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Hook::LockImpl()
{
    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_EditController->LockCacheStates();
}

//----------------------------------------------------------
void
SoundArchiveFileEditor::Hook::UnlockImpl()
{
    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_EditController->UnlockCacheStates();
}

//----------------------------------------------------------
#ifdef NW_PLATFORM_CTR
io::FileStream*
#else
ut::FileStream*
#endif
SoundArchiveFileEditor::Hook::OpenFileImpl(void* buffer, u32 bufferLength, void* cacheBuffer, u32 cacheBufferLength, const char* itemLabel, const char* fileType)
{
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERT_NOT_NULL(itemLabel);
    NW_ASSERT(*itemLabel != '\0');
    NW_ASSERT_NOT_NULL(fileType);
    NW_ASSERT(StringToResDataType(fileType) == RES_DATA_TYPE_STREAM_SOUND_FILE);

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    ResDataType dataType = RES_DATA_TYPE_UNKNOWN;

    // ファイルパスを参照するためにサウンド情報を取得します。
    const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(
        m_Owner->m_EditController->GetSoundInfo(
        itemLabel,
        &ResItemInfoUtility::IsSoundInfoDataType,
        &dataType));

    if(soundInfo == NULL)
    {
        return NULL;
    }

    // 現状はストリームサウンドのみ。
    if(dataType != RES_DATA_TYPE_STREAM_SOUND)
    {
        NW_WARNING(false, "[sndedit] item '%s' is not StreamSound.\n", itemLabel);
        return NULL;
    }

    const ResName* pcFilePath = ResItemInfoUtility::GetFilePath(
        *soundInfo,
        RES_DATA_TYPE_STREAM_SOUND,
        RES_DATA_TYPE_STREAM_SOUND_FILE);

    if(pcFilePath == NULL ||
        pcFilePath->GetLength() == 0)
    {
        return NULL;
    }

#if defined(NW_PLATFORM_WIN32)
    // PC版の時は、代わりのファイルを鳴らすようにするためにパスを書き換える
    char replacedPath[nw::snd::internal::FILE_PATH_MAX];
    ReplaceFilePath(soundInfo, pcFilePath->GetName(), replacedPath);
#endif

    void* fileStreamBuffer = ut::AddOffsetToPtr(buffer, sizeof(snd::internal::fnd::LegacyFileStreamAdaptor));

    snd::internal::fnd::FileStream* fileStream = m_Owner->m_EditController->OpenFile(
        fileStreamBuffer,
        bufferLength - sizeof(snd::internal::fnd::LegacyFileStreamAdaptor),
#if defined(NW_PLATFORM_WIN32)
        replacedPath
#else
        pcFilePath->GetName()
#endif
        );

    if(fileStream == NULL)
    {
        return NULL;
    }

    if((cacheBuffer != NULL) && (cacheBufferLength > 0))
    {
        fileStream->EnableCache(cacheBuffer, cacheBufferLength);
    }

    return new(buffer) snd::internal::fnd::LegacyFileStreamAdaptor(fileStream);
}

//----------------------------------------------------------
const void*
SoundArchiveFileEditor::Hook::GetFileAddressImpl(
    const char* itemLabel,
    const char* itemType,
    const char* fileType,
    u32 fileIndex/*= 0*/)
{
    if(itemLabel == NULL || *itemLabel == '\0')
    {
        return NULL;
    }

    NW_ASSERT_NOT_NULL(itemType);
    NW_ASSERT_NOT_NULL(fileType);
    NW_ASSERT_NOT_NULL(m_Owner);

    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    ResDataType itemDataType = StringToResDataType(itemType);
    ResDataType fileDataType = StringToResDataType(fileType);

    const ResName* pcFilePath = NULL;

    // SEQ が参照するバンクファイルか波形アーカイブの場合
    if(itemDataType == RES_DATA_TYPE_SEQUENCE_SOUND &&
        fileDataType != RES_DATA_TYPE_SEQUENCE_FILE)
    {
        pcFilePath = GetBankFilePath(itemLabel, fileDataType, fileIndex);
    }
    else
    {
        ResDataType cachedItemDataType = RES_DATA_TYPE_UNKNOWN;

        // ファイルパスを参照するためにサウンド情報とバンク情報を取得します。
        const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(
            m_Owner->m_EditController->GetSoundInfo(
            itemLabel,
            &ResItemInfoUtility::IsSoundInfoDataType,
            &cachedItemDataType)
            );

        if(soundInfo == NULL || itemDataType != cachedItemDataType)
        {
            return NULL;
        }

        pcFilePath = ResItemInfoUtility::GetFilePath(
            *soundInfo,
            static_cast<ResDataType>(cachedItemDataType),
            fileDataType);
    }

    if(pcFilePath == NULL ||
        pcFilePath->GetLength() == 0)
    {
        return NULL;
    }

    return m_Owner->m_EditController->GetFile(pcFilePath->GetName(), fileDataType);
}

//----------------------------------------------------------
const ResName*
SoundArchiveFileEditor::Hook::GetBankFilePath(const char* itemLabel, ResDataType fileDataType, u32 bankIndex)
{
    NW_ASSERT_NOT_NULL(itemLabel);
    NW_ASSERT(*itemLabel != '\0');
    NW_ASSERT_NOT_NULL(m_Owner);

    ResDataType itemInfoDataType = RES_DATA_TYPE_UNKNOWN;

    // ファイルパスを参照するためにサウンド情報とバンク情報を取得します。
    const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(
        m_Owner->m_EditController->GetSoundInfo(
        itemLabel,
        &ResItemInfoUtility::IsSequenceSoundInfoDataType,
        &itemInfoDataType)
        );

    if(bankIndex >= SEQ_BANK_COUNT)
    {
        NW_FATAL_ERROR("[sndedit] invalid bank index.\n");
        return NULL;
    }

    const char* bankName = NULL;

    if(soundInfo != NULL)
    {
        const ResSequenceSoundInfo* sequenceSoundInfo = reinterpret_cast<const ResSequenceSoundInfo*>(soundInfo);
        bankName = sequenceSoundInfo->sequenceSoundParam.bankNameOffsets[bankIndex].to_ptr<ResName>()->GetName();
    }
    else
    {
        const SoundArchive& soundArchive = m_Owner->m_SoundArchivePlayer->GetSoundArchive();

        SoundArchive::SequenceSoundInfo sequenceSoundInfo;
        if(!soundArchive.ReadSequenceSoundInfo(soundArchive.GetItemId(itemLabel), &sequenceSoundInfo))
        {
            return NULL;
        }

        bankName = soundArchive.GetItemLabel(sequenceSoundInfo.bankIds[bankIndex]);
    }

    if(bankName == NULL || *bankName == '\0')
    {
        return NULL;
    }

    const ResBankInfo* bankInfo = m_Owner->m_EditController->GetBankInfo(bankName);

    if(bankInfo == NULL)
    {
        return NULL;
    }

    switch(fileDataType)
    {
    case RES_DATA_TYPE_BANK_FILE:
    case RES_DATA_TYPE_WAVE_ARCHIVE_FILE:
        break;

    default:
        NW_FATAL_ERROR("[sndedit] invalid file data type.\n");
        return NULL;
    }

    return ResItemInfoUtility::GetFilePath(*bankInfo, fileDataType);
}

//----------------------------------------------------------
ResDataType
SoundArchiveFileEditor::Hook::StringToResDataType(const char* value) const
{
    NW_ASSERT_NOT_NULL(value);

    // 終端文字も含めて比較するために strlen() + 1 しています。
    if(std::strncmp(
        value,
        Hook::ITEM_TYPE_STREAM_SOUND,
        std::strlen(Hook::ITEM_TYPE_STREAM_SOUND) + 1) == 0)
    {
        return RES_DATA_TYPE_STREAM_SOUND;
    }

    if(std::strncmp(
        value,
        Hook::ITEM_TYPE_WAVE_SOUND,
        std::strlen(Hook::ITEM_TYPE_WAVE_SOUND) + 1) == 0)
    {
        return RES_DATA_TYPE_WAVE_SOUND;
    }

    if(std::strncmp(
        value,
        Hook::ITEM_TYPE_SEQUENCE_SOUND,
        std::strlen(Hook::ITEM_TYPE_SEQUENCE_SOUND) + 1) == 0)
    {
        return RES_DATA_TYPE_SEQUENCE_SOUND;
    }

    if(std::strncmp(
        value,
        Hook::FILE_TYPE_STREAM_BINARY,
        std::strlen(Hook::FILE_TYPE_STREAM_BINARY) + 1) == 0)
    {
        return RES_DATA_TYPE_STREAM_SOUND_FILE;
    }

    if(std::strncmp(
        value,
        Hook::FILE_TYPE_WAVE_SOUND_BINARY,
        std::strlen(Hook::FILE_TYPE_WAVE_SOUND_BINARY) + 1) == 0)
    {
        return RES_DATA_TYPE_WAVE_SOUND_FILE;
    }

    if(std::strncmp(
        value,
        Hook::FILE_TYPE_SEQUENCE_BINARY,
        std::strlen(Hook::FILE_TYPE_SEQUENCE_BINARY) + 1) == 0)
    {
        return RES_DATA_TYPE_SEQUENCE_FILE;
    }

    if(std::strncmp(
        value,
        Hook::FILE_TYPE_BANK_BINARY,
        std::strlen(Hook::FILE_TYPE_BANK_BINARY) + 1) == 0)
    {
        return RES_DATA_TYPE_BANK_FILE;
    }

    if(std::strncmp(
        value,
        Hook::FILE_TYPE_WAVE_ARCHIVE_BINARY,
        std::strlen(Hook::FILE_TYPE_WAVE_ARCHIVE_BINARY) + 1) == 0)
    {
        return RES_DATA_TYPE_WAVE_ARCHIVE_FILE;
    }

    if(std::strncmp(
        value,
        Hook::FILE_TYPE_STREAM_PREFETCH_BINARY,
        std::strlen(Hook::FILE_TYPE_STREAM_PREFETCH_BINARY) + 1) == 0)
    {
        return RES_DATA_TYPE_STREAM_PREFTCH_SOUND_FILE;
    }

    return RES_DATA_TYPE_UNKNOWN;
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
