﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundArchiveEditController.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_SoundHandle.h>
#include <nw/snd/snd_SoundArchivePlayer.h>
#include <nw/snd/snd_SoundPlayer.h>
#include <nw/snd/fnd/io/sndfnd_FileStream.h>
#include <nw/snd/fnd/string/sndfnd_Path.h>
#include <nw/snd/edit/sndedit_Config.h>
#include <nw/snd/edit/sndedit_IErrorProvider.h>
#include <nw/snd/edit/sndedit_SoundEditConnection.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>
#include <nw/snd/edit/res/sndedit_ResourceManager.h>
#include <nw/snd/edit/protocol/sndedit_QueryItemInfoPacket.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_CACHE_DEBUG
#endif

namespace nw {
namespace snd {
namespace edit {
namespace internal {

namespace {

//----------------------------------------------------------
//! @brief  キャッシュを更新する関数オブジェクトです。
class CacheUpdator
{
    NW_DISALLOW_COPY_AND_ASSIGN(CacheUpdator);

public:
    CacheUpdator(SoundArchiveEditController& editController) :
      m_EditController(editController) { }

public:
    void operator()(const char* name)
    {
        m_EditController.MakeItemInfoEditable(name);
    }

private:
    SoundArchiveEditController& m_EditController;
};

}

//----------------------------------------------------------
Result
SoundArchiveEditController::Initialize(const InitializeArgs& args)
{
    if(args.errorProvider == NULL ||
        args.buffer == NULL ||
        args.maxItemName == 0 ||
        args.resourceManager == NULL ||
        args.soundArchive == NULL ||
        args.soundArchivePlayer == NULL)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }

    if(args.bufferLength < GetRequiredMemorySize(args.maxItemName))
    {
        NW_FATAL_ERROR("out of memory.\n");
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

#if defined(NW_PLATFORM_CTR)
    m_CacheLockObject.Initialize();
    m_UnfreezeCacheLockObject.Initialize();
    m_ErrorItemLockObject.Initialize();
#endif

    snd::internal::fnd::FrameHeap frameHeap;
    frameHeap.Initialize(args.buffer, args.bufferLength);

    m_OpenFileBufferLength = SoundEditConnection::GetRequiredMemorySizeForOpenFile();
    m_OpenFileBuffer = frameHeap.Alloc(m_OpenFileBufferLength);

    if(m_OpenFileBuffer == NULL)
    {
        NW_FATAL_ERROR("out of memory.\n");
        Finalize();
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    m_ItemNameBufferLength = GetItemNameBufferRequiredMemorySize(args.maxItemName);
    m_ItemNameBuffer = reinterpret_cast<char*>(frameHeap.Alloc(m_ItemNameBufferLength));

    if(m_ItemNameBuffer == NULL)
    {
        NW_FATAL_ERROR("out of memory.\n");
        Finalize();
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    Result result = InitializeForPlatform(frameHeap);

    if(result.IsFailed())
    {
        NW_FATAL_ERROR("failed to initialize.\n");
        Finalize();
        return result;
    }

    m_ErrorProvider = args.errorProvider;
    m_ResourceManager = args.resourceManager;
    m_SoundArchive = args.soundArchive;
    m_SoundArchivePlayer = args.soundArchivePlayer;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundArchiveEditController::Finalize()
{
    if(IsStarted())
    {
        Stop();
    }

    FinalizeForPlatform();

    m_ErrorProvider = NULL;
    m_ResourceManager = NULL;
    m_SoundArchive = NULL;
    m_SoundArchivePlayer = NULL;
    m_OpenFileBuffer = NULL;
    m_OpenFileBufferLength = 0;
    m_ItemNameBuffer = NULL;
    m_ItemNameBufferLength = 0;

#if defined(NW_PLATFORM_CTR)
    m_ErrorItemLockObject.Finalize();
    m_UnfreezeCacheLockObject.Finalize();
    m_CacheLockObject.Finalize();
#endif
}

//----------------------------------------------------------
Result
SoundArchiveEditController::Start(SoundEditConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);

    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditController is not initialized.\n");

    m_Connection = connection;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundArchiveEditController::Stop()
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditController is not initialized.\n");

    m_Connection = NULL;

    if(m_ResourceManager != NULL && m_ResourceManager->IsInitialized())
    {
        snd::internal::fnd::FndCriticalSectionScopedLock unfreezeLock(m_UnfreezeCacheLockObject);
        snd::internal::fnd::FndCriticalSectionScopedLock cacheLock(m_CacheLockObject);

        // アイテム情報は接続毎に破棄します。
        m_ResourceManager->RemoveAllItemInfos();

        // ファイルの参照カウントをクリアし、ガベージコレクション対象にします。
        m_ResourceManager->ClearAllItemFileReferenceCounts();
    }
}

//----------------------------------------------------------
u32
SoundArchiveEditController::GetRequiredMemorySize(u32 maxItemName) const
{
    return
        SoundEditConnection::GetRequiredMemorySizeForOpenFile() +
        GetItemNameBufferRequiredMemorySize(maxItemName) +
        GetRequiredMemorySizeForPlatform();
}

//----------------------------------------------------------
u32
SoundArchiveEditController::GetTotalMemoryUsage() const
{
    if(!IsInitialized())
    {
        return 0;
    }

    // SoundArchiveEditController 用のメモリ領域
    u32 result = m_OpenFileBufferLength + m_ItemNameBufferLength;

    // プラットフォーム固有のメモリ領域
    result += GetRequiredMemorySizeForPlatform();

    // ResourceManager 用のメモリ領域
    result += m_ResourceManager->GetMemoryUsage();

    return result;
}

//----------------------------------------------------------
void
SoundArchiveEditController::LockCacheStates()
{
    ++m_LockCacheStatesCount;

    if(m_LockCacheStatesCount > 1)
    {
        return;
    }

    m_UnfreezeCacheLockObject.Enter();
}

//----------------------------------------------------------
void
SoundArchiveEditController::UnlockCacheStates()
{
    if(m_LockCacheStatesCount == 0)
    {
        return;
    }

    --m_LockCacheStatesCount;

    if(m_LockCacheStatesCount == 0)
    {
        m_UnfreezeCacheLockObject.Leave();
    }
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateCaches(ICanceller* canceller /*= NULL*/, void* cancelerParam /*= NULL*/)
{
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    Result result = Result(SNDEDIT_RESULT_TRUE);

    if(m_Connection == NULL || !m_Connection->IsOpened())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    if(canceller != NULL)
    {
        canceller->Reset(cancelerParam);
    }

    if (m_SoundArchive == NULL || !m_SoundArchive->IsAvailable())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    for(;;)
    {
        // IsCancel() が true なら終了する。
        if(canceller != NULL && canceller->IsCanceled(cancelerParam))
        {
            break;
        }

        {
            ResourceManager::ErrorItem* errorItem = NULL;
            {
                snd::internal::fnd::FndCriticalSectionScopedLock lock(m_ErrorItemLockObject);
                errorItem = m_ResourceManager->PopErrorItem();
            }
            if (errorItem != NULL)
            {
                m_Connection->EndUpdateItemCache(errorItem->GetResult(), errorItem->GetItemName());
                m_ResourceManager->DestroyErrorItem(errorItem);

                break;
            }
        }

        // すべてのアイテムを対象としている場合は、一括処理します。
        UpdateCachesForAllItems();

        ResourceManager::RESOURCE_STATE itemState = ResourceManager::RESOURCE_STATE_UNKNOWN;
        const char* itemName = NULL;

        // 非フリーズアイテムを取得します。
        {
            snd::internal::fnd::FndCriticalSectionScopedLock lock(m_UnfreezeCacheLockObject);

            if(m_IsItemInfoInfoDictionaryInvalid)
            {
                m_ResourceManager->UpdateItemInfoDictionary();
                m_IsItemInfoInfoDictionaryInvalid = false;
            }

            itemName = m_ResourceManager->GetFirstUnfreezedItemName(&itemState);
        }

        if(itemName == NULL)
        {
            break;
        }

        switch(itemState)
        {
        case ResourceManager::RESOURCE_STATE_UPDATING:
            result = UpdateCache(itemName);
            NotifyEditItemsChanged();
            break;

        case ResourceManager::RESOURCE_STATE_REMOVING:
            result = RemoveCache(itemName);
            NotifyEditItemsChanged();
            break;

        default:
            NW_FATAL_ERROR("[sndedit] invalid RESOURCE_STATE.\n");
            break;
        }

        // メモリ不足時は、その時のメモリ使用量を記憶しておき、
        // それを下回るまで追加のキャッシュ処理を行いません。
        // ファイル数オーバーフロー時は、
        // 解消されるまで追加のキャッシュ処理を行いません。
        switch(result.GetValue())
        {
        case SNDEDIT_RESULT_OUT_OF_MEMORY:
            m_MemoryUsageAtOutOfMemory = GetTotalMemoryUsage();
            break;

        case SNDEDIT_RESULT_CACHE_OVER_FLOW:
            m_IsFilesOverFlow = true;
            break;

        default:
            m_MemoryUsageAtOutOfMemory = 0;
            m_IsFilesOverFlow = false;
            break;
        }

        if(result.IsFailed())
        {
            m_ErrorProvider->SetLastError(result);
            break;
        }
    }

    return result;
}

//----------------------------------------------------------
CacheState
SoundArchiveEditController::GetItemCacheState(const char* name) const
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT(*name != '\0');

    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);

    return m_ResourceManager->GetItemCacheState(name);
}

//----------------------------------------------------------
const ResSoundInfo*
SoundArchiveEditController::GetSoundInfo(
     const char* name,
     FILTER_FUNC filter /*= NULL*/,
     ResDataType* itemInfoDataType /*= NULL*/) const
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT(*name != '\0');

    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);

    ResDataType dataType = RES_DATA_TYPE_UNKNOWN;
    const void* result = m_ResourceManager->GetItemInfo(name, &dataType);

    if(itemInfoDataType != NULL)
    {
        *itemInfoDataType = static_cast<ResDataType>(dataType);
    }

    if(result == NULL || !ResItemInfoUtility::IsSoundInfoDataType(dataType))
    {
        return NULL;
    }

    if(filter != NULL && !filter(dataType))
    {
        return NULL;
    }

    return reinterpret_cast<const ResSoundInfo*>(result);
}

//----------------------------------------------------------
const ResBankInfo*
SoundArchiveEditController::GetBankInfo(const char* name) const
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT(*name != '\0');

    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);

    // キャッシュがあれば、それを返します。
    ResDataType dataType = RES_DATA_TYPE_UNKNOWN;
    const void* result = m_ResourceManager->GetItemInfo(name, &dataType);

    if(result == NULL || dataType != RES_DATA_TYPE_BANK)
    {
        return NULL;
    }

    return reinterpret_cast<const ResBankInfo*>(result);
}

//----------------------------------------------------------
const void*
    SoundArchiveEditController::GetFile(
    const char* filePath,
    ResDataType fileDataType,
    const Hash32** hashCode /*= NULL*/) const
{
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);
    return m_ResourceManager->GetFile(filePath, fileDataType, hashCode);
}

//----------------------------------------------------------
void
SoundArchiveEditController::InvokeUpdateAllItemInfos()
{
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_UnfreezeCacheLockObject);

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] invoke update all itemInfos.\n");
#endif

    m_ResourceManager->SetAllItemsState(ResourceManager::RESOURCE_STATE_UPDATING);
}

//----------------------------------------------------------
void
SoundArchiveEditController::InvokeRemoveAllItemInfos()
{
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_UnfreezeCacheLockObject);

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] invoke remove all itemInfos.\n");
#endif

    m_ResourceManager->SetAllItemsState(ResourceManager::RESOURCE_STATE_REMOVING);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::InvokeRemoveItemInfo(const char* name)
{
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_UnfreezeCacheLockObject);

    Result result = m_ResourceManager->UnfreezeItemInfo(name, ResourceManager::RESOURCE_STATE_REMOVING);

    if(result.IsFailed())
    {
        m_ErrorProvider->SetLastError(result);
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] invoke remove itemInfo '%s'. : result=%s\n", name, result.ToString());
#endif

    // リソースから見つからないということは削除できているのでTRUE扱いにする
    if (result == SNDEDIT_RESULT_NAME_NOT_FOUND)
    {
        ResourceManager::ErrorItem* errorItem = m_ResourceManager->CreateErrorItem(name, Result(SNDEDIT_RESULT_TRUE));
        if (errorItem != NULL)
        {
            snd::internal::fnd::FndCriticalSectionScopedLock errorItemLock(m_ErrorItemLockObject);
            m_ResourceManager->PushErrorItem(errorItem);
        }
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::MakeItemInfoEditable(const char* name)
{
    NW_ASSERT_NOT_NULL(name);
    NW_ASSERT(*name != '\0');

    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_UnfreezeCacheLockObject);

    // 既存のキャッシュが見つかった場合は、キャッシュのフリーズを解除し更新可能にします。
    // 既存のキャッシュが見つからない場合は、新しくエントリーを追加します。
    if(m_ResourceManager->ContainsItemInfo(name))
    {
        m_ResourceManager->UnfreezeItemInfo(name, ResourceManager::RESOURCE_STATE_UPDATING);
        return Result(SNDEDIT_RESULT_TRUE);
    }

    if (!m_SoundArchive->IsAvailable())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    Result result = m_ResourceManager->NewItemInfoEntry(
        name,
        m_SoundArchive->GetItemId(name));

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] make itemInfo editable '%s'. : result=%s\n", name, result.ToString());
#endif

    // キャッシュ上限に達した時は、
    // キャッシュが削除されるまで追加のキャッシュ処理を行いません。
    m_IsItemInfosOverFlow = result == SNDEDIT_RESULT_CACHE_OVER_FLOW;

    if(result.IsFailed())
    {
        if(m_IsItemInfosOverFlow)
        {
            ResourceManager::ErrorItem* errorItem = m_ResourceManager->CreateErrorItem(name, result);
            if (errorItem != NULL)
            {
                snd::internal::fnd::FndCriticalSectionScopedLock errorItemLock(m_ErrorItemLockObject);
                m_ResourceManager->PushErrorItem(errorItem);
            }
        }

        m_ErrorProvider->SetLastError(result);
        return result;
    }

    m_IsItemInfoInfoDictionaryInvalid = true;

    return result;
}

//----------------------------------------------------------
u32
SoundArchiveEditController::GetItemNameBufferRequiredMemorySize(u32 maxItemName) const
{
    return ut::RoundUp(maxItemName + 1, snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
}

//----------------------------------------------------------
void
SoundArchiveEditController::NotifyEditItemsChanged()
{
    if(m_EditItemsChangedCallback != NULL)
    {
        m_EditItemsChangedCallback(m_EditItemsChangedCallbackParam);
    }
}

//----------------------------------------------------------
void
SoundArchiveEditController::UpdateCachesForAllItems()
{
    ResourceManager::RESOURCE_STATE resourceState = ResourceManager::RESOURCE_STATE_UNKNOWN;

    {
        snd::internal::fnd::FndCriticalSectionScopedLock lock(m_UnfreezeCacheLockObject);

        resourceState = m_ResourceManager->GetAllItemsState();

        // 状態をクリアします。
        m_ResourceManager->SetAllItemsState(ResourceManager::RESOURCE_STATE_UNKNOWN);
    }

    switch(resourceState)
    {
    case ResourceManager::RESOURCE_STATE_UNKNOWN:
        break;

    case ResourceManager::RESOURCE_STATE_UPDATING:
        UpdateAllCaches();
        break;

    case ResourceManager::RESOURCE_STATE_REMOVING:
        RemoveAllCaches();
        break;

    default:
        NW_FATAL_ERROR("[sndedit] invalid RESOURCE_STATE.\n");
        break;
    }
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateCache(const char* itemName)
{
    // メモリ不足中の場合は、メモリ使用量が下がるまで、
    // 追加のキャッシュ処理を行いません。
    if(m_MemoryUsageAtOutOfMemory > 0 &&
        m_MemoryUsageAtOutOfMemory <= GetTotalMemoryUsage())
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    // ファイル数がオーバーフロー中は、解消されるまで、
    // 追加のキャッシュ処理を行いません。
    if(m_IsFilesOverFlow)
    {
        return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
    }

    Result result = Result(SNDEDIT_RESULT_FAILED);

    result = m_Connection->BeginUpdateItemCache(itemName);

    if(result.IsFailed())
    {
        return result;
    }

    // アイテム情報を取得します。
    ResDataType currentItemDataType = RES_DATA_TYPE_UNKNOWN;
    const void* currentItemInfo = m_ResourceManager->GetItemInfo(itemName, &currentItemDataType, true);

    ResDataType newItemDataType = RES_DATA_TYPE_UNKNOWN;
    const void* newItemInfo = NULL;

    result = QueryItemInfo(&newItemInfo, itemName, &newItemDataType);

    if(result.IsFailed())
    {
        return result;
    }

    NW_ASSERT_NOT_NULL(newItemInfo);

    if(currentItemInfo != NULL)
    {
        // 未キャッシュ状態の場合は、有効値が格納されていないので NULL ポインタにします。
        if(currentItemDataType == RES_DATA_TYPE_UNKNOWN)
        {
            currentItemInfo = NULL;
        }
        // データ種別が違う場合は、ファイルを削除します。
        else if(currentItemDataType != newItemDataType)
        {
            m_ResourceManager->RemoveItemFiles(itemName);
            currentItemInfo = NULL;
            currentItemDataType = RES_DATA_TYPE_UNKNOWN;
        }
    }

    // ファイルを更新します。
    switch(newItemDataType)
    {
    case RES_DATA_TYPE_STREAM_SOUND:
        // ストリームサウンドは、再生時にファイルを読み込むためキャッシュしませんが、
        // プリフェッチストリームデータについてキャッシュを行います。
        result = UpdateStreamSoundFile(
            reinterpret_cast<const ResStreamSoundInfo*>(currentItemInfo),
            reinterpret_cast<const ResStreamSoundInfo*>(newItemInfo)
            );
        break;

    case RES_DATA_TYPE_WAVE_SOUND:
        result = UpdateWaveSoundFile(
            reinterpret_cast<const ResWaveSoundInfo*>(currentItemInfo),
            reinterpret_cast<const ResWaveSoundInfo*>(newItemInfo)
            );
        break;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
#if defined(NW_ENABLE_CACHE_DEBUG)
        {
            const ResSequenceSoundInfo* inf = reinterpret_cast<const ResSequenceSoundInfo*>(newItemInfo);
            NW_LOG("[sndedit] Update SeqInfo : Name=%s, StartOffset=%d\n", itemName, inf->sequenceSoundParam.startOffset);
        }
#endif

        result = UpdateSequenceSoundFile(
            reinterpret_cast<const ResSequenceSoundInfo*>(currentItemInfo),
            reinterpret_cast<const ResSequenceSoundInfo*>(newItemInfo)
            );
        break;

    case RES_DATA_TYPE_BANK:
        result = UpdateBankFile(
            reinterpret_cast<const ResBankInfo*>(currentItemInfo),
            reinterpret_cast<const ResBankInfo*>(newItemInfo)
            );
        break;

    default:
        NW_FATAL_ERROR("[sndedit] invalid cache data type.\n");
        break;
    }

    if(result.IsSucceeded())
    {
        snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);

        // アイテム情報を登録、フリーズします。
        result = m_ResourceManager->SetItemInfo(
            itemName,
            m_SoundArchive->GetItemId(itemName),
            newItemDataType,
            newItemInfo);

        if(result.IsFailed())
        {
            NW_FATAL_ERROR("unexpected error.\n");
        }
        else
        {
            m_ResourceManager->UpdateItemInfoDictionary();

            result = m_ResourceManager->FreezeItemInfo(itemName);

            if(result.IsFailed())
            {
                NW_WARNING(false, "[sndedit] item info cache state is invalid.\n");
            }
            else
            {
                if(ResItemInfoUtility::IsSoundInfoDataType(newItemDataType))
                {
                    UpdatePlayingSoundParameters(itemName);
                }
            }
        }
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] update itemInfo '%s'. : result=%s\n", itemName, result.ToString());
#endif

    Result notifyResult = m_Connection->EndUpdateItemCache(result, itemName);

    if(result.IsSucceeded() && notifyResult.IsFailed())
    {
        return notifyResult;
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::RemoveCache(const char* itemName)
{
    Result result = Result(SNDEDIT_RESULT_FAILED);

    m_ResourceManager->GetItemInfoDataType(itemName, &result);

    if(result.IsFailed())
    {
        NW_WARNING(false, "[sndedit] item info cache state is invalid.\n");
        return result;
    }

    result = m_Connection->BeginRemoveItemCache(itemName);

    if(result.IsFailed())
    {
        return result;
    }

    // アイテム情報キャッシュを削除すると名前にアクセスできなくなるので、
    // バッファに退避します。
    NW_ASSERT(std::strlen(itemName) < m_ItemNameBufferLength);
    ut::strncpy(m_ItemNameBuffer, m_ItemNameBufferLength, itemName, std::strlen(itemName));

    // アイテム情報とファイルのキャッシュを削除します。
    {
        snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);

        result = m_ResourceManager->RemoveItemData(itemName);

#if defined(NW_ENABLE_CACHE_DEBUG)
        NW_LOG("[sndedit] remove itemInfo '%s'. : result=%s\n", m_ItemNameBuffer, result.ToString());
#endif
    }

    if(result.IsTrue())
    {
        m_IsItemInfosOverFlow = false;
        m_IsFilesOverFlow = false;
    }

    Result notifyResult = m_Connection->EndRemoveItemCache(result, m_ItemNameBuffer);

    if(result.IsSucceeded() && notifyResult.IsFailed())
    {
        return notifyResult;
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateAllCaches()
{
    // ファイルのキャッシュを削除してから全更新します。
    {
        snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);
        m_ResourceManager->RemoveAllItemFiles();
    }

    m_IsItemInfosOverFlow = false;
    m_IsFilesOverFlow = false;

    // 全アイテム更新対象とします。
    CacheUpdator cacheUpdator(*this);
    m_ResourceManager->ForEachItemInfo(cacheUpdator);

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::RemoveAllCaches()
{
    Result result = m_Connection->BeginRemoveAllItemCaches();

    if(result.IsFailed())
    {
        return result;
    }

    // アイテム情報とファイルのキャッシュを削除します。
    {
        snd::internal::fnd::FndCriticalSectionScopedLock lock(m_CacheLockObject);
        m_ResourceManager->RemoveAllItemInfos();
        m_ResourceManager->RemoveAllItemFiles();
    }

    m_IsItemInfosOverFlow = false;
    m_IsFilesOverFlow = false;

    result = m_Connection->EndRemoveAllItemCaches(Result(SNDEDIT_RESULT_TRUE));

    NotifyEditItemsChanged();

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateWaveSoundFile(
    const ResWaveSoundInfo* currentInfo,
    const ResWaveSoundInfo* newInfo)
{
    NW_ASSERT_NOT_NULL(newInfo);

    // bxwsd ファイルを更新します。
    const ResName* newWsdFilePath = NULL;
    {
        const ResName* currentWsdFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, RES_DATA_TYPE_WAVE_SOUND, RES_DATA_TYPE_WAVE_SOUND_FILE);

        newWsdFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, RES_DATA_TYPE_WAVE_SOUND, RES_DATA_TYPE_WAVE_SOUND_FILE);

        s32 fileSize = newInfo->waveSoundParam.fileSize;
        Hash32 wsdFileHashCode(newInfo->waveSoundParam.hashCode.value);

        Result result = UpdateFile(
            currentWsdFilePath,
            newWsdFilePath,
            RES_DATA_TYPE_WAVE_SOUND_FILE,
            fileSize,
            &wsdFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    // bxwar ファイルを更新します。
    const ResName* newWarFilePath = NULL;
    {
        const ResName* currentWarFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, RES_DATA_TYPE_WAVE_SOUND, RES_DATA_TYPE_WAVE_ARCHIVE_FILE);

        newWarFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, RES_DATA_TYPE_WAVE_SOUND, RES_DATA_TYPE_WAVE_ARCHIVE_FILE);

        s32 warFileSize = newInfo->waveSoundParam.waveArchiveFileSize;
        Hash32 warFileHashCode(newInfo->waveSoundParam.waveArchiveHashCode.value);

        Result result = UpdateFile(
            currentWarFilePath,
            newWarFilePath,
            RES_DATA_TYPE_WAVE_ARCHIVE_FILE,
            warFileSize,
            &warFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateSequenceSoundFile(
    const ResSequenceSoundInfo* currentInfo,
    const ResSequenceSoundInfo* newInfo)
{
    NW_ASSERT_NOT_NULL(newInfo);

    // bxseq ファイルを更新します。
    const ResName* currentSeqFilePath = currentInfo == NULL ?
        NULL :
        ResItemInfoUtility::GetFilePath(*currentInfo, RES_DATA_TYPE_SEQUENCE_SOUND, RES_DATA_TYPE_SEQUENCE_FILE);

    const ResName* newSeqFilePath =
        ResItemInfoUtility::GetFilePath(*newInfo, RES_DATA_TYPE_SEQUENCE_SOUND, RES_DATA_TYPE_SEQUENCE_FILE);

    s32 fileSize = newInfo->sequenceSoundParam.fileSize;
    Hash32 seqFileHashCode(newInfo->sequenceSoundParam.hashCode.value);

    Result result = UpdateFile(
        currentSeqFilePath,
        newSeqFilePath,
        RES_DATA_TYPE_SEQUENCE_FILE,
        fileSize,
        &seqFileHashCode);

    if(result.IsFailed())
    {
        return result;
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateStreamSoundFile(
    const ResStreamSoundInfo* currentInfo,
    const ResStreamSoundInfo* newInfo)
{
    NW_ASSERT_NOT_NULL(newInfo);

    if (newInfo->streamSoundParam.streamFileType == 2)
    {
        return Result(SNDEDIT_RESULT_TRUE);
    }

    if (newInfo->streamSoundParam.streamPrefetchFileSize <= 0)
    {
        return Result(SNDEDIT_RESULT_TRUE);
    }

    // bxstp ファイルを更新します。
    const ResName* currentStrmPrefetchFilePath = currentInfo == NULL ?
        NULL :
        ResItemInfoUtility::GetFilePath(*currentInfo, RES_DATA_TYPE_STREAM_SOUND, RES_DATA_TYPE_STREAM_PREFTCH_SOUND_FILE);

    const ResName* newStrmPrefetchFilePath =
        ResItemInfoUtility::GetFilePath(*newInfo, RES_DATA_TYPE_STREAM_SOUND, RES_DATA_TYPE_STREAM_PREFTCH_SOUND_FILE);

    s32 strmPrefetchFileSize = newInfo->streamSoundParam.streamPrefetchFileSize;
    Hash32 strmPrefetchFileHashCode(newInfo->streamSoundParam.streamPrefetchHashCode.value);

    Result result = UpdateFile(
        currentStrmPrefetchFilePath,
        newStrmPrefetchFilePath,
        RES_DATA_TYPE_STREAM_PREFTCH_SOUND_FILE,
        strmPrefetchFileSize,
        &strmPrefetchFileHashCode);

    if(result.IsFailed())
    {
        return result;
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateBankFile(
    const ResBankInfo* currentInfo,
    const ResBankInfo* newInfo)
{
    NW_ASSERT_NOT_NULL(newInfo);

    // bxbnk ファイルを更新します。
    const ResName* newBnkFilePath = NULL;
    {
        const ResName* currentBnkFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, RES_DATA_TYPE_BANK_FILE);

        newBnkFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, RES_DATA_TYPE_BANK_FILE);

        s32 fileSize = newInfo->bankParam.fileSize;
        Hash32 bnkFileHashCode(newInfo->bankParam.hashCode.value);

        Result result = UpdateFile(
            currentBnkFilePath,
            newBnkFilePath,
            RES_DATA_TYPE_BANK_FILE,
            fileSize,
            &bnkFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    // bxwar ファイルを更新します。
    const ResName* newWarFilePath = NULL;
    {
        const ResName* currentWarFilePath = currentInfo == NULL ?
            NULL :
            ResItemInfoUtility::GetFilePath(*currentInfo, RES_DATA_TYPE_WAVE_ARCHIVE_FILE);

        newWarFilePath =
            ResItemInfoUtility::GetFilePath(*newInfo, RES_DATA_TYPE_WAVE_ARCHIVE_FILE);

        s32 warFileSize = newInfo->bankParam.waveArchiveFileSize;
        Hash32 warFileHashCode(newInfo->bankParam.waveArchiveHashCode.value);

        Result result = UpdateFile(
            currentWarFilePath,
            newWarFilePath,
            RES_DATA_TYPE_WAVE_ARCHIVE_FILE,
            warFileSize,
            &warFileHashCode);

        if(result.IsFailed())
        {
            return result;
        }
    }

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::QueryItemInfo(
    const void** data,
    const char* itemName,
    ResDataType* itemDataType /*= NULL*/,
    FILTER_FUNC filter /*= NULL*/) const
{
    NW_ASSERT_NOT_NULL(data);
    NW_ASSERT_NOT_NULL(itemName);
    NW_ASSERT(*itemName != '\0');

    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.");

    if(m_Connection == NULL || !m_Connection->IsOpened())
    {
        *data = NULL;
        return Result(SNDEDIT_RESULT_FAILED);
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG("[sndedit] query itemInfo '%s'.\n", itemName);
#endif

    const HioPacket* replyPacket = NULL;
    Result result = m_Connection->QueryItemInfo(replyPacket, itemName);

    if(result.IsFailed())
    {
        return result;
    }

    NW_ASSERT_NOT_NULL(replyPacket);

    const QueryItemInfoReplyPacket* queryItemInfoReply =
        reinterpret_cast<const QueryItemInfoReplyPacket*>(replyPacket);

    NW_ASSERT_NOT_NULL(queryItemInfoReply);

    if(filter != NULL && !filter(queryItemInfoReply->GetItemType()))
    {
        return Result(SNDEDIT_RESULT_INVALID_DATA_TYPE);
    }

    if(itemDataType != NULL)
    {
        *itemDataType = queryItemInfoReply->GetItemType();
    }

    *data = queryItemInfoReply->GetItemInfo();
    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
SoundArchiveEditController::UpdateFile(
    const ResName* currentFilePath,
    const ResName* newFilePath,
    ResDataType fileDataType,
    s32 fileSize,
    Hash32* hashCode /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(newFilePath);
    NW_ASSERT(*newFilePath->GetName() != '\0');

    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    if(currentFilePath != NULL)
    {
        // ファイルパスが異なる場合は、古いファイルを削除します。
        if(currentFilePath->GetLength() == newFilePath->GetLength() &&
            std::strncmp(currentFilePath->GetName(), newFilePath->GetName(), newFilePath->GetLength()) != 0)
        {
            m_ResourceManager->RemoveFile(currentFilePath->GetName());
        }
    }

    // 該当キャッシュがあれば、それを返します。
    // なければ、SoundEditConnection から読み込みます。
    const Hash32* fileDataHashCode = NULL;
    const void* fileData = m_ResourceManager->GetFile(newFilePath->GetName(), fileDataType, &fileDataHashCode);

    if(fileData != NULL)
    {
        if(hashCode == NULL || *hashCode == *fileDataHashCode)
        {
            // 有効なキャッシュが見つかったら、参照カウントをインクリメントします。
            if(m_ResourceManager->IncrementFileReferenceCount(newFilePath->GetName()).IsFailed())
            {
                NW_FATAL_ERROR("unexpected error : file cache state is invalid. Path=%s\n", newFilePath->GetName());
                return Result(SNDEDIT_RESULT_FAILED);
            }

            return Result(SNDEDIT_RESULT_TRUE);
        }

        // ハッシュ値が異なる場合は、古いファイルを削除して再キャッシュします。
        if(currentFilePath != NULL && currentFilePath->GetLength() > 0)
        {
            m_ResourceManager->RemoveFile(currentFilePath->GetName());
        }
    }

    // キーの重複を防ぐために、事前に削除しておきます。
    m_ResourceManager->RemoveFile(newFilePath->GetName());

    Result result = ReadAndCacheFile(newFilePath->GetName(), fileDataType, fileSize, hashCode);

    if(result.IsSucceeded())
    {
        // 辞書を更新します。
        m_ResourceManager->UpdateFileDictionary();
    }

    return result;
}

//----------------------------------------------------------
Result
SoundArchiveEditController::ReadAndCacheFile(
    const char* filePath,
    ResDataType fileDataType,
    s32 fileSize,
    Hash32* hashCode /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERT(*filePath != '\0');
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    snd::internal::fnd::FileStream* fileStream =
        OpenFileForCache(m_OpenFileBuffer, m_OpenFileBufferLength, filePath);

    if(fileStream == NULL)
    {
        return Result(SNDEDIT_RESULT_FAILED);
    }

    if(fileSize == 0)
    {
        fileStream->Close();
        return Result(SNDEDIT_RESULT_FAILED);
    }

    s32 alignedFileSize = fileSize + Alignments::FILE_IO_BUFFER;

    // ファイルキャッシュ領域を作成します。
    void* fileBuffer = NULL;
    Result cacheResult = m_ResourceManager->NewFileEntry(filePath, alignedFileSize, fileDataType, &fileBuffer, hashCode);

    if(cacheResult.IsFailed())
    {
        fileStream->Close();

#if defined(NW_ENABLE_CACHE_DEBUG)
        NW_LOG(
            "[sndedit] cache allocation failed. : usage=%dkb, alignedFileSize=%dbyte, result=%s.\n",
            m_ResourceManager->GetMemoryUsage() / 1024,
            alignedFileSize,
            cacheResult.ToString()
            );
#endif

        return cacheResult;
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] cache allocation succeeded. : usage=%dkb, alignedFileSize=%dbyte.\n",
        m_ResourceManager->GetMemoryUsage() / 1024,
        alignedFileSize
        );
#endif

    NW_ASSERT_NOT_NULL(fileBuffer);

    // ファイル入出力用にアライメントして、データを読み込みます。
    void* alignedFileBuffer = ut::RoundUp(fileBuffer, Alignments::FILE_IO_BUFFER);

    s32 readResult = fileStream->Read(alignedFileBuffer, fileSize);
    fileStream->Close();

#if defined(NW_ENABLE_CACHE_DEBUG)
    if(fileSize > 32)
    {
        const u8* data = reinterpret_cast<const u8*>(alignedFileBuffer);
        NW_LOG(
            "[sndedit] ReadAndCacheFile : Data=%02x%02x%02x%02x%02x%02x%02x%02x %02x%02x%02x%02x%02x%02x%02x%02x %02x%02x%02x%02x%02x%02x%02x%02x %02x%02x%02x%02x%02x%02x%02x%02x, Path=%s\n",
            data[0],data[1],data[2],data[3],
            data[4],data[5],data[6],data[7],
            data[8],data[9],data[10],data[11],
            data[12],data[13],data[14],data[15],
            data[16],data[17],data[18],data[19],
            data[20],data[21],data[22],data[23],
            data[24],data[25],data[26],data[27],
            data[28],data[29],data[30],data[31],
            filePath);
    }
#endif

    if(readResult < 0)
    {
        m_ResourceManager->RemoveFile(filePath);
        return Result(SNDEDIT_RESULT_FAILED);
    }

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundArchiveEditController::UpdatePlayingSoundParameters(const char* soundName)
{
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return;
    }

    ResDataType soundInfoType = RES_DATA_TYPE_UNKNOWN;
    const ResSoundInfo* soundInfo = GetSoundInfo(soundName, NULL, &soundInfoType);

    if(soundInfo == NULL)
    {
        return;
    }

    UpdatePlayingSoundParameters(soundName, soundInfo, soundInfoType);
}

//----------------------------------------------------------
void
SoundArchiveEditController::UpdatePlayingSoundParameters(
    const char* soundName,
    const ResSoundInfo* soundInfo,
    ResDataType soundInfoType)
{
    NW_ASSERT_NOT_NULL(soundInfo);

    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditController is not started.\n");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return;
    }

    SoundArchive::ItemId soundID = m_SoundArchive->GetItemId(soundName);

    if(soundID == SoundArchive::INVALID_ID)
    {
        return;
    }

    // 再生中サウンドのパラメータに適用します。
    for(u32 playerIndex = 0; playerIndex < m_SoundArchive->GetPlayerCount(); ++playerIndex)
    {
        snd::SoundArchive::ItemId playerID = m_SoundArchive->GetPlayerIdFromIndex(playerIndex);

        m_SoundArchivePlayer->GetSoundPlayer(playerID).ForEachSound(
            ResSoundInfoApplier(soundID, soundInfo, soundInfoType));
    }
}

//----------------------------------------------------------
SoundArchiveEditController::ResSoundInfoApplier::ResSoundInfoApplier(
    SoundArchive::ItemId soundID,
    const ResSoundInfo* soundEditInfo,
    ResDataType soundInfoType) :
m_SoundID(soundID),
m_SoundEditInfo(soundEditInfo),
m_SoundInfoType(soundInfoType)
{
    NW_ASSERT_NOT_NULL(soundEditInfo);
    NW_ASSERT(ResItemInfoUtility::IsSoundInfoDataType(soundInfoType));
}

//----------------------------------------------------------
void
SoundArchiveEditController::ResSoundInfoApplier::operator()(SoundHandle& soundHandle)
{
    if(m_SoundID != soundHandle.GetId())
    {
        return;
    }

    snd::internal::BasicSound* sound = soundHandle.detail_GetAttachedSound();

    ApplySoundInitialVolume(sound);
    ApplyStreamSoundTrackInitialVolume(sound);
}

//----------------------------------------------------------
void
SoundArchiveEditController::ResSoundInfoApplier::ApplySoundInitialVolume(
    snd::internal::BasicSound* sound)
{
    NW_ASSERT_NOT_NULL(sound);

    sound->SetInitialVolume(static_cast<f32>(m_SoundEditInfo->soundBasicParam.volume) / 127.f);
}

//----------------------------------------------------------
void
SoundArchiveEditController::ResSoundInfoApplier::ApplyStreamSoundTrackInitialVolume(
    snd::internal::BasicSound* sound)
{
    NW_ASSERT_NOT_NULL(sound);

    if(!ResItemInfoUtility::IsStreamSoundInfoDataType(m_SoundInfoType))
    {
        return;
    }

    snd::internal::StreamSound* streamSound = ut::DynamicCast<snd::internal::StreamSound*>(sound);

    if(streamSound == NULL)
    {
        return;
    }

    const ResStreamSoundInfo* streamSoundEditInfo =
        reinterpret_cast<const ResStreamSoundInfo*>(m_SoundEditInfo);

    u32 trackFlags = 1;

    for(u32 index = 0; index < SoundArchive::STRM_TRACK_NUM; ++index)
    {
        streamSound->SetTrackInitialVolume(
            trackFlags,
            streamSoundEditInfo->streamSoundParam.trackParams[index].volume);

        trackFlags <<= 1;
    }
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
