﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits.h>
#include <nw/snd/dw/control/snddw_WaveGraph.h>

namespace {
    const s32 VERTEX_NUM = 2000;
    const s32 INDEX_NUM = VERTEX_NUM;
}

namespace nw {
namespace snd {
namespace dw {
namespace internal {

WaveGraph::WaveGraph() :
m_pWaveData(NULL),
m_WaveDataSize(0),
m_WaveDataStartIndex(INVALID_INDEX),
m_WaveDataCachedIndex(INVALID_INDEX),
m_pAllocator(NULL),
m_SamplesPerSec(48000),
m_ScaleX(1.f)
{
    m_DrawBuffer.vertex = NULL;
    m_DrawBuffer.index = NULL;

    SetMargin(nw::internal::dw::Thickness(0.f));
    SetPadding(1.f);
    SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
}

void
WaveGraph::Initialize(nw::ut::IAllocator& allocator)
{
    m_pAllocator = &allocator;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    m_DrawBuffer.vertex = reinterpret_cast<nw::dev::Vertex*>( allocator.Alloc( sizeof( nw::dev::Vertex ) * VERTEX_NUM, 0x40 ) );
    m_DrawBuffer.index  = reinterpret_cast<u16*>( allocator.Alloc( sizeof( u16 ) * INDEX_NUM, 0x40 ) );
#else
    m_DrawBuffer.vertex = reinterpret_cast<nw::dev::Vertex*>( allocator.Alloc( sizeof( nw::dev::Vertex ) * VERTEX_NUM, GX2_VERTEX_BUFFER_ALIGNMENT ) );
    m_DrawBuffer.index  = reinterpret_cast<u16*>( allocator.Alloc( sizeof( u16 ) * INDEX_NUM, GX2_INDEX_BUFFER_ALIGNMENT ) );
#endif

    for (s32 i = 0; i < VERTEX_NUM; ++i)
    {
        m_DrawBuffer.vertex[i].pos.Set( 0.f, 0.f, 0.f );
        m_DrawBuffer.vertex[i].color.Set( 1.f, 1.f, 1.f, 0.f );
    }

    for (s32 i = 0; i < INDEX_NUM; ++i)
    {
         m_DrawBuffer.index[i] = static_cast<u16>(i);
    }

    m_DrawBuffer.vertexNum = VERTEX_NUM;
    m_DrawBuffer.indexNum  = INDEX_NUM;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    nw::dev::PrimitiveRenderer::GetInstance()->CreateVertexBuffer( m_DrawBuffer );
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
#else
    GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_DrawBuffer.vertex, sizeof( nw::dev::Vertex ) * VERTEX_NUM );
    GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_DrawBuffer.index, sizeof( u16 ) * INDEX_NUM );
#endif
}

void
WaveGraph::Finalize()
{
    NW_ASSERT_NOT_NULL(m_pAllocator);

    if (m_DrawBuffer.index != NULL)
    {
        nw::ut::SafeFree(m_DrawBuffer.index, m_pAllocator);
    }

    if (m_DrawBuffer.vertex != NULL)
    {
        nw::ut::SafeFree(m_DrawBuffer.vertex, m_pAllocator);
    }

    m_pAllocator = NULL;
}

u32
WaveGraph::GetRequiredDrawBufferSize() const
{
    return sizeof( nw::dev::Vertex ) * VERTEX_NUM + sizeof( u16 ) * INDEX_NUM + 1024;
}

void
WaveGraph::SetWaveData(const s16* pWaveData, u32 waveDataSize)
{
    // バッファ位置を検証する処理にて、+waveDataSize しても
    // オーバーフローしないサイズである必要があります。
    NW_ASSERT(waveDataSize < INT_MAX / 2);

    m_pWaveData = pWaveData;
    m_WaveDataSize = waveDataSize;
    m_WaveDataStartIndex = 0;
}

void
WaveGraph::SetWaveDataStartIndex(u32 value)
{
    NW_ASSERT(value < static_cast<u32>(m_WaveDataSize));

    s32 newValue = static_cast<s32>(value);
    s32 logicalNewValue = newValue < m_WaveDataStartIndex ? newValue + m_WaveDataSize : newValue;
    NW_ASSERT(m_WaveDataStartIndex <= logicalNewValue);

    // 波形データ開始位置がキャッシュ位置を抜かした場合、
    // すべてのキャッシュを無効化します。
    if(m_WaveDataCachedIndex != INVALID_INDEX &&
        m_WaveDataStartIndex < m_WaveDataCachedIndex &&
        m_WaveDataCachedIndex < logicalNewValue)
    {
        m_WaveDataCachedIndex = INVALID_INDEX;
    }

    m_WaveDataStartIndex = value;
}

u32
WaveGraph::GetRequiredWaveCacheSize()
{
    nw::math::Vector2 size = GetContentAreaSize();
    return static_cast<u32>(size.x) * sizeof(ValueRange);
}

void
WaveGraph::SetWaveCacheBuffer(s16* pBuffer, u32 bufferSize)
{
    m_CacheManager.SetBuffer(pBuffer, bufferSize);
}

u32
WaveGraph::GetSamplesPerSec() const
{
    return m_SamplesPerSec;
}

void
WaveGraph::SetSamplesPerSec(u32 value)
{
    m_SamplesPerSec = value;
}

f32
WaveGraph::GetScaleX() const
{
    return m_ScaleX;
}

void
WaveGraph::SetScaleX(f32 value)
{
    NW_ASSERT(value != 0);

    if(m_ScaleX == value)
    {
        return;
    }

    m_ScaleX = value;

    m_WaveDataCachedIndex = INVALID_INDEX;
    m_CacheManager.Clear();
}

void
WaveGraph::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    (void)context;
    CacheValueRanges();
}

void
WaveGraph::OnRender(const nw::internal::dw::UIElementTreeContext& context, nw::internal::dw::UIRenderer& renderer, nw::internal::dw::UIElementRenderArgs& args) const
{
    (void)args;
    if(!IsValidWaveData())
    {
        return;
    }

    static nw::internal::dw::DrawLineArgs xAxisDrawLineArgs =
        nw::internal::dw::DrawLineArgs().
        SetColor0(nw::ut::Color4f(0.3f, 0.3f, 0.3f, 1.f)).
        SetColor1(nw::ut::Color4f(0.3f, 0.3f, 0.3f, 1.f));

    static nw::internal::dw::DrawLineArgs drawLineArgs =
        nw::internal::dw::DrawLineArgs().
        SetColor0(nw::ut::Color4f::X_DEEP_SKY_BLUE()).
        SetColor1(nw::ut::Color4f::X_DEEP_SKY_BLUE());

    nw::internal::dw::Thickness padding = GetPadding();
    nw::math::Vector2 size = GetContentAreaSize();
    f32 yAxis = padding.top + (size.y / 2);

    // X軸を描画します。
    renderer.DrawLine(
        &context,
        xAxisDrawLineArgs.SetFrom(padding.left, yAxis).SetTo(padding.left + size.x, yAxis));

    nw::internal::dw::DrawMultiLineArgs drawLinesArgs =
        nw::internal::dw::DrawMultiLineArgs(m_DrawBuffer).
        SetColor(nw::ut::Color4f::X_DEEP_SKY_BLUE());

    renderer.DrawLine(
        &context,
        drawLinesArgs
    );
}

void
WaveGraph::CacheValueRanges()
{
    if(!IsValidWaveData())
    {
        return;
    }

    nw::internal::dw::Thickness padding = GetPadding();
    nw::math::Vector2 size = GetContentAreaSize();

    s32 samplePerPixel = static_cast<s32>(
        nw::math::FFloor(static_cast<f32>(m_SamplesPerSec) * m_ScaleX / size.x)
        );

    if(samplePerPixel == 0)
    {
        m_WaveDataCachedIndex = INVALID_INDEX;
        m_CacheManager.Clear();
        return;
    }

    s32 currentIndex = 0;
    s32 restSamples = 0;

    if(m_WaveDataCachedIndex == INVALID_INDEX)
    {
        currentIndex = ValidateIndex(m_WaveDataStartIndex);
        restSamples = m_WaveDataSize;
    }
    else
    {
        currentIndex = m_WaveDataCachedIndex;
        restSamples = m_WaveDataCachedIndex < m_WaveDataStartIndex ?
            m_WaveDataStartIndex - m_WaveDataCachedIndex :
            m_WaveDataSize - m_WaveDataCachedIndex + m_WaveDataStartIndex;
    }

    f32 yScale = size.y / 2 / SHRT_MAX;

    while(restSamples >= samplePerPixel)
    {
        ValueRange* range = m_CacheManager.GetNextWriteCache();
        NW_ASSERT_NOT_NULL(range);

        GetValueRange(currentIndex, samplePerPixel, yScale, *range);
        m_CacheManager.IncrementWriteIndex();

        restSamples -= samplePerPixel;

        currentIndex = ValidateIndex(currentIndex + samplePerPixel);
        m_WaveDataCachedIndex = currentIndex;
    }

    UpdateDrawBuffer();
}

void
WaveGraph::UpdateDrawBuffer()
{
    nw::internal::dw::Thickness padding = GetPadding();
    nw::math::Vector2 size = GetContentAreaSize();
    f32 yAxis = padding.top + (size.y / 2);

    if(m_WaveDataCachedIndex == INVALID_INDEX)
    {
        return;
    }

    static const u32 invalidHoldX = std::numeric_limits<u32>::max();

    u32 x = static_cast<u32>(padding.left + size.x) - 1;
    f32 floatX = static_cast<f32>(x);

    u32 paddingLeft = static_cast<u32>(padding.left);
    u32 holdX = invalidHoldX;
    f32 holdY = -1.f;

    s32 lastIndex = m_CacheManager.GetWriteIndex() - 1;

    s32 bufferIndex = 0;

    for(s32 offset = 0; offset < m_CacheManager.GetValidLength(); ++offset)
    {
        ValueRange* range = m_CacheManager.GetCache(lastIndex - offset);

        if(range == NULL)
        {
            break;
        }

        if (bufferIndex >= VERTEX_NUM - 5)
        {
            // ここに来ることはないはず。
            NW_ASSERT(false);
            break;
        }

        if(ut::FloatEquals(range->min, range->max))
        {
            // ValueRange が点を指す場合は、Y軸の値が変化するまでラインを描画しません。
            if(range->min != holdY)
            {
                if(holdX > 0 && holdY > 0.f)
                {
                    if(holdX > x + 1)
                    {
                        f32 prevX = static_cast<f32>(x + 1);

                        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                            static_cast<f32>(holdX),
                            yAxis + holdY,
                            0.0f
                        );
                        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                            prevX,
                            yAxis + holdY,
                            0.0f
                        );

                        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                            prevX,
                            yAxis + holdY,
                            0.0f
                        );
                        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                            floatX,
                            yAxis + range->min,
                            0.0f
                        );
                    }
                    else
                    {
                        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                            static_cast<f32>(holdX),
                            yAxis + holdY,
                            0.0f
                        );
                        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                            floatX,
                            yAxis + range->min,
                            0.0f
                        );
                    }
                }

                holdX = x;
                holdY = range->min;
            }
        }
        else
        {
            // 座標 x に 出力範囲ラインを描画
            m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                floatX,
                yAxis + range->max,
                0.0f
            );
            m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                floatX,
                yAxis + range->min,
                0.0f
            );

            // 座標 holdX から x に時間幅ラインを描画
            if(holdX != invalidHoldX)
            {
                if(holdX > x + 1)
                {
                    f32 prevX = static_cast<f32>(x + 1);

                    m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                        static_cast<f32>(holdX),
                        yAxis + holdY,
                        0.0f
                    );
                    m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                        prevX,
                        yAxis + holdY,
                        0.0f
                    );

                    m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                        prevX,
                        yAxis + holdY,
                        0.0f
                    );
                    m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                        floatX,
                        yAxis + range->first,
                        0.0f
                    );
                }
                else
                {
                    m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                        static_cast<f32>(holdX),
                        yAxis + holdY,
                        0.0f
                    );
                    m_DrawBuffer.vertex[bufferIndex++].pos.Set(
                        floatX,
                        yAxis + range->first,
                        0.0f
                    );
                }
            }

            holdX = x;
            holdY = range->last;
        }

        u32 nextX = x - 1;

        if(nextX < paddingLeft)
        {
            break;
        }

        x = nextX;
        floatX = static_cast<f32>(x);
    }

    if(x > paddingLeft)
    {
        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
            static_cast<f32>(holdX),
            yAxis + holdY,
            0.0f
        );
        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
            floatX,
            yAxis + holdY,
            0.0f
        );

        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
            floatX,
            yAxis + holdY,
            0.0f
        );
        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
            floatX - 1.f,
            yAxis,
            0.0f
        );

        holdX = x - 1;
        holdY = 0.f;
    }

    if(holdX > paddingLeft)
    {
        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
            static_cast<f32>(holdX),
            yAxis + holdY,
            0.0f
        );
        m_DrawBuffer.vertex[bufferIndex++].pos.Set(
            padding.left,
            yAxis + holdY,
            0.0f
        );
    }

    m_DrawBuffer.vertexNum = bufferIndex;
    m_DrawBuffer.indexNum = m_DrawBuffer.vertexNum;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();

    glBindBuffer( GL_ARRAY_BUFFER, m_DrawBuffer.buffer[ 0 ] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( nw::dev::Vertex ) * m_DrawBuffer.vertexNum, m_DrawBuffer.vertex, GL_STATIC_DRAW );
    NW_GL_ASSERT();

    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, m_DrawBuffer.buffer[ 1 ] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( u16 ) * m_DrawBuffer.indexNum, m_DrawBuffer.index, GL_STATIC_DRAW );
    NW_GL_ASSERT();

    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
#else
    GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_DrawBuffer.vertex, sizeof( nw::dev::Vertex ) * m_DrawBuffer.vertexNum );
    GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_DrawBuffer.index, sizeof( u16 ) * m_DrawBuffer.indexNum );
#endif
}

inline void
WaveGraph::GetValueRange(s32 index, s32 samples, f32 yScale, WaveGraph::ValueRange& valueRange) const
{
    NW_ASSERT(samples > 0);
    NW_ASSERT(m_WaveDataSize > 0);

    s32 lastIndex = ValidateIndex(index + samples - 1);
    s16 min = m_pWaveData[index];
    s16 max = m_pWaveData[index];

    for(s32 currentIndex = index + 1; currentIndex < index + samples; currentIndex++)
    {
        s32 physicalIndex = ValidateIndex(currentIndex);

        min = nw::ut::Min(min, m_pWaveData[physicalIndex]);
        max = nw::ut::Max(max, m_pWaveData[physicalIndex]);
    }

    valueRange.first = ValueToYPosition(m_pWaveData[index], yScale);
    valueRange.last = ValueToYPosition(m_pWaveData[lastIndex], yScale);
    valueRange.min = ValueToYPosition(min, yScale);
    valueRange.max = ValueToYPosition(max, yScale);
}

inline s32
WaveGraph::ValidateIndex(s32 logicalIndex) const
{
    s32 result = logicalIndex;

    if(result < m_WaveDataSize)
    {
        if(result < 0)
        {
            result += m_WaveDataSize;
        }
    }
    else
    {
        result -= m_WaveDataSize;
    }

    NW_ASSERT(result < m_WaveDataSize);

    return result;
}

inline f32
WaveGraph::ValueToYPosition(s16 value, f32 scale) const
{
    return static_cast<f32>(value) * scale;
}

inline bool
WaveGraph::IsValidWaveData() const
{
    return m_pWaveData != NULL && m_WaveDataSize > 0;
}

//----------------------------------------------------------
WaveGraph::CacheManager::CacheManager() :
m_pData(NULL),
m_DataLength(0),
m_ValidLength(0),
m_WriteIndex(0)
{
}

//----------------------------------------------------------
void
WaveGraph::CacheManager::SetBuffer(s16* pBuffer, u32 bufferLength)
{
    m_pData = reinterpret_cast<ValueRange*>(pBuffer);
    m_DataLength = bufferLength / sizeof(ValueRange);

    Clear();
}

//----------------------------------------------------------
WaveGraph::ValueRange*
WaveGraph::CacheManager::GetCache(s32 logicalIndex) const
{
    if(m_ValidLength == 0)
    {
        return NULL;
    }

    s32 physicalIndex = GetPhysicalIndex(logicalIndex);

    if(m_ValidLength < m_DataLength &&
        physicalIndex >= m_ValidLength)
    {
        return NULL;
    }

    return &m_pData[physicalIndex];
}

//----------------------------------------------------------
WaveGraph::ValueRange*
WaveGraph::CacheManager::GetNextWriteCache() const
{
    return &m_pData[GetPhysicalIndex(m_WriteIndex)];
}

//----------------------------------------------------------
void
WaveGraph::CacheManager::IncrementWriteIndex()
{
    s32 newIndex = m_WriteIndex + 1;

    if(newIndex >= m_DataLength)
    {
        newIndex = 0;
    }

    m_WriteIndex = newIndex;

    if(m_ValidLength < m_DataLength)
    {
        m_ValidLength++;
    }
}

//----------------------------------------------------------
void
WaveGraph::CacheManager::Clear()
{
    m_ValidLength = 0;
    m_WriteIndex = 0;
}

//----------------------------------------------------------
inline s32
WaveGraph::CacheManager::GetPhysicalIndex(s32 logicalIndex) const
{
    s32 result = logicalIndex;

    if(result < m_DataLength)
    {
        if(result < 0)
        {
            result += m_DataLength;
        }
    }
    else
    {
        result -= m_DataLength;
    }

    if(result < 0 || m_DataLength <= result)
    {
        return INVALID_INDEX;
    }

    return result;
}

} // internal
} // dw
} // snd
} // nw
