﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_WaveArchiveFileReader.h>

// #define NW_SND_DEBUG_PRINT_ENABLE

namespace nw {
namespace snd {
namespace internal {

namespace
{

// const u32 SIGNATURE_INFO_BLOCK = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );
// const u32 SIGNATURE_FILE_BLOCK = NW_UT_MAKE_SIGWORD( 'F', 'I', 'L', 'E' );

const u32 SUPPORTED_FILE_VERSION_WAR = 0x00010000;  // ライブラリがサポートする最低バージョン
const u32 CURRENT_FILE_VERSION_WAR   = 0x00010000;  // ライブラリがサポートする最新バージョン

bool IsValidFileHeaderWar( const void* waveArchiveData )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( waveArchiveData );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( waveArchiveData );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header.signature == WaveArchiveFileReader::SIGNATURE_FILE,
            "invalid file signature." );
    if ( header.signature != WaveArchiveFileReader::SIGNATURE_FILE )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_WAR <= header.version) &&
                                   (header.version <= CURRENT_FILE_VERSION_WAR) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfwar file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_WAR, CURRENT_FILE_VERSION_WAR, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace

const u32 WaveArchiveFileReader::SIGNATURE_WARC_TABLE = NW_UT_MAKE_SIGWORD( 'F', 'W', 'A', 'T' );

WaveArchiveFileReader::WaveArchiveFileReader()
: m_pHeader( NULL ),
  m_pInfoBlockBody( NULL ),
  m_pLoadTable( NULL ),
  m_IsInitialized( false )
{
}

WaveArchiveFileReader::WaveArchiveFileReader(
        const void* pWaveArchiveFile,
        bool isIndividual )
: m_pHeader( NULL ),
  m_pInfoBlockBody( NULL ),
  m_pLoadTable( NULL ),
  m_IsInitialized( false )
{
    Initialize( pWaveArchiveFile, isIndividual );
}

void WaveArchiveFileReader::Initialize(
        const void* pWaveArchiveFile,
        bool isIndividual )
{
    if ( pWaveArchiveFile == NULL )
    {
        return;
    }

    if ( ! IsValidFileHeaderWar( pWaveArchiveFile ) )
    {
        return;
    }

    m_pHeader =
        reinterpret_cast<const WaveArchiveFile::FileHeader*>( pWaveArchiveFile );
    m_pInfoBlockBody = &m_pHeader->GetInfoBlock()->body;

    m_IsInitialized = true;

    // MemorySoundArchive 内の波形アーカイブが個別ロード ON であっても、
    // Hasindividualloadtable が false を返すので、m_pLoadTable は NULL のままに
    // なり、IsLoaded が正しく動作する。
    m_pLoadTable = NULL;
    if ( isIndividual && HasIndividualLoadTable() )
    {
        m_pLoadTable = reinterpret_cast<IndividualLoadTable*>(
                ut::AddOffsetToPtr(
                    const_cast<void*>( pWaveArchiveFile ),
                    m_pHeader->GetFileBlockOffset() +
                    sizeof( SIGNATURE_WARC_TABLE ) ) );
            // INFO ブロックと FILE ブロックの間に、
            // アライメントがはさまっているので、その分ムダになるが、
            // 見通しのよさを優先する。
    }
}

void WaveArchiveFileReader::Finalize()
{
    if ( m_IsInitialized )
    {
        m_pHeader = NULL;
        m_pInfoBlockBody = NULL;
        m_pLoadTable = NULL;
        m_IsInitialized = false;
    }
}

void WaveArchiveFileReader::InitializeFileTable()
{
    NW_NULL_ASSERT( m_pInfoBlockBody );

    // ロード管理用ファイルテーブルの初期化
    for ( u32 i = 0; i < GetWaveFileCount(); i++ )
    {
        m_pLoadTable->waveFile[ i ] = NULL;
    }
}

u32 WaveArchiveFileReader::GetWaveFileCount() const
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetWaveFileCount();
}

const void* WaveArchiveFileReader::GetWaveFile( u32 waveIndex ) const
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NW_NULL_ASSERT( m_pInfoBlockBody );
    if ( waveIndex >= GetWaveFileCount() ) return NULL;

    if ( m_pLoadTable != NULL )
    {
        return GetWaveFileForIndividual( waveIndex );
    }
    return GetWaveFileForWhole( waveIndex );
}

u32 WaveArchiveFileReader::GetWaveFileSize( u32 waveIndex ) const
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetSize( waveIndex );
}

u32 WaveArchiveFileReader::GetWaveFileOffsetFromFileHead( u32 waveIndex ) const
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    NW_NULL_ASSERT( m_pInfoBlockBody );
    u32 result =
        + m_pHeader->GetFileBlockOffset()
        + offsetof( WaveArchiveFile::FileBlock, body )
        + m_pInfoBlockBody->GetOffsetFromFileBlockBody( waveIndex );
    return result;
}

const void*
WaveArchiveFileReader::SetWaveFile(
        u32 waveIndex, const void* pWaveFile )
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    if ( m_pLoadTable == NULL ) return NULL;
    if ( waveIndex >= GetWaveFileCount() ) return NULL;

    const void* preAddress = GetWaveFileForIndividual( waveIndex );
    m_pLoadTable->waveFile[ waveIndex ] = pWaveFile;

#ifdef NW_SND_DEBUG_PRINT_ENABLE
    // デバッグ
    for ( u32 i = 0; i < GetWaveFileCount(); i++ )
    {
        NN_LOG("  [%3d] %p\n", i, m_pLoadTable->waveFile[i] );
    }
#endif /* NW_SND_DEBUG_PRINT_ENABLE */

    return preAddress;
}

bool WaveArchiveFileReader::HasIndividualLoadTable() const
{
    if ( m_IsInitialized == false )
    {
        return 0;
    }

    const u32* signature = reinterpret_cast<const u32*>(
            ut::AddOffsetToPtr( m_pHeader, m_pHeader->GetFileBlockOffset() ) );
    if ( *signature == SIGNATURE_WARC_TABLE )
    {
        return true;
    }
    return false;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
