﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SoundDataManager.h>

#include <nw/snd/snd_DisposeCallbackManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_SoundArchiveLoader.h>
#include <nw/snd/snd_GroupFileReader.h>
#include <nw/snd/snd_WaveArchiveFile.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_WaveArchiveFileReader.h>

namespace nw {
namespace snd {


SoundDataManager::SoundDataManager()
: m_pFileTable ( NULL ),
  m_pFileManager( NULL )
{
}

SoundDataManager::~SoundDataManager()
{
}

size_t SoundDataManager::GetRequiredMemSize( const SoundArchive* arc ) const
{
    NW_ASSERT_NOT_NULL( arc );
    size_t size = 0;

    size += ut::RoundUp(
            sizeof( u32 ) +
            sizeof( FileAddress ) * arc->detail_GetFileCount(), 4 );
    return size;
}

bool SoundDataManager::Initialize( const SoundArchive* arc, void* buffer, u32 size )
{
    NW_ASSERT_NOT_NULL( arc );
    NW_ASSERT_NOT_NULL( buffer );
    NW_ASSERT_ALIGN4( buffer ); // 4 バイト境界でないと CreateFileAddressTable が失敗する
    NW_ASSERT( size >= GetRequiredMemSize( arc ) );

    void* endp = static_cast<char*>(buffer) + size;
    void* buf = buffer;

    if ( ! CreateFileAddressTable( arc, &buf, endp ) )
    {
        return false;
    }

    NW_ASSERT( static_cast<char*>(buf) - static_cast<char*>(buffer) == static_cast<s32>(GetRequiredMemSize( arc )) );
    SetSoundArchive( arc );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();

    internal::DriverCommandDisposeCallback* command =
        cmdmgr.AllocCommand<internal::DriverCommandDisposeCallback>();
    command->id = internal::DRIVER_COMMAND_REGIST_DISPOSE_CALLBACK;
    command->callback = this;
    cmdmgr.PushCommand(command);

    return true;
}

void SoundDataManager::Finalize()
{
    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();

    internal::DriverCommandDisposeCallback* command =
        cmdmgr.AllocCommand<internal::DriverCommandDisposeCallback>();
    command->id = internal::DRIVER_COMMAND_UNREGIST_DISPOSE_CALLBACK;
    command->callback = this;
    cmdmgr.PushCommand(command);
    u32 tag = cmdmgr.FlushCommand( true );
    cmdmgr.WaitCommandReply( tag );

    m_pFileManager = NULL;
    m_pFileTable = NULL;
}

bool SoundDataManager::CreateFileAddressTable(
    const SoundArchive* arc,
    void** buffer,
    void* endp
)
{
    size_t requiredSize =
        sizeof( u32 ) +
        sizeof( FileAddress ) * arc->detail_GetFileCount();

    void* ep = ut::RoundUp( ut::AddOffsetToPtr( *buffer, requiredSize ), 4 );
    if ( ut::ComparePtr( ep, endp ) > 0 )
    {
        return false;
    }
    m_pFileTable = reinterpret_cast< FileTable* >(*buffer);
    *buffer = ep;


    m_pFileTable->count = arc->detail_GetFileCount();
    for( u32 i = 0 ; i < m_pFileTable->count ; i++ )
    {
        m_pFileTable->item[ i ].address = NULL;
    }

    return true;
}


/*---------------------------------------------------------------------------*
  Name:         InvalidateData

  Description:  ロードされたデータが破棄されたときにテーブルから削除する

  Arguments:    start - 開始アドレス
                end   - 終了アドレス

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SoundDataManager::InvalidateData( const void* start, const void* end )
{
    // ファイルアドレステーブルから破棄
    if ( m_pFileTable != NULL )
    {
        for( u32 i = 0 ; i < m_pFileTable->count ; i++ )
        {
            const void* addr = m_pFileTable->item[ i ].address;
            if ( start <= addr && addr <= end )
            {
                m_pFileTable->item[ i ].address = NULL;
            }
        }
    }

    // まだロードされている「波形アーカイブテーブル」上のから、
    // 波形ファイルアドレスを削除
    const SoundArchive* arc = GetSoundArchive();
    if ( arc )
    {
        u32 waveArchiveCount = arc->GetWaveArchiveCount();
        for ( u32 i = 0; i < waveArchiveCount; i++ )
        {
            SoundArchive::WaveArchiveInfo info;
            if ( arc->ReadWaveArchiveInfo(
                        SoundArchive::GetWaveArchiveIdFromIndex( i ), &info ) )
            {
                if ( info.isLoadIndividual )
                {
                    const void* pWarcTable = GetFileAddressFromTable( info.fileId );
                    if ( pWarcTable != NULL )
                    {
                        internal::WaveArchiveFileReader reader( pWarcTable, true );
                        for ( u32 j = 0; j < info.waveCount; j++ )
                        {
                            const void* waveAddr = reader.GetWaveFile(j);
                            if ( waveAddr && ( start <= waveAddr && waveAddr <= end ) )
                            {
                                reader.SetWaveFile(j, NULL);
                            }
                        }
                    }
                }
            }
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         detail_GetFileAddress

  Description:  記憶させたファイルアドレスを取得

  Arguments:    fileId - ファイルID

  Returns:      ファイルアドレス
 *---------------------------------------------------------------------------*/
const void* SoundDataManager::detail_GetFileAddress( SoundArchive::FileId fileId ) const
{
    return GetFileAddressImpl( fileId );
}

const void* SoundDataManager::GetFileAddressImpl( SoundArchive::FileId fileId ) const
{
    // 外部のファイルマネージャに問い合わせ
    if ( m_pFileManager != NULL )
    {
        const void* addr = m_pFileManager->GetFileAddress( fileId );
        if ( addr != NULL ) return addr;
    }

    // サウンドアーカイブに問い合わせ
    {
        const void* addr = GetFileAddressFromSoundArchive( fileId );
        if ( addr != NULL ) return addr;
    }

    // ファイルアドレステーブルを参照
    {
        const void* fileData = GetFileAddressFromTable( fileId );
        if ( fileData != NULL ) return fileData;
    }
    return NULL;
}

const void* SoundDataManager::SetFileAddressToTable(
        SoundArchive::FileId fileId,
        const void* address )
{
    if ( m_pFileTable == NULL )
    {
        NW_WARNING(
            m_pFileTable != NULL,
            "Failed to SoundDataManager::SetFileAddress because file table is not allocated.\n"
        );
        return NULL;
    }

    NW_ASSERT_MAX( fileId, m_pFileTable->count-1 );

    const void* preAddress = m_pFileTable->item[ fileId ].address;
    m_pFileTable->item[ fileId ].address = address;
    return preAddress;
}


/*---------------------------------------------------------------------------*
  Name:         GetFileAddressFromTable

  Description:  記憶させたファイルアドレスを取得

  Arguments:    fileId - ファイルID

  Returns:      ファイルアドレス
 *---------------------------------------------------------------------------*/
const void* SoundDataManager::GetFileAddressFromTable( SoundArchive::FileId fileId ) const
{
    if ( m_pFileTable == NULL )
    {
        NW_WARNING(
            m_pFileTable != NULL,
            "Failed to SoundDataManager::GetFileAddress because file table is not allocated.\n"
        );
        return NULL;
    }

    if ( fileId >= m_pFileTable->count ) return NULL;

    return m_pFileTable->item[ fileId ].address;
}

/*---------------------------------------------------------------------------*
  Name:         SetFileAddress

  Description:  ファイルアドレスを記憶

  Arguments:    fileId - ファイル番号
                address - ファイルアドレス

  Returns:      セットする前に書かれていたファイルアドレス
 *---------------------------------------------------------------------------*/
const void*
SoundDataManager::SetFileAddress( SoundArchive::FileId fileId, const void* address )
{
    return SetFileAddressToTable( fileId, address );
}

// ロード済みファイルテーブルの中から、
// 引数と一致するアドレスを持つファイルのファイル ID を取得する
u32 SoundDataManager::detail_GetFileIdFromTable( const void* address ) const
{
    for ( u32 i = 0; i < m_pFileTable->count; i++ )
    {
        if ( address == m_pFileTable->item[i].address )
        {
            return i;
        }
    }
    return SoundArchive::INVALID_ID;
}

// Clear 関数との対称性のため、size 引数をつけてあるが、現状は使わない
bool SoundDataManager::SetFileAddressInGroupFile( const void* address, size_t /*size*/ )
{
    NW_ASSERT_NOT_NULL(address);
    if ( address == NULL )
    {
        return false;
    }

    // 以下の実装は、SoundArchiveLoader::PostProcessForLoadedGroupFile と似ている
    internal::GroupFileReader reader(address);
    u32 groupItemCount = reader.GetGroupItemCount();
    for ( u32 i = 0; i < groupItemCount; i++ )
    {
        internal::GroupItemLocationInfo info;
        if ( ! reader.ReadGroupItemLocationInfo( &info, i ) )
        {
            return false;
        }

        if ( info.address == NULL )
        {
            return false;
        }
        SetFileAddressToTable( info.fileId, info.address );
    }

    return true;
}

void SoundDataManager::ClearFileAddressInGroupFile( const void* address, size_t size )
{
    InvalidateSoundData(address, size);
}

void SoundDataManager::InvalidateSoundData( const void* address, size_t size )
{
    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();

    if ( cmdmgr.IsAvailable() ) {
        internal::DriverCommandInvalidateData* command =
            cmdmgr.AllocCommand<internal::DriverCommandInvalidateData>();
        command->id = internal::DRIVER_COMMAND_INVALIDATE_DATA;
        command->mem = address;
        command->size = size;
        cmdmgr.PushCommand(command);

        u32 tag = cmdmgr.FlushCommand( true );
        cmdmgr.WaitCommandReply( tag );
    }
}

} // namespace nw::snd
} // namespace nw

