﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Sound3DActor.h>

#include <nw/snd/snd_SoundArchivePlayer.h>
#include <nw/snd/snd_Sound3DManager.h>
#include <nw/snd/snd_SoundHandle.h>
#include <nw/snd/snd_Global.h>          // nw::snd::DECAY_CURVE_LOG

namespace nw {
namespace snd {

/*---------------------------------------------------------------------------*
  Name:         Sound3DActor

  Description:  コンストラクタ

  Arguments:    player - サウンドアーカイブプレイヤー
                manager - ３Ｄサウンドマネージャー

  Returns:      None.
 *---------------------------------------------------------------------------*/
Sound3DActor::Sound3DActor( SoundArchivePlayer& player, Sound3DManager& manager )
: m_p3dManager( NULL ),
  m_pArchivePlayer( NULL ),
  m_UserParam( 0 ),
  m_Position( 0.0f, 0.0f, 0.0f ),
  m_Velocity( 0.0f, 0.0f, 0.0f ),
  m_ResetPositionFlag( true ),
  m_IsInitialized( false ),
  m_IsFinalized( true )
{
    Initialize( player, manager );
}

Sound3DActor::Sound3DActor()
: m_p3dManager( NULL ),
  m_pArchivePlayer( NULL ),
  m_UserParam( 0 ),
  m_Position( 0.0f, 0.0f, 0.0f ),
  m_Velocity( 0.0f, 0.0f, 0.0f ),
  m_ResetPositionFlag( true ),
  m_IsInitialized( false ),
  m_IsFinalized( true )
{
}

/*---------------------------------------------------------------------------*
  Name:         ~Sound3DActor

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
Sound3DActor::~Sound3DActor()
{
    Finalize();
}

void Sound3DActor::Initialize( SoundArchivePlayer& player, Sound3DManager& manager )
{
    if ( m_IsInitialized ) return;

    SoundActor::Initialize( player );
    m_p3dManager = &manager;
    m_pArchivePlayer = &player;

    m_IsInitialized = true;
    m_IsFinalized = false;
}

void Sound3DActor::Finalize()
{
    if ( m_IsFinalized ) return;

    // サウンドからの Update が来ないようにする
    SoundActor::ForEachSound( ClearUpdateCallback );
    SoundActor::Finalize();

    m_IsFinalized = true;
    m_IsInitialized = false;
    m_p3dManager = NULL;
    m_pArchivePlayer = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         SetupSound [virtual]

  Description:  再生の実装関数

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                startInfo - サウンド再生パラメータ
                setupArg - セットアップ引数

  Returns:      結果コード
 *---------------------------------------------------------------------------*/
SoundStartable::StartResult Sound3DActor::SetupSound(
    SoundHandle* handle,
    u32 soundId,
    const StartInfo* startInfo,
    void* setupArg
)
{
    if ( m_IsInitialized == false )
    {
        return StartResult( StartResult::START_ERR_ACTOR_NOT_INITIALIZED );
    }

    // 初期パラメータ設定
    Sound3DParam param;
    param.position = m_Position;
    param.velocity = m_Velocity;
    param.actorUserParam = m_UserParam;
    if ( m_pArchivePlayer != NULL )
    {
        const SoundArchive& soundArchive = m_pArchivePlayer->GetSoundArchive();

        SoundArchive::Sound3DInfo p;
        if ( soundArchive.ReadSound3DInfo( soundId, &p ) )
        {
            param.ctrlFlag = p.flags;
            param.decayRatio = p.decayRatio;
            param.dopplerFactor = p.dopplerFactor;

            // decayCurve
            switch ( p.decayCurve )
            {
            case SoundArchive::Sound3DInfo::DECAY_CURVE_LOG:
                param.decayCurve = nw::snd::DECAY_CURVE_LOG;
                break;
            case SoundArchive::Sound3DInfo::DECAY_CURVE_LINEAR:
                param.decayCurve = nw::snd::DECAY_CURVE_LINEAR;
                break;
            default:
                param.decayCurve = nw::snd::DECAY_CURVE_LOG;
                break;
            }
        }
        param.soundUserParam = soundArchive.GetSoundUserParam( soundId );
    }

    internal::BasicSound::AmbientInfo argInfo = {
        m_p3dManager,
        this,
        m_p3dManager,
        &param,
        sizeof( param )
    };

    SoundStartable::StartResult result = SoundActor::detail_SetupSoundWithAmbientInfo(
        handle,
        soundId,
        startInfo,
        &argInfo,
        setupArg
    );

    if ( handle->IsAttachedSound() )
    {
        handle->detail_GetAttachedSound()->SetPanCurve( PAN_CURVE_SINCOS );
    }

    return result;
}

/*---------------------------------------------------------------------------*
  Name:         SetPosition

  Description:  アクターの位置を設定する

  Arguments:    position - 位置を表す３次元ベクトル

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Sound3DActor::SetPosition( const nw::math::VEC3& position )
{
    if ( ! m_ResetPositionFlag ) {
        nw::math::VEC3Sub( &m_Velocity, &position, &m_Position );
    }
    m_Position = position;
    m_ResetPositionFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         ResetPosition

  Description:  アクターの位置情報をリセットする

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Sound3DActor::ResetPosition()
{
    m_ResetPositionFlag = true;
    m_Position = m_Velocity = nw::math::VEC3( 0.0f, 0.0f, 0.0f );
}

/*---------------------------------------------------------------------------*
  Name:         SetVelocity

  Description:  アクターの速度を設定する

  Arguments:    velocity - 速度を表す３次元ベクトル

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Sound3DActor::SetVelocity( const nw::math::VEC3& velocity )
{
    m_Velocity = velocity;
}

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  アクターのパラメータを更新する

  Arguments:    arg - 引数
                sound - サウンドインスタンス

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Sound3DActor::detail_UpdateAmbientArg( void* arg, const internal::BasicSound* sound )
{
    (void)sound;

    Sound3DParam* param = static_cast< Sound3DParam* >( arg );
    param->position = m_Position;
    param->velocity = m_Velocity;
    param->actorUserParam = m_UserParam;
}

/*---------------------------------------------------------------------------*
  Name:         ClearUpdateCallback [static]

  Description:  アンビエント引数の更新コールバックのクリア

  Arguments:    handle - クリア対象のサウンドハンドル

  Returns:      None.
  *---------------------------------------------------------------------------*/
void Sound3DActor::ClearUpdateCallback( SoundHandle& handle )
{
    if ( handle.IsAttachedSound() )
    {
        handle.detail_GetAttachedSound()->ClearAmbientArgUpdateCallback();
    }
}

} // namespace nw::snd
} // namespace nw

