﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SequenceSoundFileReader.h>

namespace nw {
namespace snd {
namespace internal {

namespace
{

const u32 SIGNATURE_DATA_BLOCK_SEQ  = NW_UT_MAKE_SIGWORD( 'D', 'A', 'T', 'A' );
const u32 SIGNATURE_LABEL_BLOCK_SEQ = NW_UT_MAKE_SIGWORD( 'L', 'A', 'B', 'L' );

const u32 SUPPORTED_FILE_VERSION_SEQ = 0x00010000;
const u32 CURRENT_FILE_VERSION_SEQ   = 0x00020000;

bool IsValidFileHeaderSeq( const void* sequenceFile )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader* header =
        reinterpret_cast<const ut::BinaryFileHeader*>( sequenceFile );
#else
    const BinaryFileHeader* header =
        reinterpret_cast<const BinaryFileHeader*>( sequenceFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header->signature == SequenceSoundFileReader::SIGNATURE_FILE,
            "invalid file signature. sequence file is not available." );
    if ( header->signature != SequenceSoundFileReader::SIGNATURE_FILE )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_SEQ <= header->version) &&
                                   (header->version <= CURRENT_FILE_VERSION_SEQ) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfseq file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_SEQ, CURRENT_FILE_VERSION_SEQ, header->version
    );
    return isSupportedVersion;
}

} // anonymous namespace


SequenceSoundFileReader::SequenceSoundFileReader( const void* sequenceFile )
: m_pHeader( NULL ),
  m_pDataBlockBody( NULL ),
  m_pLabelBlockBody( NULL )
{
    NW_NULL_ASSERT( sequenceFile );

    if ( ! IsValidFileHeaderSeq( sequenceFile ) ) return;

    m_pHeader = reinterpret_cast<const SequenceSoundFile::FileHeader*>( sequenceFile );

    const SequenceSoundFile::DataBlock* dataBlock = m_pHeader->GetDataBlock();
    NW_NULL_ASSERT( dataBlock );
    NW_ASSERT( dataBlock->header.kind == SIGNATURE_DATA_BLOCK_SEQ );
    if ( dataBlock->header.kind != SIGNATURE_DATA_BLOCK_SEQ )
    {
        return;
    }

    const SequenceSoundFile::LabelBlock* labelBlock = m_pHeader->GetLabelBlock();
    NW_NULL_ASSERT( labelBlock );
    NW_ASSERT( labelBlock->header.kind == SIGNATURE_LABEL_BLOCK_SEQ );
    if ( labelBlock->header.kind != SIGNATURE_LABEL_BLOCK_SEQ )
    {
        return;
    }

    m_pDataBlockBody = &dataBlock->body;
    m_pLabelBlockBody = &labelBlock->body;
}

const void* SequenceSoundFileReader::GetSequenceData() const
{
    NW_NULL_ASSERT( m_pDataBlockBody );
    return m_pDataBlockBody->GetSequenceData();
}

bool SequenceSoundFileReader::GetOffsetByLabel( const char* label, u32* offsetPtr ) const
{
    NW_NULL_ASSERT( m_pLabelBlockBody );
    return m_pLabelBlockBody->GetOffsetByLabel( label, offsetPtr );
}

const char* SequenceSoundFileReader::GetLabelByOffset( u32 offset ) const
{
    NW_NULL_ASSERT( m_pLabelBlockBody );
    return m_pLabelBlockBody->GetLabelByOffset( offset );
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

