﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_DisposeCallbackManager.h>
#include <nw/snd/snd_MultiVoice.h>
#include <nw/snd/snd_SoundThread.h>

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/*---------------------------------------------------------------------------*
    Name:           GetInstance

    Description:    シングルトンのインスタンスを取得します

    Arguments:      なし

    Returns:        なし
 *---------------------------------------------------------------------------*/
MultiVoiceManager& MultiVoiceManager::GetInstance()
{
    static MultiVoiceManager instance;
    return instance;
}


MultiVoiceManager::MultiVoiceManager()
: m_Initialized( false )
{
}

size_t MultiVoiceManager::GetRequiredMemSize( int voiceCount )
{
    return sizeof( MultiVoice ) * voiceCount;
}

void MultiVoiceManager::Initialize( void* mem, size_t memSize )
{
    if ( m_Initialized ) return;

    int voiceCount = memSize / sizeof( MultiVoice );

    u8* ptr = reinterpret_cast<u8*>( mem );
    // ボイスをリスト管理
    for ( int i = 0; i < voiceCount; i++ )
    {
        m_FreeVoiceList.PushBack( new( ptr )MultiVoice() );
        ptr += sizeof( MultiVoice );
    }
    NW_ASSERT( ptr <= reinterpret_cast<u8*>( mem ) + memSize );

    m_Initialized = true;
}

void MultiVoiceManager::Finalize()
{
    if ( !m_Initialized ) return;

    StopAllVoices();

    // ボイスをリストから削除
    while ( ! m_FreeVoiceList.IsEmpty() )
    {
        MultiVoice& voice = m_FreeVoiceList.GetFront();
        m_FreeVoiceList.PopFront();
        voice.~MultiVoice(); // デストラクタ呼び出し
    }

    m_Initialized = false;
}

void MultiVoiceManager::StopAllVoices()
{
    // 再生中のボイスを全てFree
    while ( ! m_PrioVoiceList.IsEmpty() )
    {
        MultiVoice& voice = m_PrioVoiceList.GetFront();
        voice.Stop();
        if ( voice.m_Callback != NULL )
        {
            voice.m_Callback(
                &voice,
                MultiVoice::CALLBACK_STATUS_CANCEL,
                voice.m_pCallbackData
            );
        }
        voice.Free();
    }
}

MultiVoice* MultiVoiceManager::AllocVoice(
    int voiceChannelCount,
    int priority,
    MultiVoice::VoiceCallback callback,
    void* callbackData,
    VoiceRendererType mode
)
{
    // ボイスが不足している場合にプライオリティが低いボイスを停止させる
    if ( m_FreeVoiceList.IsEmpty() )
    {
        if ( DropLowestPriorityVoice( priority ) == 0 ) return NULL;
    }

    // フリーリストから空きボイスを取得
    MultiVoice& voice = m_FreeVoiceList.GetFront();

    if ( ! voice.Alloc( voiceChannelCount, priority, callback, callbackData, mode ) )
    {
        return NULL;
    }

    // プライオリティを設定してボイスリストに追加
    voice.m_Priority = static_cast<u8>( priority );
    m_FreeVoiceList.Erase( &voice );
    AppendVoiceList( &voice );

    return &voice;
}

void MultiVoiceManager::FreeVoice( MultiVoice* voice )
{
    NW_NULL_ASSERT( voice );

    RemoveVoiceList( voice );
}

void MultiVoiceManager::UpdateAllVoiceStatus()
{
    // 波形の終端 or DSP 処理落ち防止用のボイスドロップ結果を反映
    for ( VoiceList::Iterator itr = m_PrioVoiceList.GetBeginIter();
          itr != m_PrioVoiceList.GetEndIter();
        )
    {
        VoiceList::Iterator curItr = itr++;
        curItr->UpdateVoiceStatus();
    }
}

void MultiVoiceManager::UpdateAllVoices()
{
    // すべてのボイスのパラメータ計算
    for ( VoiceList::Iterator itr = m_PrioVoiceList.GetBeginIter();
          itr != m_PrioVoiceList.GetEndIter();
    )
    {
        VoiceList::Iterator curItr = itr++;
        curItr->Calc();
    }

    // すべてのボイスを更新
    for ( VoiceList::Iterator itr = m_PrioVoiceList.GetBeginIter();
          itr != m_PrioVoiceList.GetEndIter();
    )
    {
        VoiceList::Iterator curItr = itr++;
        curItr->Update();
    }
}

void MultiVoiceManager::AppendVoiceList( MultiVoice* voice )
{
    // プライオリティリストに挿入、同プライオリティは後着の音を後ろに繋ぐ
    // 逆順に探索したほうが、同プライオリティの場合に速い。
    VoiceList::ReverseIterator itr = m_PrioVoiceList.GetBeginReverseIter();
    while ( itr != m_PrioVoiceList.GetEndReverseIter() )
    {
        if ( itr->m_Priority <= voice->m_Priority )
        {
            break;
        }
        (void)++itr;
    }

    m_PrioVoiceList.Insert( itr.GetBase(), voice );
}
void MultiVoiceManager::RemoveVoiceList( MultiVoice* voice )
{
    m_PrioVoiceList.Erase( voice );
    m_FreeVoiceList.PushBack( voice );
}

void MultiVoiceManager::ChangeVoicePriority( MultiVoice* voice )
{
    m_PrioVoiceList.Erase( voice );
    AppendVoiceList( voice );
}

void MultiVoiceManager::UpdateAllVoicesSync( u32 syncFlag )
{
    for ( VoiceList::Iterator itr = m_PrioVoiceList.GetBeginIter();
          itr != m_PrioVoiceList.GetEndIter();
        )
    {
        VoiceList::Iterator curItr = itr++;
        if ( curItr->m_IsActive ) curItr->m_SyncFlag |= syncFlag;
    }
}

int MultiVoiceManager::DropLowestPriorityVoice( int priority )
{
    if ( m_PrioVoiceList.IsEmpty() ) return 0;

    MultiVoice* dropVoice = &m_PrioVoiceList.GetFront();
    if ( dropVoice->GetPriority() > priority ) return 0;

    // もっとも優先度の低いボイスをDrop
    int dropCount = dropVoice->GetSdkVoiceCount();

    MultiVoice::VoiceCallback callbackFunc = dropVoice->m_Callback;
    void* callbackData = dropVoice->m_pCallbackData;

    dropVoice->Stop();
    dropVoice->Free();

    if ( callbackFunc != NULL )
    {
        callbackFunc(
            dropVoice,
            MultiVoice::CALLBACK_STATUS_DROP_VOICE,
            callbackData
        );
    }

    return dropCount;
}

int MultiVoiceManager::GetVoiceCount() const
{
    SoundThreadLock lock;

    int voiceCount = 0;
    for (
        VoiceList::ConstIterator itr = m_PrioVoiceList.GetBeginIter();
        itr != m_PrioVoiceList.GetEndIter();
        (void)++itr
    )
    {
        voiceCount += itr->GetSdkVoiceCount();
    }

    return voiceCount;
}

unsigned long MultiVoiceManager::GetActiveCount() const
{
    SoundThreadLock lock;

    return m_PrioVoiceList.GetSize();
}

unsigned long MultiVoiceManager::GetFreeCount() const
{
    SoundThreadLock lock;

    return m_FreeVoiceList.GetSize();
}

const MultiVoiceManager::VoiceList& MultiVoiceManager::GetVoiceList() const
{
    return m_PrioVoiceList;
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

