﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_CachedFsFileStream.h>
#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace snd {
namespace internal {

CachedFsFileStream::CachedFsFileStream()
: m_pCacheBuffer(NULL)
, m_CacheSize(0)
, m_CachePos(-1)
, m_VirtualFilePos(-1)
{}

void CachedFsFileStream::SetCacheBuffer(void* bufferAddress, size_t size)
{
    m_pCacheBuffer = bufferAddress;
    m_CacheSize = size;
    m_CachePos = -1;
    m_VirtualFilePos = -1;
}

void CachedFsFileStream::ClearCacheBuffer()
{
    m_pCacheBuffer = NULL;
    m_CacheSize = 0;
    m_CachePos = -1;
    m_VirtualFilePos = -1;
}

// Read 内で呼ばれることを想定
bool CachedFsFileStream::IsCacheAvailable() const
{
    if (m_pCacheBuffer == NULL)
    {
        return false;
    }
    if (m_CacheSize == 0)
    {
        return false;
    }
    return true;
}

bool CachedFsFileStream::IsInCache(u32 length) const
{
    if (!IsCacheAvailable())
    {
        return false;
    }
    if (m_CachePos < 0)
    {
        return false;
    }

    if (m_VirtualFilePos < m_CachePos)
    {
        return false;
    }
    if ((m_CachePos + m_CacheSize) < (m_VirtualFilePos + length))
    {
        return false;
    }

    return true;
}

bool CachedFsFileStream::Seek(s32 offset, u32 origin)
{
    if (IsCacheAvailable())
    {
        switch (origin)
        {
        case ut::FILE_STREAM_SEEK_BEGIN:
            m_VirtualFilePos = offset;
            break;
        case ut::FILE_STREAM_SEEK_CURRENT:
            m_VirtualFilePos += offset;
            break;
        case ut::FILE_STREAM_SEEK_END:
            m_VirtualFilePos = GetSize() - offset;
            break;
        default:
            NW_ASSERTMSG( false, "Unsupported Seek origin" );
            return false;
        }
        return true;
    }
    else
    {
        return FsFileStream::Seek(offset, origin);
    }
}

u32 CachedFsFileStream::Tell() const
{
    if (IsCacheAvailable())
    {
        return m_VirtualFilePos;
    }
    else
    {
        return FsFileStream::Tell();
    }
}

s32 CachedFsFileStream::Read(void* buf, u32 length)
{
    NW_ASSERT_NOT_NULL(buf);

    // キャッシュが利用できないとき
    if (!IsCacheAvailable())
    {
        // NW_LOG("[%s] Cache Unavailable : begin(%08x) length(%04x)\n", __FUNCTION__, Tell(), length);
        return FsFileStream::Read(buf, length);
    }

    // キャッシュ内
    if (IsInCache(length))
    {
        // NW_LOG("[%s] Cache Hit  : begin(%08x) length(%04x)\n", __FUNCTION__, m_VirtualFilePos, length);
        s32 diff = m_VirtualFilePos - m_CachePos;
        std::memcpy(buf, ut::AddOffsetToPtr(m_pCacheBuffer, diff), length);
        m_VirtualFilePos += length;
        return static_cast<s32>(length);
    }
    // キャッシュ外
    else
    {
        // NW_LOG("[%s] Cache Miss : begin(%08x) length(%04x)\n", __FUNCTION__, m_VirtualFilePos, length);

        // キャッシュ開始位置を更新
        if (m_VirtualFilePos != m_CachePos)
        {
            FsFileStream::Seek(m_VirtualFilePos, nw::ut::FILE_STREAM_SEEK_BEGIN);
        }
        m_CachePos = Tell();

        // キャッシュに収まる場合
        if (length <= m_CacheSize)
        {
            s32 readSize = FsFileStream::Read(m_pCacheBuffer, m_CacheSize);
            if (readSize < 0)
            {
                // キャッシュも破壊されることになる
                m_CachePos = -1;
                return readSize;
            }
            // キャッシュ → buf にコピー
            std::memcpy(buf, m_pCacheBuffer, length);
            m_VirtualFilePos += length;
            return length;
        }
        // キャッシュに収まらない場合
        else
        {
            s32 readSize = FsFileStream::Read(buf, length);
            if (readSize < 0)
            {
                m_CachePos = -1;
                return readSize;
            }
            // buf -> キャッシュにコピー
            std::memcpy(m_pCacheBuffer, buf, m_CacheSize);
            m_VirtualFilePos += length;
            return length;
        }
    }
}

}   /* namespace internal */
}   /* namespace snd */
}   /* namespace nw */

