﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Bank.h>
#include <nw/snd/snd_NoteOnCallback.h>      // NoteOnInfo
#include <nw/snd/snd_BankFileReader.h>
#include <nw/snd/snd_WaveArchiveFileReader.h>
#include <nw/snd/snd_WaveFileReader.h>
#include <nw/snd/snd_Util.h>                // GetWaveFile

namespace
{
// ベロシティの最大値の逆数です。割り算の掛け算化による高速化が狙いです。
const f32 VELOCITY_MAX_R = 1.0f / 127.0f;
// インスト音量の基準値の逆数です。割り算の掛け算化による高速化が狙いです。
const f32 INSTRUMENT_VOLUME_CRITERION_R = 1.0f / 127.0f;
} // anonymous namespace

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        public functions
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         Bank

  Description:  コンストラクタ

  Arguments:    bankData - バンクデータへの参照

  Returns:      None.
 *---------------------------------------------------------------------------*/
Bank::Bank()
{
}

/*---------------------------------------------------------------------------*
  Name:         ~Bank

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
Bank::~Bank()
{
}

Channel* Bank::NoteOn(
    const BankFileReader& bankReader,
    const WaveArchiveFileReader& warcReader,
    const NoteOnInfo& noteOnInfo
) const
{
    // VelocityRegionInfo の取得
    VelocityRegionInfo regionInfo;
    {
        if ( ! bankReader.ReadVelocityRegionInfo(
                &regionInfo,
                noteOnInfo.prgNo,
                noteOnInfo.key,
                noteOnInfo.velocity ) )
        {
            return NULL;
        }
    }

    // regnioInfo に応じた波形を取得する
    // (regionInfo.waveArchiveId に該当した波形アーカイブが、
    //  warcReader に登録されて引数で渡されてくる)
    const void* waveFile = warcReader.GetWaveFile( regionInfo.waveIndex );
    NW_NULL_ASSERT(waveFile);

    // 波形の詳細を取得、Channel 確保
    WaveInfo waveInfo;
    {
        WaveFileReader reader( waveFile );
        if ( ! reader.ReadWaveInfo( &waveInfo ) )
        {
            return NULL;
        }
    }

    Channel* pChannel = Channel::AllocChannel(
        ut::Min( static_cast<int>( waveInfo.channelCount ), 2 ),
        noteOnInfo.priority,
        noteOnInfo.channelCallback,
        noteOnInfo.channelCallbackData,
        noteOnInfo.rendererType
    );
    if ( pChannel == NULL )
    {
        return NULL;
    }

    // 初期パラメータ設定
    pChannel->SetKey( static_cast<u8>(noteOnInfo.key), regionInfo.originalKey );
    pChannel->SetVelocity( CalcChannelVelocityVolume(static_cast<u8>(noteOnInfo.velocity))  );
    pChannel->SetInstrumentVolume( regionInfo.volume * INSTRUMENT_VOLUME_CRITERION_R );
    pChannel->SetTune( regionInfo.pitch );

    pChannel->SetAttack( regionInfo.adshrCurve.attack );
    pChannel->SetHold( regionInfo.adshrCurve.hold );
    pChannel->SetDecay( regionInfo.adshrCurve.decay );
    pChannel->SetSustain( regionInfo.adshrCurve.sustain );
    pChannel->SetRelease( regionInfo.adshrCurve.release );

    float initPan = static_cast<f32>( noteOnInfo.initPan + regionInfo.pan - 64 ) / 63.0f;
    pChannel->SetInitPan( initPan );
//    pChannel->SetInitSurroundPan( 0.0f );

    pChannel->SetKeyGroupId( regionInfo.keyGroup );
    pChannel->SetIsIgnoreNoteOff( regionInfo.isIgnoreNoteOff );
    pChannel->SetInterpolationType( regionInfo.interpolationType );

    pChannel->Start( waveInfo, noteOnInfo.length, 0 );

    return pChannel;
}

f32 Bank::CalcChannelVelocityVolume(u8 velocity)
{
    return static_cast<f32>(velocity) * VELOCITY_MAX_R;
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

