﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_StreamCache.h>

#include <nw/ut/ut_Inlines.h>

#if !defined(NW_RELEASE)
//#define NW_SND_FND_STREAMIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
StreamCache::StreamCache() :
m_Stream(NULL),
m_CurrentPosition(0),
m_CacheBuffer(NULL),
m_CacheBufferLength(0),
m_CachePosition(INVALID_POSITION),
m_CachedLength(0),
m_CacheState(CACHE_NONE),
m_Padding1(0),
m_Padding2(0)
{

}

//---------------------------------------------------------------------------
StreamCache::StreamCache(Stream* sourceStream, void* buffer, u32 length) :
m_Stream(sourceStream),
m_CurrentPosition(0),
m_CacheBuffer(buffer),
m_CacheBufferLength(length),
m_CachePosition(INVALID_POSITION),
m_CachedLength(0),
m_CacheState(CACHE_NONE),
m_Padding1(0),
m_Padding2(0)
{
    NW_ASSERT_NOT_NULL(sourceStream);
}

//---------------------------------------------------------------------------
void
StreamCache::Initialize(Stream* sourceStream, void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(sourceStream);
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERT(length > 0);

    m_Stream = sourceStream;
    m_CurrentPosition = 0;
    m_CacheBuffer = buffer;
    m_CacheBufferLength = length;
    m_CachePosition = INVALID_POSITION;
    m_CachedLength = 0;
    m_CacheState = CACHE_NONE;
}

//---------------------------------------------------------------------------
void
StreamCache::Finalize()
{
    m_Stream = NULL;
    m_CurrentPosition = 0;
    m_CacheBuffer = NULL;
    m_CacheBufferLength = 0;
    m_CachePosition = INVALID_POSITION;
    m_CachedLength = 0;
    m_CacheState = CACHE_NONE;
}

//---------------------------------------------------------------------------
u32
StreamCache::Read(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    if(!IsInitialized())
    {
        NW_ASSERTMSG(false, "StreamCache must be initialized.\n");
        return m_Stream->Read(buf, length, result);
    }

    // 読み込みにキャッシュを利用するので、
    // 書き込みキャッシュがあれば、フラッシュしておきます。
    FndResult flushResult = FlushWriteCache();

    if(flushResult.IsFailed())
    {
        if(result != NULL)
        {
            *result = flushResult;
        }

        return 0;
    }

    // キャッシュから読み込む
    u32 cacheHitLength = GetReadCacheHitLength(length);

    if (cacheHitLength > 0)
    {
#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] cache hit  : cache readBegin=%08x, length=%d.\n",
            __FUNCTION__, this, m_CurrentPosition, cacheHitLength);
#endif
        std::memcpy(
            buf,
            ut::AddOffsetToPtr(m_CacheBuffer, m_CurrentPosition - m_CachePosition),
            cacheHitLength);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x.\n",
            __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + cacheHitLength);
#endif

        m_CurrentPosition += cacheHitLength;
    }

    // キャッシュから読みきれた場合は、終了
    if(cacheHitLength >= length)
    {
        return length;
    }

    u32 restLength = length - cacheHitLength;

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] cache miss : readBegin=0x%08x, restLength=%d\n",
        __FUNCTION__, this, m_CurrentPosition, restLength);
#endif

    // StreamCache の現在位置と対象 Stream の現在位置が違う場合はシークする
    FndResult syncResult = SyncStreamCurrentPosition(m_CurrentPosition);
    if(syncResult.IsFailed())
    {
        if(result != NULL)
        {
            *result = syncResult;
        }

        return 0;
    }

    u32 readLength = 0;

    // キャッシュバッファに収まる場合、キャッシュしてバッファにコピー
    if (restLength <= m_CacheBufferLength)
    {
        FndResult readResult;
        readLength = m_Stream->Read(m_CacheBuffer, m_CacheBufferLength, &readResult);

        if(result != NULL)
        {
            *result = readResult;
        }

        if (readResult.IsFailed())
        {
            ClearCache();
            return cacheHitLength;
        }

        m_CacheState = CACHE_FOR_READ;
        m_CachePosition = m_CurrentPosition;
        m_CachedLength = readLength;

        std::memcpy(
            ut::AddOffsetToPtr(buf, cacheHitLength),
            m_CacheBuffer,
            ut::Min(restLength, readLength));

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] cache : begin=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }
    // キャッシュに収まらない場合、直接バッファに読み込んでから、可能な限りキャッシュ
    else
    {
        FndResult readResult;
        readLength = m_Stream->Read(buf, restLength, &readResult);

        if(result != NULL)
        {
            *result = readResult;
        }

        if (readResult.IsFailed())
        {
            ClearCache();
            return cacheHitLength;
        }

        m_CachedLength = ut::Min(readLength, m_CacheBufferLength);
        m_CachePosition = m_CurrentPosition - m_CachedLength;

        // s16 の範囲を超えるキャッシュサイズは考慮していません。
        std::memcpy(
            m_CacheBuffer,
            ut::AddOffsetToPtr(buf, -static_cast<s16>(m_CachedLength)),
            m_CacheBufferLength);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] cache parts : begin=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] current pos : 0x%08x -> 0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + restLength);
#endif

    m_CurrentPosition += restLength;

    if(cacheHitLength + readLength < length)
    {
        return cacheHitLength + readLength;
    }

    // m_Stream をキャッシュし切った場合も、
    // 残りキャッシュがあれば、TRUE を返す。
    if(result != NULL)
    {
        *result = FndResult(SNDFND_RESULT_TRUE);
    }

    return length;
}

//---------------------------------------------------------------------------
u32
StreamCache::Write(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    if(!IsInitialized())
    {
        NW_ASSERTMSG(false, "StreamCache must be initialized.\n");
        return m_Stream->Write(buf, length, result);
    }

    // キャッシュバッファサイズを超える場合は、直接書き込みます。
    // キャッシュに書き込みデータがあれば先に処理します。
    if(length > m_CacheBufferLength)
    {
        FndResult flushResult = FlushWriteCache();

        if(flushResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = flushResult;
            }

            return 0;
        }

        return m_Stream->Write(buf, length, result);
    }

    // キャッシュバッファにコピーし書き込みを遅延します。
    // キャッシュの空き領域に収まらない場合は、先にフラッシュします。
    if(GetWritableCacheLength(length) < length)
    {
        FndResult flushResult = FlushWriteCache();

        if(flushResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = flushResult;
            }

            return 0;
        }
    }

    std::memcpy(
        ut::AddOffsetToPtr(m_CacheBuffer, m_CachedLength),
        buf,
        length);

    if(m_CacheState == CACHE_FOR_WRITE)
    {
        m_CachedLength += length;

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] add write cache : cachePos=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }
    else
    {
        NW_ASSERT(m_CachedLength == 0);

        m_CacheState = CACHE_FOR_WRITE;
        m_CachePosition = m_CurrentPosition;
        m_CachedLength = length;

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] new write cache : cachePos=0x%08x, length=%d.\n",
            __FUNCTION__, this, m_CachePosition, m_CachedLength);
#endif
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] current pos : 0x%08x -> 0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + length);
#endif

    m_CurrentPosition += length;

    // キャッシュバッファが満タンならフラッシュ
    if(m_CachedLength == m_CacheBufferLength)
    {
        FndResult flushResult = FlushWriteCache();

        if(flushResult.IsFailed())
        {
            if(result != NULL)
            {
                *result = flushResult;
            }

            return length;
        }
    }

    if(result != NULL)
    {
        *result = FndResult(SNDFND_RESULT_TRUE);
    }

    return length;
}

//---------------------------------------------------------------------------
FndResult
StreamCache::Seek(s32 offset, Stream::SeekOrigin origin)
{
    if(!IsInitialized())
    {
        NW_ASSERTMSG(false, "StreamCache must be initialized.\n");
        return m_Stream->Seek(offset, origin);
    }

    if(!m_Stream->CanSeek())
    {
        return FndResult(SNDFND_RESULT_FAILED);
    }

    // ファイルサイズを超えてシークできないようにしている
    switch (origin)
    {
    case Stream::SEEK_ORIGIN_BEGIN:
        if(offset < 0 || m_Stream->GetSize() <= static_cast<u32>(offset))
        {
            return FndResult(SNDFND_RESULT_FAILED);
        }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x (SEEK_ORIGIN_BEGIN).\n",
            __FUNCTION__, this, m_CurrentPosition, offset);
#endif

        m_CurrentPosition = offset;
        break;

    case Stream::SEEK_ORIGIN_END:
        if(m_Stream->GetSize() < static_cast<u32>(offset))
        {
            return FndResult(SNDFND_RESULT_FAILED);
        }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x (SEEK_ORIGIN_END).\n",
            __FUNCTION__, this, m_CurrentPosition, m_Stream->GetSize() - offset);
#endif

        m_CurrentPosition = m_Stream->GetSize() - offset;
        break;

    case Stream::SEEK_ORIGIN_CURRENT:
        if(m_Stream->GetSize() <= m_CurrentPosition + offset)
        {
            return FndResult(SNDFND_RESULT_FAILED);
        }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
        NW_LOG(
            "[%s][%08x] current pos : 0x%08x -> 0x%08x (SEEK_ORIGIN_CURRENT).\n",
            __FUNCTION__, this, m_CurrentPosition, m_CurrentPosition + offset);
#endif

        m_CurrentPosition += offset;
        break;

    default:
        NW_ASSERTMSG(false, "invalid seek origin.\n");
        return FndResult(SNDFND_RESULT_FAILED);
    }

    return FndResult(SNDFND_RESULT_TRUE);
}

//---------------------------------------------------------------------------
void
StreamCache::ClearCache()
{
    FlushWriteCache();

    m_CachePosition = INVALID_POSITION;
    m_CachedLength = 0;
}

//---------------------------------------------------------------------------
u32
StreamCache::GetReadCacheHitLength(u32 length) const
{
    if (!IsInitialized())
    {
        return 0;
    }

    if (m_CacheState != CACHE_FOR_READ)
    {
        return 0;
    }

    if (m_CachePosition == INVALID_POSITION)
    {
        return 0;
    }

    if (m_CurrentPosition < m_CachePosition)
    {
        return 0;
    }

    u32 offset = m_CurrentPosition - m_CachePosition;

    if (m_CachedLength < offset + length)
    {
        return m_CachedLength > offset ? m_CachedLength - offset : 0;
    }

    return length;
}

//---------------------------------------------------------------------------
u32
StreamCache::GetWritableCacheLength(u32 length) const
{
    if (!IsInitialized())
    {
        return 0;
    }

    // 読み込みにキャッシュを利用している場合、
    // キャッシュバッファ全体を利用可能とします。
    if (m_CacheState == CACHE_FOR_READ)
    {
        return ut::Min(length, m_CacheBufferLength);
    }

    return ut::Min(length, m_CacheBufferLength - m_CachedLength);
}

//---------------------------------------------------------------------------
FndResult
StreamCache::SyncStreamCurrentPosition(u32 position)
{
    if(position == m_Stream->GetCurrentPosition())
    {
        return FndResult(SNDFND_RESULT_FALSE);
    }

    // 現在位置がずれるのは StreamCache::Seek() した場合のみを想定
    // シーク不可能な場合は、内部エラー扱いとし、0を返す
    if(!m_Stream->CanSeek())
    {
        return FndResult(SNDFND_RESULT_FAILED);
    }

    FndResult result = m_Stream->Seek(position, Stream::SEEK_ORIGIN_BEGIN);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] seek : curPos=0x%08x, cachePos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CachePosition);
#endif

    return result;
}

//---------------------------------------------------------------------------
FndResult
StreamCache::FlushWriteCache()
{
    if(m_CachedLength == 0 || m_CacheState != CACHE_FOR_WRITE)
    {
        return FndResult(SNDFND_RESULT_FALSE);
    }

    // StreamCache の現在位置と対象 Stream の現在位置が違う場合はシークする
    FndResult result = SyncStreamCurrentPosition(m_CachePosition);
    if(result.IsFailed())
    {
        return result;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] flush write cache : pos=0x%08x, length=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, m_CachedLength);
#endif

    result = FndResult(SNDFND_RESULT_TRUE);
    m_Stream->Write(m_CacheBuffer, m_CachedLength, &result);

    m_CacheState = CACHE_NONE;
    m_CachePosition = INVALID_POSITION;
    m_CachedLength = 0;

    return result;
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
