﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_FileStreamImpl.h>

#if !defined(NW_RELEASE)
//#define NW_SND_FND_STREAMIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {

//---------------------------------------------------------------------------
u32
GetSeekPosition(FileStreamImpl& target, s32 offset, FileStreamImpl::SeekOrigin origin)
{
    u32 result = 0;

    switch(origin)
    {
    case FileStreamImpl::SEEK_ORIGIN_BEGIN:
        result = offset <= 0 ? 0 : offset;
        break;

    case FileStreamImpl::SEEK_ORIGIN_END:
        result = offset >= 0 ? target.GetSize() : target.GetSize() - offset;
        break;

    case FileStreamImpl::SEEK_ORIGIN_CURRENT:
        if(offset > 0)
        {
            result = target.GetCurrentPosition() + static_cast<u32>(offset);

            u32 size = target.GetSize();

            if(result > size)
            {
                result = size;
            }
        }
        else if(offset < 0)
        {
            if(target.GetCurrentPosition() > static_cast<u32>(-offset))
            {
                result = target.GetCurrentPosition() - static_cast<u32>(-offset);
            }
            else
            {
                result = 0;
            }
        }
        else
        {
            result = target.GetCurrentPosition();
        }
        break;

    default:
        NW_FATAL_ERROR("invalid seek origin.\n");
        break;
    }

    return result;
}

}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl() :
m_FileHandle(),
m_IsOpened(false),
m_CurrentPosition(0)
{
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
nn::Result
FileStreamImpl::Open(const char* filePath, nn::fs::OpenMode openMode)
{
    NW_ASSERTMSG(!IsOpened(), "FileStreamImpl is already opened.\n");

    m_CurrentPosition = 0;
    m_IsOpened = true;

    return nn::fs::OpenFile(&m_FileHandle, filePath, openMode);
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Close()
{
    if(!IsOpened())
    {
        return;
    }

    m_CurrentPosition = 0;
    m_IsOpened = false;

    nn::fs::CloseFile(m_FileHandle);
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::IsOpened() const
{
    return m_IsOpened;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::GetSize() const
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    int64_t fileSize;
    nn::Result result = nn::fs::GetFileSize(&fileSize, m_FileHandle);

    return static_cast<u32>(fileSize);
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::ReadDirect(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    size_t readFileLength = 0;
    nn::Result nnResult = nn::fs::ReadFile(&readFileLength, m_FileHandle, m_CurrentPosition, buf, length);

    FndResult readResult(SNDFND_RESULT_TRUE);
    if(nnResult.IsFailure())
    {
        if (nn::fs::ResultOutOfRange::Includes(nnResult))
        {
            readResult = FndResult(SNDFND_RESULT_IO_INVALID_ACCESS);
        }
        else
        {
            readResult = FndResult(SNDFND_RESULT_IO_ERROR);
        }

        readResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        readResult = length == readFileLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += readFileLength;
    }

    if(result != NULL)
    {
        *result = readResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] FSReadFile : curPos=%08x, buf=0x%08x, count=%d, result=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length, readFileLength);
#endif

    return readFileLength;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::WriteDirect(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    FndResult writeResult(SNDFND_RESULT_TRUE);

    nn::fs::WriteOption option;
    nn::Result nnResult = nn::fs::WriteFile(m_FileHandle, m_CurrentPosition, buf, length, option);

    if(nnResult.IsFailure())
    {
        writeResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        m_CurrentPosition += length;
    }

    if(result != NULL)
    {
        *result = writeResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] FSWriteFile : curPos=%08x, buf=0x%08x, count=%d, result=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length, length);
#endif

    return length;
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::SeekDirect(s32 offset, SeekOrigin origin)
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    u32 seekPosition = GetSeekPosition(*this, offset, origin);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] try FSSetPosFile : curPos=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, seekPosition);
#endif

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_ASSERTMSG(
        seekPosition < GetSize(),
        "[%s][%08x] seek to out of range : curPos=0x%08x, size=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, GetSize(), seekPosition);
#endif

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] FSSetPosFile : curPos=%08x.\n",
        __FUNCTION__, this, m_CurrentPosition);
#endif

    m_CurrentPosition = seekPosition;

    return FndResult(SNDFND_RESULT_TRUE);
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
