﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_HioStream.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>

#if !defined(NW_RELEASE)
//#define NW_SND_FND_HIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
void
HioStream::Initialize(void* workBuffer, u32 workBufferLength)
{
    NW_ASSERT(workBufferLength > nn::hio::WORKMEMORY_SIZE);

    snd::internal::fnd::FrameHeap allocator;
    allocator.Initialize(workBuffer, workBufferLength);

    m_HioWorkBuffer = allocator.Alloc(nn::hio::WORKMEMORY_SIZE);
    m_WorkBufferLength = workBufferLength - nn::hio::WORKMEMORY_SIZE;
    m_WorkBuffer = allocator.Alloc(m_WorkBufferLength);

    m_IsAvailable = false;
}

//---------------------------------------------------------------------------
void
HioStream::Finalize()
{
    m_IsAvailable = false;

    m_WorkBuffer = NULL;
    m_WorkBufferLength = 0;
    m_HioWorkBuffer = NULL;
}

//---------------------------------------------------------------------------
void
HioStream::OpenImpl(ChannelType channel)
{
    nn::Result ret = nn::hio::GetHioDaemonStatus( &m_IsRunning );
    if(ret.IsSuccess() && m_IsRunning == true)
    {
        nn::Result result = m_Stream.Open(static_cast<s32>(channel), m_HioWorkBuffer);
        bool isSuccess = result.IsSuccess();
        if (isSuccess)
        {
            m_IsAvailable = true;
            m_IsOpened = true;
        }
    }

#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] Open : m_IsOpened=%d\n",
        __FUNCTION__, this, m_IsOpened);
#endif
}

//---------------------------------------------------------------------------
void
HioStream::CloseImpl()
{
    Disconnect();
    if( m_IsOpened )
    {
        nn::Result result = m_Stream.Close();
        NW_ASSERT(result.IsSuccess());
        m_IsOpened = false;
        m_IsAvailable = false;
    }

#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] Close : m_IsOpened=%d.\n",
        __FUNCTION__, this, m_IsOpened);
#endif
}

//---------------------------------------------------------------------------
s32
HioStream::WriteImpl(const void* buffer, u32 length)
{
    if(!IsAvailable())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    size_t size = 0;

    nn::Result result = m_Stream.Write(&size, buffer, length);
    if (result.IsFailure())
    {
        return -1;
    }

    return static_cast<s32>(size);
}

//---------------------------------------------------------------------------
bool
HioStream::Connect()
{
    if(!IsOpened())
    {
        NW_ASSERTMSG(false, "HioStream is not opened.");
        return false;
    }

    if(!IsAvailable())
    {
        return true;
    }

    if (!m_IsConnect && m_IsRunning)
    {
        nn::Result result = m_Stream.WaitHost( nn::hio::CTR::SerialChannel::ATTRIBUTE_NO_WAIT );
        if ( result == nn::hio::CTR::ResultConnected() )
        {
            m_IsConnect = true;
            return true;
        }
    }

    return true;
}

//---------------------------------------------------------------------------
void
HioStream::Disconnect()
{
    if( m_IsConnect ){
        m_IsConnect = false;
        m_Stream.Disconnect();
    }
}

//---------------------------------------------------------------------------
bool
HioStream::IsAvailable() const
{
    return m_IsAvailable;
}

//---------------------------------------------------------------------------
void
HioStream::ReadBuffer()
{
    if (!IsAvailable())
    {
        return;
    }

    size_t readableSize = 0;
    nn::Result result = m_Stream.Read(&readableSize, m_WorkBuffer, m_WorkBufferLength, nn::hio::CTR::SerialChannel::ATTRIBUTE_NO_WAIT);
    if (!ProcessResult(result))
    {
        return;
    }
    if (readableSize == 0)
    {
        return;
    }


#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] Read : result=0x%08x, length=%08d.\n",
        __FUNCTION__, this, result, m_WorkBufferLength);
#endif

    m_RingBuffer.Write(m_WorkBuffer, readableSize);
}

bool
HioStream::ProcessResult(nn::Result result)
{
    if (result == nn::Result(nn::Result::LEVEL_PERMANENT, nn::Result::SUMMARY_STATUS_CHANGED, nn::Result::MODULE_NN_HIO, 8))
    {
        Disconnect();
        return false;
    }

    if ( result.IsFailure() )
    {
        Disconnect();
        return false;
    }

    return true;
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
