﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_HioFileStreamImpl.h>

#if !defined(NW_RELEASE)
//#define NW_SND_FND_STREAMIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {

//---------------------------------------------------------------------------
u32
GetSeekPosition(FileStream& target, s32 offset, FileStream::SeekOrigin origin)
{
    u32 result = 0;

    switch(origin)
    {
    case FileStream::SEEK_ORIGIN_BEGIN:
        result = offset <= 0 ? 0 : offset;
        break;

    case FileStream::SEEK_ORIGIN_END:
        result = offset >= 0 ? target.GetSize() : target.GetSize() - offset;
        break;

    case FileStream::SEEK_ORIGIN_CURRENT:
        if(offset > 0)
        {
            result = target.GetCurrentPosition() + static_cast<u32>(offset);

            u32 size = target.GetSize();

            if(result > size)
            {
                result = size;
            }
        }
        else if(offset < 0)
        {
            if(target.GetCurrentPosition() > static_cast<u32>(-offset))
            {
                result = target.GetCurrentPosition() - static_cast<u32>(-offset);
            }
            else
            {
                result = 0;
            }
        }
        else
        {
            result = target.GetCurrentPosition();
        }
        break;

    default:
        NW_FATAL_ERROR("invalid seek origin.\n");
        break;
    }

    return result;
}

}

//---------------------------------------------------------------------------
HioFileStreamImpl::HioFileStreamImpl() :
m_IsOpened(false),
m_OpenMode(nn::hio::HostFile::ACCESS_MODE_READ),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0),
m_DirectStream(*this)
{
}

//---------------------------------------------------------------------------
nn::Result
HioFileStreamImpl::Open(const char* filePath, bit32 openMode)
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERTMSG(!IsOpened(), "HioFileStreamImpl is already opened.\n");

    m_CurrentPosition = 0;
    nn::Result result = m_HostFile.Open(filePath, openMode, nn::hio::HostFile::OPEN_DISP_OPEN_ALWAYS);
    m_IsOpened = result.IsSuccess();
    m_OpenMode = openMode;
    m_FileSize = 0;

    return result;
}

//---------------------------------------------------------------------------
void
HioFileStreamImpl::Close()
{
    if(!IsOpened())
    {
        return;
    }

    m_HostFile.Close();
    m_CurrentPosition = 0;
    m_IsOpened = false;
}

//---------------------------------------------------------------------------
bool
HioFileStreamImpl::IsOpened() const
{
    return m_IsOpened;
}

//---------------------------------------------------------------------------
u32
HioFileStreamImpl::GetSize() const
{
    NW_ASSERTMSG(IsOpened(), "HioFileStreamImpl is closed.\n");

    if (m_OpenMode == nn::hio::HostFile::ACCESS_MODE_READ && m_FileSize == 0)
    {
        s64 currentPosition = 0;
        nn::Result seekResult = m_HostFile.Seek(&currentPosition, 0, nn::hio::HostFile::SEEK_TYPE_CURRENT);
        NW_ASSERT(seekResult.IsSuccess());

        s64 fileSize = 0;
        seekResult = m_HostFile.Seek(&fileSize, 0, nn::hio::HostFile::SEEK_TYPE_END);
        NW_ASSERT(seekResult.IsSuccess());

        s64 position = 0;
        seekResult = m_HostFile.Seek(&position, currentPosition, nn::hio::HostFile::SEEK_TYPE_SET);
        NW_ASSERT(seekResult.IsSuccess());

        m_FileSize = static_cast<u32>(fileSize);
    }

    return m_FileSize;
}

//---------------------------------------------------------------------------
u32
HioFileStreamImpl::ReadDirect(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "HioFileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    FndResult readResult(SNDFND_RESULT_TRUE);
    u32 readFileLength = 0;

    u32 readLength = 0;
    nn::Result nnResult = m_HostFile.Read(&readLength, buf, length);

    if(nnResult.IsFailure())
    {
        readResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        readResult = length == readLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += readLength;
    }

    if(result != NULL)
    {
        *result = readResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] Read : curPos=%08x, buf=0x%08x, count=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length);
#endif

    return readLength;
}

//---------------------------------------------------------------------------
u32
HioFileStreamImpl::WriteDirect(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "HioFileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    u32 writtenLength = 0;
    FndResult writeResult(SNDFND_RESULT_TRUE);

    nn::Result nnResult = m_HostFile.Write(&writtenLength, buf, length);

    if(nnResult.IsFailure())
    {
        writeResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        writeResult = length == writtenLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += writtenLength;

        m_FileSize += writtenLength;
    }

    if(result != NULL)
    {
        *result = writeResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] Write : curPos=%08x, buf=0x%08x, count=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length);
#endif

    return writtenLength;
}

//---------------------------------------------------------------------------
FndResult
HioFileStreamImpl::SeekDirect(s32 offset, SeekOrigin origin)
{
    NW_ASSERTMSG(IsOpened(), "HioFileStreamImpl is closed.\n");

    u32 seekPosition = GetSeekPosition(*this, offset, origin);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] TrySeek : curPos=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, seekPosition);
#endif

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_ASSERTMSG(
        seekPosition < GetSize(),
        "[%s][%08x] seek to out of range : curPos=0x%08x, size=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, GetSize(), seekPosition);
#endif

    s64 result = m_HostFile.Seek(static_cast<s64>(seekPosition), nn::hio::HostFile::SEEK_TYPE_SET);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] Seek : curPos=%08x.\n",
        __FUNCTION__, this, m_CurrentPosition);
#endif

    if(result < 0)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    m_CurrentPosition = seekPosition;

    return FndResult(SNDFND_RESULT_TRUE);
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
