﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_FileStreamImpl.h>

#if !defined(NW_RELEASE)
//#define NW_SND_FND_STREAMIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {

//---------------------------------------------------------------------------
u32
GetSeekPosition(FileStreamImpl& target, s32 offset, FileStreamImpl::SeekOrigin origin)
{
    u32 result = 0;

    switch(origin)
    {
    case FileStreamImpl::SEEK_ORIGIN_BEGIN:
        result = offset <= 0 ? 0 : offset;
        break;

    case FileStreamImpl::SEEK_ORIGIN_END:
        result = offset >= 0 ? target.GetSize() : target.GetSize() - offset;
        break;

    case FileStreamImpl::SEEK_ORIGIN_CURRENT:
        if(offset > 0)
        {
            result = target.GetCurrentPosition() + static_cast<u32>(offset);

            u32 size = target.GetSize();

            if(result > size)
            {
                result = size;
            }
        }
        else if(offset < 0)
        {
            if(target.GetCurrentPosition() > static_cast<u32>(-offset))
            {
                result = target.GetCurrentPosition() - static_cast<u32>(-offset);
            }
            else
            {
                result = 0;
            }
        }
        else
        {
            result = target.GetCurrentPosition();
        }
        break;

    default:
        NW_FATAL_ERROR("invalid seek origin.\n");
        break;
    }

    return result;
}

}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl() :
m_IsOpened(false),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0)
{
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
nn::Result
FileStreamImpl::Open(const char* filePath, bit32 openMode)
{
    NW_ASSERT_NOT_NULL(filePath);
    NW_ASSERTMSG(!IsOpened(), "FileStreamImpl is already opened.\n");

    m_CurrentPosition = 0;
    nn::Result result = m_FileStream.TryInitialize(filePath, openMode);
    m_IsOpened = result.IsSuccess();

    return result;
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Close()
{
    if(!IsOpened())
    {
        return;
    }

    m_FileStream.Finalize();
    m_CurrentPosition = 0;
    m_IsOpened = false;
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::IsOpened() const
{
    return m_IsOpened;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::GetSize() const
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    if(m_FileSize == INVALID_SIZE)
    {
        s64 fileSize = 0;
        nn::Result result = m_FileStream.TryGetSize(&fileSize);

        if(result.IsFailure())
        {
            return 0;
        }

        m_FileSize = static_cast<u32>(fileSize);
    }

    return m_FileSize;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::ReadDirect(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    FndResult readResult(SNDFND_RESULT_TRUE);
    u32 readFileLength = 0;

    s32 readLength = 0;
    nn::Result nnResult = m_FileStream.TryRead(&readLength, buf, length);

    if(nnResult.IsFailure())
    {
        readResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        readResult = length == readLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += readLength;
    }

    if(result != NULL)
    {
        *result = readResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] TryRead : curPos=%08x, buf=0x%08x, count=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length);
#endif

    return static_cast<u32>(readLength);
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::WriteDirect(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    ValidateAlignment(buf);

    s32 writtenLength = 0;
    FndResult writeResult(SNDFND_RESULT_TRUE);

    m_FileSize = INVALID_SIZE;

    nn::Result nnResult = m_FileStream.TryWrite(&writtenLength, buf, length, true);

    if(nnResult.IsFailure())
    {
        writeResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        writeResult = length == writtenLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += writtenLength;
    }

    if(result != NULL)
    {
        *result = writeResult;
    }

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] TryWrite : curPos=%08x, buf=0x%08x, count=%d.\n",
        __FUNCTION__, this, m_CurrentPosition, buf, length);
#endif

    return writtenLength;
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::SeekDirect(s32 offset, SeekOrigin origin)
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    u32 seekPosition = GetSeekPosition(*this, offset, origin);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] TrySeek : curPos=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, seekPosition);
#endif

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_ASSERTMSG(
        seekPosition < GetSize(),
        "[%s][%08x] seek to out of range : curPos=0x%08x, size=0x%08x, newPos=0x%08x.\n",
        __FUNCTION__, this, m_CurrentPosition, GetSize(), seekPosition);
#endif

    nn::Result nnResult = m_FileStream.TrySeek(static_cast<s64>(seekPosition), nn::fs::POSITION_BASE_BEGIN);

#if defined(NW_SND_FND_STREAMIO_DEBUG)
    NW_LOG(
        "[%s][%08x] TrySeek : curPos=%08x.\n",
        __FUNCTION__, this, m_CurrentPosition);
#endif

    if(nnResult.IsFailure())
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    m_CurrentPosition = seekPosition;

    return FndResult(SNDFND_RESULT_TRUE);
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
