﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_HioStream.h>

#if !defined(NW_PLATFORM_CTR)
#include <nw/ut/ut_String.h>
#endif

#if !defined(NW_RELEASE)
//#define NW_SND_FND_HIO_DEBUG
#endif

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
void
HioStream::Initialize(void* workBuffer, u32 workBufferLength)
{
    m_WorkBuffer = workBuffer;
    m_WorkBufferLength = workBufferLength;

    m_HioHandle = 0;
    //OSInitEvent(&m_ConnectEvent, false, OS_EVENT_MANUAL);
    m_IsReading = false;
    m_IsAvailable = false;
}

//---------------------------------------------------------------------------
void
HioStream::Finalize()
{
    m_IsAvailable = false;
    m_IsReading = false;
    m_HioHandle = 0;
    m_WorkBufferLength = 0;
    m_WorkBuffer = NULL;
}

//---------------------------------------------------------------------------
void
HioStream::OpenImpl(ChannelType channel)
{
    //OSResetEvent(&m_ConnectEvent);

    HIOAsyncStruct asyncStruct;
    asyncStruct.queue = OSGetDefaultAppIOQueue();
    asyncStruct.userData = this;
    asyncStruct.callback = (HIOCallback)ConnectionCallback;

    // READ専用、WRITE専用にすることで速くなるかもしれないので、
    // 速度が問題になる時は、ここは見直すポイントになるはずです。
    m_HioHandle = HIOOpenEx(channel, &asyncStruct, HIO_CHANNEL_OPTION_READ_WRITE, 0);

    m_IsOpened = true;
    //OSWaitEvent(&m_ConnectEvent);

#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] HIOOpen : m_IsOpened=%d, m_HioHandle=0x%08x, Name=%s.\n",
        __FUNCTION__, this, m_IsOpened, m_HioHandle, channel);
#endif
}

//---------------------------------------------------------------------------
void
HioStream::CloseImpl()
{
    if (m_IsOpened && m_HioHandle > 0)
    {
        HIOStatus status = HIOClose(m_HioHandle);
        NW_ASSERT(status == HIO_STATUS_OK);
        m_HioHandle = 0;
    }

    m_IsOpened = false;

#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] HIOClose : m_IsOpened=%d, m_HioHandle=0x%08x.\n",
        __FUNCTION__, this, m_IsOpened, m_HioHandle);
#endif
}

//---------------------------------------------------------------------------
s32
HioStream::WriteImpl(const void* buffer, u32 length)
{
    if(!IsAvailable())
    {
        return 0;
    }

    if(length == 0)
    {
        return 0;
    }

    HIOStatus result = HIO_STATUS_OK;

    for(;;)
    {
        result = HIOWrite(m_HioHandle, length, buffer);

        if(result != HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE)
        {
            break;
        }
    }

    return result;
}

//---------------------------------------------------------------------------
bool
HioStream::Connect()
{
    return true;
}

//---------------------------------------------------------------------------
void
HioStream::Disconnect()
{
}

//---------------------------------------------------------------------------
bool
HioStream::IsAvailable() const
{
    return m_IsAvailable;
}

//----------------------------------------------------------
void
HioStream::ConnectionCallback(HIOStatus status, void* context)
{
    HioStream* target = reinterpret_cast<HioStream*>(context);
    NW_ASSERT_NOT_NULL(target);

#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG("[nw::snd::internal::fnd::HioStream] ConnectionCallback : channel=%s, status=%d.\n", target->m_Channel, status);
#endif

    // 新たに接続された。
    if (status == HIO_STATUS_NEW_CONNECTION)
    {
        target->m_IsAvailable = true;
        //OSSignalEvent(&target->m_ConnectEvent);

#if defined(NW_SND_FND_HIO_DEBUG)
        NW_LOG(
            "[nw::snd::internal::fnd::HioStream] HIO_STATUS_NEW_CONNECTION : channel=%s, isAvailable=%d.\n",
            target->m_Channel,
            target->IsAvailable());
#endif
        return;
    }

    // すべての接続が切断された。
    // Cafe の HIO では、複数の PC アプリが同一ポートに接続可能。
    if (status == HIO_STATUS_NO_CONNECTIONS)
    {
        target->Close();
        target->m_IsAvailable = false;

#if defined(NW_SND_FND_HIO_DEBUG)
        NW_LOG(
            "[nw::snd::internal::fnd::HioStream] HIO_STATUS_NO_CONNECTIONS : channel=%s, isAvailable=%d.\n",
            target->m_Channel,
            target->IsAvailable());
#endif

        //OSSignalEvent(&target->m_ConnectEvent);
        return;
    }

    // エラー。
    if(status < 0)
    {
        target->Close();
        target->m_IsAvailable = false;

#if defined(NW_SND_FND_HIO_DEBUG)
        NW_LOG(
            "[nw::snd::internal::fnd::HioStream] ConnectionCallback : channel=%s, isAvailable=%d.\n",
            target->m_Channel,
            target->IsAvailable());
#endif
        return;
    }
}

//---------------------------------------------------------------------------
void
HioStream::ReadBufferFinishedCallback(HIOStatus status, void* context)
{
    if (status == HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE)
    {
#if defined(NW_SND_FND_HIO_DEBUG)
        HioStream* stream = static_cast<HioStream*>(context);
        NW_LOG(
            "[%s][%08x] HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE.\n",
            __FUNCTION__, stream);
#endif
        return;
    }

    HioStream* stream = static_cast<HioStream*>(context);

#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] : status=0x%08x, ch=%s, isReading=%d.\n",
        __FUNCTION__, stream, status, stream->m_Channel, stream->m_IsReading);
#endif

    if (status <= 0)
    {
        stream->m_IsReading = false;
        return;
    }

    stream->m_RingBuffer.Write(stream->m_WorkBuffer, status);
    stream->m_IsReading = false;
}

//---------------------------------------------------------------------------
void
HioStream::ReadBuffer()
{
    if (!IsAvailable())
    {
        return;
    }

    if (m_IsReading)
    {
        return;
    }

    HIOStatus result = HIO_STATUS_OK;

    HIOAsyncStruct asyncStruct;
    asyncStruct.queue = OSGetDefaultAppIOQueue();
    asyncStruct.userData = this;
    asyncStruct.callback = (HIOCallback)ReadBufferFinishedCallback;

    m_IsReading = true;
    result = HIOReadAsyncEx(m_HioHandle, m_WorkBufferLength, m_WorkBuffer, &asyncStruct);

#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] HIOReadAsyncEx : result=0x%08x, length=%08d, isReading=%d.\n",
        __FUNCTION__, this, result, m_WorkBufferLength, m_IsReading);
#endif

    if (result < 0)
    {
        m_IsReading = false;
        NW_ASSERT(false);
        return;
    }

    if (result != HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE)
    {
        return;
    }

    m_IsReading = false;
#if defined(NW_SND_FND_HIO_DEBUG)
    NW_LOG(
        "[%s][%08x] HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE.\n",
        __FUNCTION__, this);
#endif
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
