﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/io/sndfnd_FileStreamImpl.h>

#include <cstdio>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

namespace {

//---------------------------------------------------------------------------
int
SeekOriginToMoveMethod(Stream::SeekOrigin origin)
{
    switch(origin)
    {
        case FileStreamImpl::SEEK_ORIGIN_BEGIN:
            return SEEK_SET;

        case FileStreamImpl::SEEK_ORIGIN_END:
            return SEEK_END;

        case FileStreamImpl::SEEK_ORIGIN_CURRENT:
            return SEEK_CUR;
    }

    NW_FATAL_ERROR("invalid seek origin.\n");
    return SEEK_SET;
}

}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl() :
m_FileHandle(NULL),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0)
{
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
FileStreamImpl::FileStreamImpl(void* fileHandle) :
m_FileHandle(fileHandle),
m_FileSize(INVALID_SIZE),
m_CurrentPosition(0)
{
    NW_ASSERT_NOT_NULL(fileHandle);
    m_DirectStream.Initialize(*this);
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Open(void* fileHandle)
{
    NW_ASSERT_NOT_NULL(fileHandle);
    NW_ASSERTMSG(!IsOpened(), "FileStreamImpl is already opened.\n");
    m_FileHandle = fileHandle;
    m_CurrentPosition = 0;
}

//---------------------------------------------------------------------------
void
FileStreamImpl::Close()
{
    if(m_FileHandle == NULL)
    {
        return;
    }

    int result = std::fclose(reinterpret_cast<FILE*>(m_FileHandle));
    NW_ASSERT(result !=  EOF);
    m_FileHandle = NULL;
    m_CurrentPosition = 0;
}

//---------------------------------------------------------------------------
bool
FileStreamImpl::IsOpened() const
{
    return m_FileHandle != NULL;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::GetSize() const
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");
    int result = 0;
    fpos_t fpos;

    // TODO: 本当はfstatを利用して取得するべき
    //       https://www.jpcert.or.jp/sc-rules/c-fio19-c.html

    FILE *fp = reinterpret_cast<FILE*>(m_FileHandle);
    result = std::fseek(fp, 0, SEEK_END);
    if (result < 0)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    result = std::fgetpos(fp, &fpos);
    if (result < 0)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }
    m_FileSize = static_cast<u32>(fpos);


    fpos = m_CurrentPosition;
    result = std::fsetpos(fp, &fpos);
    if (result < 0)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    return m_FileSize;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::ReadDirect(void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    FndResult readResult(SNDFND_RESULT_TRUE);
    size_t readFileLength = 0;

    readFileLength = std::fread(buf, 1, length, reinterpret_cast<FILE*>(m_FileHandle));

    if (!readFileLength > 0)
    {
        readResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        readResult = length == readFileLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += readFileLength;
    }

    if(result != NULL)
    {
        *result = readResult;
    }

    return readFileLength;
}

//---------------------------------------------------------------------------
u32
FileStreamImpl::WriteDirect(const void* buf, u32 length, FndResult* result /*= NULL*/)
{
    NW_ASSERT_NOT_NULL(buf);
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    FndResult writeResult(SNDFND_RESULT_TRUE);
    size_t writtenFileLength = 0;

    writtenFileLength = std::fwrite(buf, 1, length, reinterpret_cast<FILE*>(m_FileHandle));

    if (!writtenFileLength > 0)
    {
        writeResult = FndResult(SNDFND_RESULT_IO_ERROR);
    }
    else
    {
        writeResult = length == writtenFileLength ?
            FndResult(SNDFND_RESULT_TRUE) : FndResult(SNDFND_RESULT_FALSE);

        m_CurrentPosition += writtenFileLength;
    }

    if(result != NULL)
    {
        *result = writeResult;
    }

    return writtenFileLength;
}

//---------------------------------------------------------------------------
FndResult
FileStreamImpl::SeekDirect(s32 offset, SeekOrigin origin)
{
    NW_ASSERTMSG(IsOpened(), "FileStreamImpl is closed.\n");

    int result = std::fseek(reinterpret_cast<FILE*>(m_FileHandle), offset, SeekOriginToMoveMethod(origin));

    if(result < 0)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    fpos_t currentPosition = std::ftell(reinterpret_cast<FILE*>(m_FileHandle));

    if (currentPosition < 0)
    {
        return FndResult(SNDFND_RESULT_IO_ERROR);
    }

    m_CurrentPosition = currentPosition;

    return FndResult(SNDFND_RESULT_TRUE);
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
