﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Types.h>
#include <nw/lyt/lyt_TexMap.h>
#include <nw/lyt/lyt_Common.h>

namespace
{


}

namespace nw
{
namespace lyt
{

TexMap::TexMap()
: m_TexObject(0)
, m_Width(0)
, m_Height(0)
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
, m_Sampler(0)
#elif defined(NW_PLATFORM_CAFE)
, m_Texture(NULL)
#endif
{
    m_Bits.format = TEXFORMAT_MAX;
    m_Bits.wrapS = TEXWRAP_CLAMP;
    m_Bits.wrapT = TEXWRAP_CLAMP;
    m_Bits.minFilter = TEXFILTER_LINEAR;
    m_Bits.magFilter = TEXFILTER_LINEAR;

    InitSampler();
}

TexMap::TexMap(const TextureInfo* textureInfo)
{
    m_Bits.wrapS = TEXWRAP_CLAMP;
    m_Bits.wrapT = TEXWRAP_CLAMP;
    m_Bits.minFilter = TEXFILTER_LINEAR;
    m_Bits.magFilter = TEXFILTER_LINEAR;

    this->Set(textureInfo);

    InitSampler();
}

TexMap::~TexMap()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    glDeleteSamplers( 1, &m_Sampler );
#endif
}

void
TexMap::SetWrapMode(TexWrap wrapS, TexWrap wrapT)
{
    NW_ASSERT(wrapS < TEXWRAP_MAX);
    NW_ASSERT(wrapT < TEXWRAP_MAX);

    m_Bits.wrapS = wrapS;
    m_Bits.wrapT = wrapT;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    static const GLint wrapMode[TEXWRAP_MAX] =
    {
        GL_CLAMP_TO_EDGE,
        GL_REPEAT,
        GL_MIRRORED_REPEAT,
    };
    glSamplerParameteri(m_Sampler, GL_TEXTURE_WRAP_S, wrapMode[wrapS]);
    glSamplerParameteri(m_Sampler, GL_TEXTURE_WRAP_T, wrapMode[wrapT]);
    NW_GL_ASSERT();
#elif defined(NW_PLATFORM_CAFE)
    static const GX2TexClamp wrapMode[TEXWRAP_MAX] =
    {
        GX2_TEX_CLAMP_CLAMP, // clamps at the last texel value
        GX2_TEX_CLAMP_WRAP, // repeat
        GX2_TEX_CLAMP_MIRROR, // mirrored repeat
        //GX2_TEX_CLAMP_MIRROR_ONCE, // mirrors once, then clamps like previous option
        //GX2_TEX_CLAMP_CLAMP_HALF_BORDER, // clamps: values outside are part border, part last texel
        //GX2_TEX_CLAMP_MIRROR_ONCE_HALF_BORDER, // mirrors once, then clamps like previous option
        //GX2_TEX_CLAMP_CLAMP_BORDER, // clamps: values outside may be 100% border
        //GX2_TEX_CLAMP_MIRROR_ONCE_BORDER, //mirrors once, then clamps like previous option
    };

    GX2InitSamplerClamping(
        &m_Sampler,
        wrapMode[wrapS],
        wrapMode[wrapT],
        GX2_TEX_CLAMP_CLAMP);
#endif
}

void
TexMap::SetFilter(TexFilter minFlt, TexFilter magFlt)
{
    NW_ASSERT(minFlt <= TEXFILTER_MAX);
    NW_ASSERT(magFlt <= TEXFILTER_MAX);

    m_Bits.minFilter = minFlt;
    m_Bits.magFilter = magFlt;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    static const GLint filterMode[TEXFILTER_MAX] =
    {
        GL_NEAREST,
        GL_LINEAR,
    };
    glSamplerParameteri(m_Sampler, GL_TEXTURE_MIN_FILTER, filterMode[minFlt]);
    glSamplerParameteri(m_Sampler, GL_TEXTURE_MAG_FILTER, filterMode[magFlt]);
    NW_GL_ASSERT();
#elif defined(NW_PLATFORM_CAFE)
    static const GX2TexXYFilterType filterMode[GX2_TEX_XY_FILTER_LAST + 1] =
    {
        GX2_TEX_XY_FILTER_POINT,
        GX2_TEX_XY_FILTER_BILINEAR
    };
    static const GX2TexAnisoRatio maxAniso[GX2_TEX_ANISO_LAST + 1] =
    {
        GX2_TEX_ANISO_1_TO_1,
        GX2_TEX_ANISO_2_TO_1,
        GX2_TEX_ANISO_4_TO_1,
        GX2_TEX_ANISO_8_TO_1,
        GX2_TEX_ANISO_16_TO_1
    };
    GX2InitSamplerXYFilter(
        &m_Sampler,
        filterMode[magFlt],
        filterMode[minFlt],
        maxAniso[0]);
#endif
}

void
TexMap::Set(const TextureInfo* src)
{
    NW_WARNING(src, "TextureInfo is NULL.\n");
    if (src)
    {
        m_TexObject = src->GetTextureObject();
        m_Width = src->GetSize().width;
        m_Height = src->GetSize().height;
        m_Bits.format = src->GetFormat();
#if defined(NW_PLATFORM_CAFE)
        m_Texture = &src->GetTexture();
#endif
    }
}

void
TexMap::GetTextureInfo(TextureInfo* dst) const
{
    NW_ASSERT_NOT_NULL(dst);

    dst->Set(m_TexObject, TexSize(m_Width, m_Height), static_cast<TexFormat>(m_Bits.format));
#if defined(NW_PLATFORM_CAFE)
    dst->GetTexture() = *m_Texture;
#endif
}

void
TexMap::CopyTextureInfo(const TexMap& src)
{
    m_TexObject = src.GetTextureObject();
    m_Width = src.GetSize().width;
    m_Height = src.GetSize().height;
    m_Bits.format = src.GetFormat();
#if defined(NW_PLATFORM_CAFE)
    m_Texture = src.GetTexture();
#endif
}

void
TexMap::InitSampler()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    glGenSamplers(1, &m_Sampler);
    // 固定設定
    glSamplerParameterf(m_Sampler, GL_TEXTURE_LOD_BIAS, 0.0f);
    glSamplerParameteri(m_Sampler, GL_TEXTURE_MIN_LOD, -1000);
#elif defined(NW_PLATFORM_CAFE)
    GX2InitSampler(&m_Sampler, GX2_TEX_CLAMP_CLAMP, GX2_TEX_XY_FILTER_BILINEAR);
    // 固定設定
    GX2InitSamplerLOD(&m_Sampler, 0.0f, 15.9f, 0.0f);
    GX2InitSamplerZMFilter(&m_Sampler, GX2_TEX_Z_FILTER_USE_XY, GX2_TEX_MIP_FILTER_NO_MIP);
#endif
}

} // namespace nw::lyt
} // namespace nw
