﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_Layout.h>

#include <nw/lyt/lyt_Animation.h>
#include <nw/lyt/lyt_Bounding.h>
#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_DrawInfo.h>
#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Group.h>
#include <nw/lyt/lyt_Material.h>
#include <nw/lyt/lyt_Picture.h>
#include <nw/lyt/lyt_ResourceAccessor.h>
#include <nw/lyt/lyt_Resources.h>
#include <nw/lyt/lyt_TextBox.h>
#include <nw/lyt/lyt_Util.h>
#include <nw/lyt/lyt_Window.h>
#include <nw/lyt/lyt_ControlCreator.h>

#include <nw/dev/dev_Profile.h>

namespace nw
{
namespace lyt
{

nw::ut::IAllocator* Layout::s_pAllocator = 0;

namespace
{

/*---------------------------------------------------------------------------*
  @brief 再帰的にペインにタグプロセッサ設定します。
 *---------------------------------------------------------------------------*/
void
SetTagProcessorImpl(
    Pane*                           pPane,
    font::TagProcessorBase<char16>*  pTagProcessor
)
{
    // TextBox ペインか
    if (TextBox* pTextBox = ut::DynamicCast<TextBox*>(pPane))
    {
        pTextBox->SetTagProcessor(pTagProcessor);
    }

    // 再帰的にセット
    PaneList::Iterator it_end = pPane->GetChildList().GetEndIter();
    for (PaneList::Iterator it = pPane->GetChildList().GetBeginIter(); it != it_end; ++it)
    {
        SetTagProcessorImpl(&(*it), pTagProcessor);
    }
}

bool
IsIncludeAnimationGroupRef(
    GroupContainer*                 pGroupContainer,
    const res::AnimationGroupRef *const  groupRefs,
    u16                             bindGroupNum,
    bool                            bDescendingBind,
    Pane*                           pTargetPane
)
{
    for (u16 grpIdx = 0; grpIdx < bindGroupNum; ++grpIdx)
    {
        // 共有対象のペインがバインド指定のグループが持つペインリストに含まれるか調べます。
        Group *const pGroup = pGroupContainer->FindGroupByName(groupRefs[grpIdx].GetName());
        PaneLinkList& paneList = pGroup->GetPaneList();
        PaneLinkList::Iterator it_end = paneList.GetEndIter();
        for (PaneLinkList::Iterator it = paneList.GetBeginIter(); it != it_end; ++it)
        {
            if (it->target == pTargetPane)
            {
                return true;
            }

            if (bDescendingBind)
            {
                // pTargetPaneの親方向で一致するものが無いか調べる。
                for (Pane* pParentPane = pTargetPane->GetParent(); pParentPane; pParentPane = pParentPane->GetParent())
                {
                    if (it->target == pParentPane)
                    {
                        return true;
                    }
                }
            }
        }
    }

    return false;
}

} // namespace nw::lyt::{no-name}

void
Layout::SetAllocator(nw::ut::IAllocator* pAllocator)
{
    s_pAllocator = pAllocator;
}

void*
Layout::AllocMemory(u32 size, u32 alignment)
{
    NW_ASSERT_NOT_NULL(s_pAllocator);

    void *const pMem = s_pAllocator->Alloc(size, alignment);
    NW_WARNING(pMem, "can't alloc memory.");
    return pMem;
}

void
Layout::FreeMemory(void* mem)
{
    NW_ASSERT_NOT_NULL(s_pAllocator);

    s_pAllocator->Free(mem);
}

Layout::Layout()
: m_pRootPane(0),
  m_pGroupContainer(0),
  m_LayoutSize(0.f, 0.f),
  m_Name(NULL),
  m_ResourceAccessor(NULL),
  m_MaxCharAttributeNum(0),
  m_pSharedCharAttrs(NULL)
{
}

Layout::Layout(const Layout& src, Layout* parentLayout /* = NULL */, const char* rootPaneName /* = NULL */)
 : m_pRootPane(0),
  m_pGroupContainer(0),
  m_LayoutSize(src.m_LayoutSize),
  m_Name(src.m_Name),
  m_ResourceAccessor(src.m_ResourceAccessor),
  m_MaxCharAttributeNum(src.m_MaxCharAttributeNum),
  m_pSharedCharAttrs(NULL)
{
    // 文字属性バッファのコピー
    if ( ! src.IsPartsLayout() && m_MaxCharAttributeNum > 0) {
        const u32 size = sizeof(nw::font::internal::CharAttribute) * m_MaxCharAttributeNum;
        m_pSharedCharAttrs = static_cast<nw::font::internal::CharAttribute*>(Layout::AllocMemory(size));
    }

    // ペインツリーのコピー
    m_pRootPane = ClonePaneTreeWithPartsLayout(src.GetRootPane(), src.IsPartsLayout() ? this : NULL);
    if (rootPaneName) {
        m_pRootPane->SetName(rootPaneName);
    }

    // グループのコピー
    m_pGroupContainer = NewObj<GroupContainer>();
    const nw::lyt::GroupList& src_groups = src.GetGroupContainer()->GetGroupList();
    for (nw::lyt::GroupList::ConstIterator it = src_groups.GetBeginIter(), it_end = src_groups.GetEndIter(); it != it_end; ++it) {
        m_pGroupContainer->AppendGroup(NewObj<Group, const Group&, Pane*>(static_cast<const Group&>(*it), m_pRootPane));
    }

    // 親のレイアウトがある場合は、RootPaneが部品ペインのはずなので、親のレイアウトの部品ペインリストに登録する
    if (parentLayout) {
        Parts* parts = nw::ut::DynamicCast<Parts*>(m_pRootPane);
        if (parts) {
            parentLayout->GetPartsPaneList().PushBack(parts);
        }
    }
}

Layout::~Layout()
{
    if (m_pSharedCharAttrs != NULL)
    {
        Layout::FreeMemory(m_pSharedCharAttrs);
        m_pSharedCharAttrs = NULL;
    }

    DeleteObj(m_pGroupContainer);

    if (m_pRootPane && !m_pRootPane->IsUserAllocated())
    {
        DeleteObj(m_pRootPane);
    }

    for (AnimTransformList::Iterator it = m_AnimTransList.GetBeginIter(); it != m_AnimTransList.GetEndIter();)
    {
        AnimTransformList::Iterator currIt = it++;
        m_AnimTransList.Erase(currIt);
        DeleteObj(&(*currIt));
    }
}

bool Layout::Build(
    const void* lytResBuf,
    ResourceAccessor* pResAcsr,
    ControlCreator* pControlCreator /* = NULL */,
    TextSearcher* pTextSearcher /* = NULL */,
    const BuildOption& buildOption /* = BuildOption() */)
{
    BuildArgSet buildArgSet;
    buildArgSet.magnify.x = 1.f;
    buildArgSet.magnify.y = 1.f;
    buildArgSet.partsSize.x = 0.f;
    buildArgSet.partsSize.y = 0.f;
    buildArgSet.pControlCreator = pControlCreator;
    buildArgSet.pTextSearcher = pTextSearcher;
    buildArgSet.pLayout = NULL;
    buildArgSet.pBodyLayout = this;
    buildArgSet.isShareTextBoxCharAttributeBuffer = buildOption.isShareTextBoxCharAttributeBuffer;
    buildArgSet.isRootPaneParts = buildOption.isRootPaneParts;
#if ! defined(NW_RELEASE)
    buildArgSet.isViewerInvisibleByParent = false;
#endif
    return BuildImpl(lytResBuf, pResAcsr, buildArgSet, NULL);
}

bool Layout::BuildWithName(
    const char* layoutFileName,
    ResourceAccessor* pResAcsr,
    ControlCreator* pControlCreator /* = NULL */,
    TextSearcher* pTextSearcher /* = NULL */,
    const BuildOption& buildOption /* = BuildOption() */)
{
    const void* lytRes = pResAcsr->GetResource(nw::lyt::res::RESOURCETYPE_LAYOUT, layoutFileName);
    if (lytRes)
    {
        return Build(lytRes, pResAcsr, pControlCreator, pTextSearcher, buildOption);
    }
    else
    {
        return false;
    }
}

bool
Layout::BuildImpl(
    const void* lytResBuf,
    ResourceAccessor* pResAcsr,
    const BuildArgSet& parentBuildArgSet,
    const PartsBuildDataSet* partsBuildDataSet
)
{
    NW_ASSERT_NOT_NULL(s_pAllocator);
    NW_ASSERT_NOT_NULL(lytResBuf);
    m_ResourceAccessor = pResAcsr;

    const ut::BinaryFileHeader *const pFileHead = static_cast<const ut::BinaryFileHeader*>(lytResBuf);

    if (! ut::IsValidBinaryFile(pFileHead, res::FILESIGNATURE_FLYT, res::BinaryFileFormatVersion))
    {
        NW_ERR("not valid layout file.");
    }

    BuildResSet buildResSet = { 0 };
    buildResSet.pResAccessor = pResAcsr;
    buildResSet.pLayout = this;

    // 親レイアウトの情報を元に、自分のBuildArgSetを作成する
    BuildArgSet buildArgSet;
    if (partsBuildDataSet) {
        buildArgSet.magnify = partsBuildDataSet->GetMagnify();
        buildArgSet.pOverrideBuildResSet = partsBuildDataSet->GetPropertyBuildResSet();
    } else {
        // 部品レイアウトでない場合は1.f
        buildArgSet.magnify.x = 1.f;
        buildArgSet.magnify.y = 1.f;
        buildArgSet.pOverrideBuildResSet = NULL;
    }
    buildArgSet.pControlCreator = parentBuildArgSet.pControlCreator;
    buildArgSet.pTextSearcher = parentBuildArgSet.pTextSearcher;
    buildArgSet.pLayout = this;
    buildArgSet.pBodyLayout = parentBuildArgSet.pBodyLayout;
    buildArgSet.pCurrentBuildResSet = &buildResSet;
    buildArgSet.overrideUsageFlag = 0;
    buildArgSet.overrideMaterialUsageFlag = 0;
    buildArgSet.pParentPane = NULL;
    buildArgSet.isShareTextBoxCharAttributeBuffer = parentBuildArgSet.isShareTextBoxCharAttributeBuffer;
    buildArgSet.isRootPaneParts = parentBuildArgSet.isRootPaneParts;
#if ! defined(NW_RELEASE)
    buildArgSet.isViewerInvisibleByParent = parentBuildArgSet.isViewerInvisibleByParent
        || (partsBuildDataSet && partsBuildDataSet->GetPartsPane()->IsViewerInvisible());
#endif
    // この値は後で決まるので、ここでは0を入れる
    buildArgSet.partsSize.Set(0.f, 0.f);

    m_MaxCharAttributeNum = 0;

    Pane* pLastPane = 0;

    bool bReadRootGroup = false;
    int groupNestLevel = 0;

    const void* dataPtr = static_cast<const u8*>(lytResBuf) + pFileHead->headerSize;
    for (int i = 0; i < pFileHead->dataBlocks; ++i)
    {
        const ut::BinaryBlockHeader* pDataBlockHead = static_cast<const ut::BinaryBlockHeader*>(dataPtr);
        ut::SigWord kind = pDataBlockHead->kind;
        switch (kind)
        {
        case res::DATABLOCKKIND_LAYOUT:
            {
                const res::Layout* pResLyt = static_cast<const res::Layout*>(dataPtr);
                this->m_LayoutSize.width = pResLyt->layoutSize.x;
                this->m_LayoutSize.height = pResLyt->layoutSize.y;
                this->m_Name = internal::ConvertOffsToPtr<char>(dataPtr, sizeof(res::Layout));
                buildArgSet.partsSize.Set(pResLyt->partsSize.x, pResLyt->partsSize.y);
            }
            break;

        case res::DATABLOCKKIND_CONTROL:
            if (buildArgSet.pControlCreator) {
                // このコントロールブロックはレイアウトの構築の一番最後に来る
                const res::ExtUserDataList* extUserDataBlock = NULL;
                const void* controlSrcDataPtr = dataPtr;
                // 次のデータブロックを覗く、もしDATABLOCKKIND_USERDATALISTがあれば、取得する
                if (i + 1 < pFileHead->dataBlocks) {
                    // これが最後のブロックでなければ
                    const ut::BinaryBlockHeader* pNextDataBlockHead = internal::ConvertOffsToPtr<ut::BinaryBlockHeader>(dataPtr, pDataBlockHead->size);
                    if (pNextDataBlockHead->kind == res::DATABLOCKKIND_USERDATALIST) {
                        extUserDataBlock = reinterpret_cast<const res::ExtUserDataList*>(pNextDataBlockHead);
                        i++;
                        dataPtr = reinterpret_cast<const u8*>(pNextDataBlockHead);
                        pDataBlockHead = pNextDataBlockHead;
                    }
                }
                ControlSrc controlSrc(controlSrcDataPtr, extUserDataBlock);
                buildArgSet.pControlCreator->CreateControl(controlSrc, this);
            }
            break;

        case res::DATABLOCKKIND_TEXTURELIST:
            buildResSet.pTextureList = static_cast<const res::TextureList*>(dataPtr);
            break;

        case res::DATABLOCKKIND_FONTLIST:
            buildResSet.pFontList = static_cast<const res::FontList*>(dataPtr);
            break;

        case res::DATABLOCKKIND_MATERIALLIST:
            buildResSet.pMaterialList = static_cast<const res::MaterialList*>(dataPtr);
            break;

        case res::DATABLOCKKIND_PANE:
        case res::DATABLOCKKIND_PICTURE:
        case res::DATABLOCKKIND_TEXTBOX:
        case res::DATABLOCKKIND_WINDOW:
        case res::DATABLOCKKIND_BOUNDING:
        case res::DATABLOCKKIND_PARTS:
            {
                Pane* pPane = NULL;
                const void* resBlock = dataPtr;
                const void* overrideResBlock = NULL;
                buildArgSet.overrideUsageFlag = 0;
                buildArgSet.overrideBasicUsageFlag = 0;
                buildArgSet.overrideMaterialUsageFlag = 0;
                buildArgSet.pOverridePartsPaneBasicInfo = NULL;
                buildArgSet.pExtUserDataList = NULL;
                // 次のデータブロックを覗く、もしDATABLOCKKIND_USERDATALISTがあれば、buildArgSetにセットする
                // ペインを構築する前にユーザーデータをセットするため、外側のswitch文ではこの処理は行えない
                if (i + 1 < pFileHead->dataBlocks) {
                    // これが最後のブロックでなければ
                    const ut::BinaryBlockHeader* pNextDataBlockHead = internal::ConvertOffsToPtr<ut::BinaryBlockHeader>(dataPtr, pDataBlockHead->size);
                    if (pNextDataBlockHead->kind == res::DATABLOCKKIND_USERDATALIST) {
                        // 次のブロックがDATABLOCKKIND_USERDATALISTであれば、buildArgSetにセットしてforループを一つ進める
                        buildArgSet.pExtUserDataList = reinterpret_cast<const res::ExtUserDataList*>(pNextDataBlockHead);
                        i++;
                        dataPtr = reinterpret_cast<const u8*>(pNextDataBlockHead);
                        pDataBlockHead = pNextDataBlockHead;
                    }
                }
                if (partsBuildDataSet) {
                    // 部品の場合
                    if (this->GetRootPane() == 0) {
                        // ルートペインとして既に作られている部品ペインを使用する
                        pPane = partsBuildDataSet->GetPartsPane();
                    } else {
                        const res::Pane* pResPane = static_cast<const res::Pane*>(resBlock);
                        const res::PartsProperty* prop = partsBuildDataSet->FindPartsPropertyFromName(pResPane->name);
                        if (prop) {
                            // 上書きがある。
                            overrideResBlock = partsBuildDataSet->GetPropertyResBlock(prop);
                            NW_ASSERT(overrideResBlock == NULL || (static_cast<const ut::BinaryBlockHeader*>(overrideResBlock))->kind == kind);
                            buildArgSet.overrideUsageFlag = prop->usageFlag;
                            buildArgSet.overrideBasicUsageFlag = prop->basicUsageFlag;
                            buildArgSet.overrideMaterialUsageFlag = prop->materialUsageFlag;
                            buildArgSet.pOverridePartsPaneBasicInfo = partsBuildDataSet->GetPartsPaneBasicInfoResBlock(prop);
                            bool isOverrideExtUserData = false;
                            const res::ExtUserDataList* list = partsBuildDataSet->GetExtUserDataListResBlock(&isOverrideExtUserData, prop);
                            if (isOverrideExtUserData) {
                                buildArgSet.pExtUserDataList = list;
                            }
                        }
                    }
                } else {
                    if (this->GetRootPane() == 0 && buildArgSet.isRootPaneParts)
                    {
                        // 部品ではなく、まだルートペインがなく、かつルートペインをPartsにする設定の場合
                        // res::Partsをその場で作成して構築
                        res::Parts res_parts;
                        *static_cast<res::Pane*>(&res_parts) = *static_cast<const res::Pane*>(resBlock);
                        res_parts.blockHeader.kind = res::DATABLOCKKIND_PARTS;
                        res_parts.propertyNum = 0;
                        res_parts.magnify.x = 1.f;
                        res_parts.magnify.y = 1.f;
                        pPane = BuildPaneObj(res::DATABLOCKKIND_PARTS, &res_parts, overrideResBlock, buildArgSet);
                        static_cast<Parts*>(pPane)->SetLayout(this);
                    }
                }
                if (pPane == NULL) {
                    // ここまででペインが構築されなかった場合は通常の方法でペインを作成する
                    pPane = BuildPaneObj(kind, resBlock, overrideResBlock, buildArgSet);
                }
                if (pPane)
                {
                    // ルートペインがまだ設定されていないときはルートペインに設定
                    if (this->GetRootPane() == 0)
                    {
                        this->SetRootPane(pPane);
                    }

                    // 親がいたら親に登録
                    if (buildArgSet.pParentPane)
                    {
                        buildArgSet.pParentPane->AppendChild(pPane);
                    }

                    if (kind == res::DATABLOCKKIND_PARTS) {
                        // 部品ペインの場合に必要な処理
                        Parts* partsPane = static_cast<Parts*>(pPane);
                        const void* localResBlock;
                        const BuildResSet* localBuildResSet;
                        if (overrideResBlock)
                        {
                            // 上書きが発生している場合は、リソースブロックのPaneの部分は正しい値でなくなるが、
                            // 以後は使用しないので問題ない。(もし必要になったときも、PartsBuildDataSet内のパーツ
                            // ペインから同等の情報を取得できる。)
                            localResBlock = overrideResBlock;
                            localBuildResSet = buildArgSet.pOverrideBuildResSet;
                        }
                        else
                        {
                            localResBlock = resBlock;
                            localBuildResSet = &buildResSet;
                        }
                        const res::Parts* pResParts = static_cast<const res::Parts*>(localResBlock);
                        PartsBuildDataSet localPartsBuildDataSet(partsPane, pResParts, localBuildResSet, &static_cast<const res::Parts*>(resBlock)->size);
                        // 部品のレイアウトを構築
                        Layout* layout = BuildPartsLayout(
                            internal::ConvertOffsToPtr<char>(localResBlock, sizeof(res::Parts) + sizeof(res::PartsProperty) * pResParts->propertyNum),
                            localPartsBuildDataSet, buildArgSet);
                        if (layout) {
                            // 作成したレイアウトをパーツペインにセット
                            partsPane->SetLayout(layout);
                            // 部品のレイアウトのペインリストに登録
                            m_PartsPaneList.PushBack(partsPane);
                        }
                        // BuildPartsLayoutがNULLを返したときは、部品としてこのレイアウトにくっつける必要がなかったということになる
                    }

                    pLastPane = pPane;
                }
            }
            break;

        case res::DATABLOCKKIND_PANEBEGIN:
            NW_ASSERT_NOT_NULL(pLastPane);
            buildArgSet.pParentPane = pLastPane;    // 最後に作成したペインを親とする
            break;
        case res::DATABLOCKKIND_PANEEND:
            pLastPane = buildArgSet.pParentPane;    // 親ペインを最後に作成したペインとする
            buildArgSet.pParentPane = pLastPane->GetParent();
            break;
        case res::DATABLOCKKIND_GROUP:
            if (! bReadRootGroup)   // グループのルートに初めて到達
            {
                bReadRootGroup = true;
                this->SetGroupContainer(NewObj<GroupContainer>());
            }
            else
            {
                if (this->GetGroupContainer() && groupNestLevel == 1)
                {
                    if (Group* pGroup = NewObj<Group>(reinterpret_cast<const res::Group*>(pDataBlockHead), this->GetRootPane()))
                    {
                        this->GetGroupContainer()->AppendGroup(pGroup);
                    }
                }
            }
            break;
        case res::DATABLOCKKIND_GROUPBEGIN:
            groupNestLevel++;
            break;
        case res::DATABLOCKKIND_GROUPEND:
            groupNestLevel--;
            break;
        default:
            break;
        }

        dataPtr = static_cast<const u8*>(dataPtr) + pDataBlockHead->size;    // 次のブロック位置へ
    }

    if (buildArgSet.isShareTextBoxCharAttributeBuffer && ! IsPartsLayout() && m_MaxCharAttributeNum > 0)
    {
        const u32 size = sizeof(nw::font::internal::CharAttribute) * m_MaxCharAttributeNum;
        m_pSharedCharAttrs = static_cast<nw::font::internal::CharAttribute*>(Layout::AllocMemory(size));
    }

    return true;
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic()
{
    return CreateAnimTransform<AnimTransformBasic>();
}

void
Layout::DeleteAnimTransform(AnimTransform *pAnimTransform)
{
    NW_ASSERTMSG(pAnimTransform, "pAnimTransform must not be NULL for Layout[%s]", GetName());

    this->GetAnimTransformList().erase(pAnimTransform);
    DeleteObj(pAnimTransform);
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic(
    const void* animResBuf)
{
    return CreateAnimTransform<AnimTransformBasic>(animResBuf);
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic(
    const AnimResource& animRes)
{
    return CreateAnimTransform<AnimTransformBasic>(animRes);
}

AnimTransformBasic*
Layout::CreateAnimTransformBasic(
    const char* tagName)
{
    return CreateAnimTransform<AnimTransformBasic>(tagName);
}

PaneAnimator* Layout::CreatePaneAnimator(const char* tagName, Pane* pane, bool enable /* = true */)
{
    const void* resData = GetAnimResourceData(tagName);
    NW_ASSERTMSG(resData, "cannot find resource of tagName[%s] pane[%s] Layout[%s]", tagName, pane->GetName(), GetName());
    PaneAnimator* animator = CreateAnimTransform<PaneAnimator>(resData);
    animator->Setup(pane, enable);
    return animator;
}

GroupAnimator* Layout::CreateGroupAnimator(const char* tagName, Group* group, bool enable /* = true */)
{
    const void* resData = GetAnimResourceData(tagName);
    NW_ASSERTMSG(resData, "cannot find resource of tagName[%s] group[%s] Layout[%s]", tagName, group->GetName(), GetName());
    GroupAnimator* animator = CreateAnimTransform<GroupAnimator>(resData);
    animator->Setup(group, enable);
    return animator;
}

GroupAnimator* Layout::CreateGroupAnimatorWithIndex(const char* tagName, u32 groupIndex, bool enable /* = true */)
{
    const void* resData = GetAnimResourceData(tagName);
    NW_ASSERTMSG(resData, "cannot find resource of tagName[%s] groupIndex[%d] Layout[%s]", tagName, groupIndex, GetName());
    AnimResource animRes(resData);
    GroupAnimator* animator = CreateAnimTransform<GroupAnimator>(animRes);
    animator->Setup(animRes, m_pGroupContainer, groupIndex, enable);
    return animator;
}

GroupArrayAnimator* Layout::CreateGroupArrayAnimator(const AnimResource& animRes, bool enable)
{
    // newの回数を減らすためにCreateAnimTransformをばらして使う
    NW_ASSERTMSG(s_pAllocator, "s_pAllocator must not be NULL for Layout[%s]", GetName());

    const res::AnimationBlock *const pAnimBlock = animRes.GetResourceBlock();

    if (! pAnimBlock)
    {
        return NULL;
    }
    // グループが設定されていない場合は構築に失敗してNULLを返す
    if (animRes.GetGroupNum() == 0)
    {
        return NULL;
    }
    void* ptr = Layout::AllocMemory(sizeof(GroupArrayAnimator) + animRes.GetGroupNum() * sizeof(Group*));
    if (ptr == NULL) {
        return NULL;
    }
    GroupArrayAnimator* animator = new(ptr) GroupArrayAnimator();
    animator->SetResource(pAnimBlock, m_ResourceAccessor);
    this->GetAnimTransformList().PushBack(animator);

    animator->Setup(animRes, m_pGroupContainer, internal::ConvertOffsToPtr<Group*>(ptr, sizeof(GroupArrayAnimator)), enable);
    return animator;
}

GroupArrayAnimator* Layout::CreateGroupArrayAnimator(const char* tagName, bool enable /* = true */)
{
    const void* resData = GetAnimResourceData(tagName);
    NW_ASSERTMSG(resData, "cannot find resource of tagName[%s] Layout[%s]", tagName, GetName());
    AnimResource animRes(resData);
    return CreateGroupArrayAnimator(animRes, enable);
}

Animator* Layout::CreateGroupAnimatorAuto(const char* tagName, bool enable /* = true */)
{
    const void* resData = GetAnimResourceData(tagName);
    NW_ASSERTMSG(resData, "cannot find resource of tagName[%s] Layout[%s]", tagName, GetName());
    AnimResource animRes(resData);

    if (animRes.GetGroupNum() > 1)
    {
        return CreateGroupArrayAnimator(animRes, enable);
    }
    else
    {
        GroupAnimator* animator = CreateAnimTransform<GroupAnimator>(animRes);
        animator->Setup(animRes, m_pGroupContainer, 0, enable);
        return animator;
    }
}

void
Layout::BindAnimation(AnimTransform* pAnimTrans)
{
    if (this->GetRootPane())
    {
        pAnimTrans->BindPane(this->GetRootPane(), true);
    }
}

void
Layout::UnbindAnimation(AnimTransform* pAnimTrans)
{
    pAnimTrans->UnbindAll();
}

void
Layout::UnbindAnimation(Pane* pPane)
{
    AnimTransformList::Iterator it_end = m_AnimTransList.GetEndIter();
    for (AnimTransformList::Iterator it = m_AnimTransList.GetBeginIter(); it != it_end; ++it)
    {
        it->UnbindPane(pPane);
    }
}

void
Layout::UnbindAllAnimation()
{
    AnimTransformList::Iterator it_end = m_AnimTransList.GetEndIter();
    for (AnimTransformList::Iterator it = m_AnimTransList.GetBeginIter(); it != it_end; ++it)
    {
        it->UnbindAll();
    }
}

bool
Layout::BindAnimationAuto(
    const AnimResource& animRes
)
{
    if (! this->GetRootPane())
    {
        return false;
    }

    if (! animRes.GetResourceBlock())
    {
        return false;
    }

    AnimTransform *const pAnimTrans = CreateAnimTransformBasic();  // AnimTransform オブジェクトの作成
    if (pAnimTrans == NULL)
    {
        NW_WARNING(false, "Create AnimTransform failed.");
        return false;
    }

    bool bResult = true;

    // 最初に名前によるバインドを一通り行う
    const u16 bindGroupNum = animRes.GetGroupNum();

    u16 animNum = 0;
    if (bindGroupNum == 0)  // バインドする対象を限定していない
    {
        animNum = animRes.GetResourceBlock()->animContNum;

        // バインドするアニメーション数を明示的に指定してリソースをセットします。
        pAnimTrans->SetResource(animRes.GetResourceBlock(), m_ResourceAccessor, animNum);

        const bool bRecursive = true;
        this->GetRootPane()->BindAnimation(pAnimTrans, bRecursive, false/* bEnable */);
    }
    else    // グループを指定してのバインド
    {
        const res::AnimationGroupRef *const groupRefs = animRes.GetGroupArray();
        for (int grpIdx = 0; grpIdx < bindGroupNum; ++grpIdx)
        {
            // グループにあるペイン全てで必要になるアニメーションの個数を数えます。
            Group *const pGroup = this->GetGroupContainer()->FindGroupByName(groupRefs[grpIdx].GetName());
            if (pGroup == NULL)
            {
                NW_WARNING(false, "Group not found: %s", groupRefs[grpIdx].GetName());
                bResult = false;
                continue;
            }

            animNum += animRes.CalcAnimationNum(pGroup, animRes.IsDescendingBind());
        }

        // バインドするアニメーション数を明示的に指定してリソースをセットします。
        pAnimTrans->SetResource(animRes.GetResourceBlock(), m_ResourceAccessor, animNum);

        for (int grpIdx = 0; grpIdx < bindGroupNum; ++grpIdx)
        {
            // グループにあるペイン全てで必要になるアニメーションの個数を数えます。
            Group *const pGroup = this->GetGroupContainer()->FindGroupByName(groupRefs[grpIdx].GetName());
            if (pGroup == NULL)
            {
                continue;
            }

            // アニメーションをバインドします。
            nw::lyt::BindAnimation(pGroup, pAnimTrans, false/* bEnable */);
        }
    }

    const u16 animShareInfoNum = animRes.GetAnimationShareInfoNum();
    if (animShareInfoNum > 0)   // アニメーション共有によるバインド
    {
        const res::AnimationShareInfo *const animShareInfoAry = animRes.GetAnimationShareInfoArray();
        NW_ASSERT_NOT_NULL(animShareInfoAry);

        for (int i = 0; i < animShareInfoNum; ++i)
        {
            Pane *const pSrcPane = this->GetRootPane()->FindPaneByName(animShareInfoAry[i].GetSrcPaneName());
            if (pSrcPane == NULL)
            {
                NW_WARNING(false, "Source pane of animation-share is not found: %s", animShareInfoAry[i].GetSrcPaneName());
                bResult = false;
                continue;
            }

            internal::AnimPaneTree animPaneTree(pSrcPane, animRes);
            if (! animPaneTree.IsEnabled()) // 共有元ペインにアニメーションが無い?
            {
                continue;
            }

            Group *const pGroup = this->GetGroupContainer()->FindGroupByName(animShareInfoAry[i].GetTargetGroupName());
            if (pGroup == NULL)
            {
                NW_WARNING(false, "Target group of animation-share is not found: %s", animShareInfoAry[i].GetTargetGroupName());
                bResult = false;
                continue;
            }

            PaneLinkList& paneList = pGroup->GetPaneList();
            u32 animIdx = 0;
            for (PaneLinkList::Iterator it = paneList.GetBeginIter(); it != paneList.GetEndIter(); ++it, ++animIdx)
            {
                if (it->target != pSrcPane)
                {
                    if (bindGroupNum > 0)   // バインド対象の指定があり
                    {
                        // 関連グループに含まれてない場合は共有しない
                        const bool bInclude = IsIncludeAnimationGroupRef(
                            this->GetGroupContainer(),
                            animRes.GetGroupArray(),
                            bindGroupNum,
                            animRes.IsDescendingBind(),
                            it->target);

                        if (! bInclude)
                        {
                            continue;
                        }
                    }

                    // srcPaneNameと異なるペインに対してアニメーションをバインド
                    animPaneTree.Bind(this, it->target, m_ResourceAccessor);
                }
            }
        }
    }

    return bResult;
}

void
Layout::CalculateMtx(const DrawInfo& drawInfo, bool forceGlbMtxDirty /* = false */)
{
    NW_PROFILE("nw::lyt::Layout::CalculateMtx");

    if (! this->GetRootPane())
    {
        return;
    }

    Pane::CalculateMtxContext context(drawInfo, this);
    this->GetRootPane()->CalculateMtx(context, forceGlbMtxDirty);
}

void
Layout::Draw(DrawInfo& drawInfo)
{
    NW_PROFILE("nw::lyt::Layout::Draw");

    if (! this->GetRootPane())
    {
        return;
    }

    drawInfo.ResetGlState();
    drawInfo.ResetGlProgramState();

    drawInfo.SetLayout(this);

    this->GetRootPane()->Draw(drawInfo);

    drawInfo.SetLayout(0);
}

void
Layout::Animate()
{
    NW_PROFILE("nw::lyt::Layout::Animate");

    {
        AnimTransformList::Iterator it_end = m_AnimTransList.GetEndIter();
        for (AnimTransformList::Iterator it = m_AnimTransList.GetBeginIter(); it != it_end; ++it)
        {
            it->Animate();
        }
    }

    // 部品のレイアウトについても呼び出す
    {
        PartsPaneList::Iterator it_end = m_PartsPaneList.GetEndIter();
        for (PartsPaneList::Iterator it = m_PartsPaneList.GetBeginIter(); it != it_end; ++it)
        {
            it->GetLayout()->Animate();
        }
    }
}

void
Layout::UpdateAnimFrame(f32 progress_frame)
{
    {
        AnimTransformList::Iterator it_end = m_AnimTransList.GetEndIter();
        for (AnimTransformList::Iterator it = m_AnimTransList.GetBeginIter(); it != it_end; ++it)
        {
            it->UpdateFrame(progress_frame);
        }
    }

    // 部品のレイアウトについても呼び出す
    {
        PartsPaneList::Iterator it_end = m_PartsPaneList.GetEndIter();
        for (PartsPaneList::Iterator it = m_PartsPaneList.GetBeginIter(); it != it_end; ++it)
        {
            it->GetLayout()->UpdateAnimFrame(progress_frame);
        }
    }
}

void
Layout::AnimateAndUpdateAnimFrame(f32 progress_frame)
{
    NW_PROFILE("nw::lyt::Layout::AnimateAndUpdateAnimFrame");

    {
        AnimTransformList::Iterator it_end = m_AnimTransList.GetEndIter();
        for (AnimTransformList::Iterator it = m_AnimTransList.GetBeginIter(); it != it_end; ++it)
        {
            it->Animate();
            it->UpdateFrame(progress_frame);
        }
    }

    // 部品のレイアウトについても呼び出す
    {
        PartsPaneList::Iterator it_end = m_PartsPaneList.GetEndIter();
        for (PartsPaneList::Iterator it = m_PartsPaneList.GetBeginIter(); it != it_end; ++it)
        {
            it->GetLayout()->AnimateAndUpdateAnimFrame(progress_frame);
        }
    }
}

const ut::Rect
Layout::GetLayoutRect() const
{
    return ut::Rect(- m_LayoutSize.width / 2, m_LayoutSize.height / 2, m_LayoutSize.width / 2, - m_LayoutSize.height / 2);
}

void
Layout::SetTagProcessor(font::TagProcessorBase<char16>* pTagProcessor)
{
    SetTagProcessorImpl(this->GetRootPane(), pTagProcessor);
}

Parts*
Layout::FindPartsPaneByName(const char* findName)
{
    PartsPaneList::Iterator it_end = m_PartsPaneList.GetEndIter();
    for (PartsPaneList::Iterator it = m_PartsPaneList.GetBeginIter(); it != it_end; ++it)
    {
        if (internal::EqualsResName(findName, it->GetName()))
        {
            return &(*it);
        }
    }

    return NULL;
}

#if defined(NW_PLATFORM_CAFE)
const ArchiveShaderInfo*
Layout::GetArchiveShader(const char* name)
{
    return m_ResourceAccessor->GetShader(name);
}
#endif


Pane*
Layout::BuildPaneObj(
    ut::SigWord        kind,
    const void*        pBlock,
    const void*        pOverrideBlock,
    const BuildArgSet& buildArgSet
)
{
    switch (kind)
    {
    case res::DATABLOCKKIND_PANE:
        {
            const res::Pane* pResPane = static_cast<const res::Pane*>(pBlock);
            return NewObj<Pane, const res::Pane*, const BuildArgSet&>(pResPane, buildArgSet);
        }
    case res::DATABLOCKKIND_PICTURE:
        {
            const res::Picture* pResPic = static_cast<const res::Picture*>(pBlock);
            const res::Picture* pResPicOverride = static_cast<const res::Picture*>(pOverrideBlock);
            return NewObj<Picture, const res::Picture*, const res::Picture*, const BuildArgSet&>(pResPic, pResPicOverride, buildArgSet);
        }
    case res::DATABLOCKKIND_TEXTBOX:
        {
            const res::TextBox* pResTextBox = static_cast<const res::TextBox*>(pBlock);
            const res::TextBox* pResTextBoxOverride = static_cast<const res::TextBox*>(pOverrideBlock);
            TextBox::InitStringParam initStringParam;
            TextBox* textBox = NewObj<TextBox, const res::TextBox*, const res::TextBox*, const BuildArgSet&>(pResTextBox, pResTextBoxOverride, buildArgSet, &initStringParam);
            textBox->InitString(buildArgSet, initStringParam);
            return textBox;
        }
    case res::DATABLOCKKIND_WINDOW:
        {
            const res::Window* pResWindow = static_cast<const res::Window*>(pBlock);
            const res::Window* pResWindowOverride = static_cast<const res::Window*>(pOverrideBlock);
            return NewObj<Window, const res::Window*, const res::Window*, const BuildArgSet&>(pResWindow, pResWindowOverride, buildArgSet);
        }
    case res::DATABLOCKKIND_BOUNDING:
        {
            const res::Bounding* pResBounding = static_cast<const res::Bounding*>(pBlock);
            const res::Bounding* pResBoundingOverride = static_cast<const res::Bounding*>(pOverrideBlock);
            return NewObj<Bounding, const res::Bounding*, const res::Bounding*, const BuildArgSet&>(pResBounding, pResBoundingOverride, buildArgSet);
        }
    case res::DATABLOCKKIND_PARTS:
        {
            const res::Parts* pResParts = static_cast<const res::Parts*>(pBlock);
            const res::Parts* pResPartsOverride = static_cast<const res::Parts*>(pOverrideBlock);
            return NewObj<Parts, const res::Parts*, const res::Parts*, const BuildArgSet&>(pResParts, pResPartsOverride, buildArgSet);
        }
    default:
        NW_ASSERTMSG(false, "unknown data type");
        break;
    }

    return 0;
}

Layout* Layout::BuildPartsLayout(const char* name, const PartsBuildDataSet& partsBuildDataSet, const BuildArgSet& buildArgSet)
{
    const void* res = GetLayoutResourceData(name);
    NW_ASSERTMSG(res, "cannot find resource of parts Layout[%s] in body Layout[%s]", name, GetName());

    // レイアウトオブジェクトの作成
    Layout* layout = NewObj<Layout>();

    // レイアウトのビルド
    layout->BuildImpl(res, m_ResourceAccessor, buildArgSet, &partsBuildDataSet);

    return layout;
}

const void* Layout::GetAnimResourceData(const char* tagName)
{
    char brlan_path[LAYOUT_NAME_MAX + AnimTagNameStrMax + 8];
    nw::ut::snprintf(brlan_path, sizeof(brlan_path), sizeof(brlan_path) - 1, "%s_%s.bflan", m_Name, tagName);

    return m_ResourceAccessor->GetResource(res::RESOURCETYPE_ANIMATION, brlan_path);
}

const void* Layout::GetLayoutResourceData(const char* layoutName)
{
    char bclyt_path[LAYOUT_NAME_MAX + 8];
    nw::ut::snprintf(bclyt_path, sizeof(bclyt_path), sizeof(bclyt_path) - 1, "%s.bflyt", layoutName);

    return m_ResourceAccessor->GetResource(res::RESOURCETYPE_LAYOUT, bclyt_path);
}

Layout::PartsBuildDataSet::PartsBuildDataSet(Parts* partsPane, const res::Parts* resParts, const BuildResSet* buildResSet, const res::Vec2* originalSize)
 : m_PropertyNum(resParts->propertyNum)
 , m_PropertyTable(internal::ConvertOffsToPtr<res::PartsProperty>(resParts, sizeof(res::Parts)))
 , m_PartsPane(partsPane)
 , m_ResParts(resParts)
 , m_PropertyBuildResSet(buildResSet)
 , m_Magnify((partsPane->GetSize().width / originalSize->x) * resParts->magnify.x, (partsPane->GetSize().height / originalSize->y) * resParts->magnify.y)
{
}

const res::PartsProperty* Layout::PartsBuildDataSet::FindPartsPropertyFromName(const char* name) const
{
    for (u32 i = 0; i < m_PropertyNum; i++) {
        if (internal::EqualsResName(name, m_PropertyTable[i].name))
        {
            return &m_PropertyTable[i];
        }
    }
    return NULL;
}

const void* Layout::PartsBuildDataSet::GetPropertyResBlock(const res::PartsProperty* prop) const
{
    if (prop->propertyOffset == 0)
    {
        return NULL;
    }
    else
    {
        return internal::ConvertOffsToPtr<void>(m_ResParts, prop->propertyOffset);
    }
}

const res::ExtUserDataList* Layout::PartsBuildDataSet::GetExtUserDataListResBlock(bool* isOverride, const res::PartsProperty* prop) const
{
    if (prop->extUserDataOffset == EXTUSERDATAOVERRIDEOFFSET_NO_OVERRIDE)
    {
        // 上書きしない
        *isOverride = false;
        return NULL;
    }
    else if (prop->extUserDataOffset == EXTUSERDATAOVERRIDEOFFSET_NO_DATA)
    {
        // 上書きするが、データがない
        *isOverride = true;
        return NULL;
    }
    else
    {
        // 上書きする。データのリストを返す
        *isOverride = true;
        return static_cast<const res::ExtUserDataList*>(internal::ConvertOffsToPtr<void>(m_ResParts, prop->extUserDataOffset));
    }
}

const res::PartsPaneBasicInfo* Layout::PartsBuildDataSet::GetPartsPaneBasicInfoResBlock(const res::PartsProperty* prop) const
{
    if (prop->paneBasicInfoOffset == 0)
    {
        return NULL;
    }
    else
    {
        return static_cast<const res::PartsPaneBasicInfo*>(internal::ConvertOffsToPtr<void>(m_ResParts, prop->paneBasicInfoOffset));
    }
}

} // namespace nw::lyt
} // namespace nw
