﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/lyt/lyt_GraphicsResource.h>
#include <nw/lyt/lyt_Common.h>
#include <nw/lyt/lyt_Layout.h>
#include <nw/gfnd/gfnd_Memory.h>
#include <nw/ut/ut_Inlines.h>

#include <nw/dev/dev_Profile.h>

namespace shhelp = nw::gfnd::shader_helper;

//########################################
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
//########################################
#include    <cstdlib>

namespace {

//------------------------------------------------------------------------------
const char cVertexShaderSource[] =
{
#include "../../../../Resources/LytShaders/lyt_BuildinShader_gl.vsh"
};
//------------------------------------------------------------------------------
const char cPixelShaderSource[] =
{
#include "../../../../Resources/LytShaders/lyt_BuildinShader_gl.psh"
};

//------------------------------------------------------------------------------
bool createShader_( GLuint shader, const char* source )
{
    GLint length = strlen(source) + 1;
    // コンパイル.
    glShaderSource( shader, 1, &source, &length );
    glCompileShader( shader );

    // 結果をチェック.
    GLint result;
    glGetShaderiv( shader, GL_COMPILE_STATUS, &result );
    if ( result != GL_TRUE )
    {
        int infologLength = 0;
        glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &infologLength);

        if (0 < infologLength)
        {
            const int MAX_LOG_LENGTH = 1024;
            char infoLog[MAX_LOG_LENGTH];
            int charsWritten  = 0;
            glGetShaderInfoLog(shader, nw::ut::Min(infologLength, MAX_LOG_LENGTH), &charsWritten, infoLog);

            NW_LOG(infoLog);
        }

        return false;
    }
    return true;
}
//------------------------------------------------------------------------------
}   // namespace

namespace nw
{
namespace lyt
{

const GLushort
GraphicsResource::s_VertexIndex[VBO_INDEX_COUNT] =
{
    VERTEX_RT, VERTEX_LT, VERTEX_RB, VERTEX_LB
};
const GLshort
GraphicsResource::s_UniformVertexIndex[VERTEX_MAX][VERTEXATTRSIZE_INDEX] =
{
    { 0, 0 }, // LT
    { 1, 0 }, // RT
    { 0, 1 }, // LB
    { 1, 1 }, // RB
};

//----------------------------------------
void GraphicsResource::Setup(u32 charMax)
{
    // 描画用バッファの確保と設定
    {
        const u32 vtxBufferSize = nw::font::RectDrawer::GetWorkBufferSize(charMax);
        m_FontDrawer.Initialize(Layout::AllocMemory(vtxBufferSize), charMax);
    }

    {
        // VS
        {
            NW_ASSERT( m_VertexShader == 0 );
            m_VertexShader = glCreateShader( GL_VERTEX_SHADER );
            NW_GL_ASSERT();
            bool ret = createShader_( m_VertexShader, cVertexShaderSource );
            if ( !ret )
            {
                NW_ERR( "Can't create VertexShader." );
                glDeleteShader( m_VertexShader );
                m_VertexShader = 0;
                return;
            }
        }
        // FS
        {
            NW_ASSERT( m_FragmentShader == 0 );
            m_FragmentShader = glCreateShader( GL_FRAGMENT_SHADER );
            NW_GL_ASSERT();

            bool ret = createShader_( m_FragmentShader, cPixelShaderSource );
            if ( !ret )
            {
                NW_ERR( "Can't create FragmentShader." );
                glDeleteShader( m_FragmentShader );
                m_FragmentShader = 0;
                glDeleteShader( m_VertexShader );
                m_VertexShader = 0;
                return;
            }
        }
        // リンク.
        {
            NW_ASSERT( m_GlProgram == 0 );
            m_GlProgram = glCreateProgram();
            glAttachShader( m_GlProgram, m_VertexShader );
            glAttachShader( m_GlProgram, m_FragmentShader );
            NW_GL_ASSERT();
            glBindAttribLocation(m_GlProgram, VERTEXATTR_VERTEX_INDEX, "aVertexIndex");
            NW_GL_ASSERT();
            glLinkProgram( m_GlProgram );
            GLint ret;
            glGetProgramiv( m_GlProgram, GL_LINK_STATUS, &ret );
            NW_GL_ASSERT();
            if ( ret == GL_FALSE )
            {
                // heap上に文字列を確保してエラー表示.
#if !defined(NW_RELEASE)
                GLint result;
                glGetProgramiv( m_GlProgram, GL_INFO_LOG_LENGTH, &result );
                char* buffer = static_cast<char*>(Layout::AllocMemory(result));
                glGetProgramInfoLog( m_GlProgram, result, &result, buffer );
                NW_ASSERTMSG(false, "%s", buffer);
                Layout::FreeMemory(buffer);
#endif
                glDeleteProgram( m_GlProgram );
                m_GlProgram = 0;
                glDeleteShader( m_FragmentShader );
                m_FragmentShader = 0;
                glDeleteShader( m_VertexShader );
                m_VertexShader = 0;
                return;
            }
        }
        // ユニフォーム/頂点属性位置のキャッシュを取得.
        {
            m_UniformIds[UNIFORM_uProjection] = glGetUniformLocation( m_GlProgram, "uProjection" );
            m_UniformIds[UNIFORM_uModelView] = glGetUniformLocation( m_GlProgram, "uModelView" );
            m_UniformIds[UNIFORM_uTransform] = glGetUniformLocation( m_GlProgram, "uTransform" );
            m_UniformIds[UNIFORM_uTexMtx0_xz] = glGetUniformLocation( m_GlProgram, "uTexMtx0_xz" );
            m_UniformIds[UNIFORM_uTexMtx0_yw] = glGetUniformLocation( m_GlProgram, "uTexMtx0_yw" );
            m_UniformIds[UNIFORM_uTexMtx1_xz] = glGetUniformLocation( m_GlProgram, "uTexMtx1_xz" );
            m_UniformIds[UNIFORM_uTexMtx1_yw] = glGetUniformLocation( m_GlProgram, "uTexMtx1_yw" );
            m_UniformIds[UNIFORM_uTexMtx2_xz] = glGetUniformLocation( m_GlProgram, "uTexMtx2_xz" );
            m_UniformIds[UNIFORM_uTexMtx2_yw] = glGetUniformLocation( m_GlProgram, "uTexMtx2_yw" );
            m_UniformIds[UNIFORM_uColor] = glGetUniformLocation( m_GlProgram, "uColor" );
            m_UniformIds[UNIFORM_uVertexColor] = glGetUniformLocation( m_GlProgram, "uVertexColor" );
            m_UniformIds[UNIFORM_uFrameSpec] = glGetUniformLocation( m_GlProgram, "uFrameSpec" );
            m_UniformIds[UNIFORM_uFrameSize] = glGetUniformLocation( m_GlProgram, "uFrameSize" );
            m_UniformIds[UNIFORM_uPaneSize] = glGetUniformLocation( m_GlProgram, "uPaneSize" );
            m_UniformIds[UNIFORM_uVertexTexCoord0] = glGetUniformLocation( m_GlProgram, "uVertexTexCoord0" );
            m_UniformIds[UNIFORM_uVertexTexCoord1] = glGetUniformLocation( m_GlProgram, "uVertexTexCoord1" );
            m_UniformIds[UNIFORM_uVertexTexCoord2] = glGetUniformLocation( m_GlProgram, "uVertexTexCoord2" );
            m_UniformIds[UNIFORM_uGeneratingTexCoord] = glGetUniformLocation( m_GlProgram, "uGeneratingTexCoord" );
            m_UniformIds[UNIFORM_uRcpTexSize0] = glGetUniformLocation( m_GlProgram, "uRcpTexSize0" );
            m_UniformIds[UNIFORM_uTextureMode] = glGetUniformLocation( m_GlProgram, "uTextureMode" );
            m_UniformIds[UNIFORM_uInterpolateWidth] = glGetUniformLocation( m_GlProgram, "uInterpolateWidth" );
            m_UniformIds[UNIFORM_uInterpolateOffset] = glGetUniformLocation( m_GlProgram, "uInterpolateOffset" );
            m_UniformIds[UNIFORM_uIndirectMtx0] = glGetUniformLocation( m_GlProgram, "uIndirectMtx0" );
            m_UniformIds[UNIFORM_uIndirectMtx1] = glGetUniformLocation( m_GlProgram, "uIndirectMtx1" );
            m_UniformIds[UNIFORM_uTexture0] = glGetUniformLocation( m_GlProgram, "uTexture0" );
            m_UniformIds[UNIFORM_uTexture1] = glGetUniformLocation( m_GlProgram, "uTexture1" );
            m_UniformIds[UNIFORM_uTexture2] = glGetUniformLocation( m_GlProgram, "uTexture2" );
        }
    }

    NW_ASSERT(m_GlProgram != 0);

    InitVBO();

    NW_GL_ASSERT();

    m_Initialized = true;
}

//----------------------------------------
void
GraphicsResource::InitVBO()
{
    // VAOの無効化
    glBindVertexArray(0);

    glGenBuffers(this->VBO_MAX, m_GlVertexBufferObject);
    NW_GL_ASSERT();

    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->GetVBO(this->VBO_ELEMENT));
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(s_VertexIndex), s_VertexIndex, GL_STATIC_DRAW);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
    NW_GL_ASSERT();

    glBindBuffer(GL_ARRAY_BUFFER, this->GetVBO(this->VBO_VERTEX_INDEX));
    glBufferData(GL_ARRAY_BUFFER, sizeof(s_UniformVertexIndex), s_UniformVertexIndex, GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    NW_GL_ASSERT();
}

//----------------------------------------
void
GraphicsResource::ActiveVBO()
{
    // VAOの無効化
    glBindVertexArray(0);

    // 頂点インデックス
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->GetVBO(this->VBO_ELEMENT));

    // 頂点座標
    glBindBuffer(GL_ARRAY_BUFFER, this->GetVBO(this->VBO_VERTEX_INDEX));
    glEnableVertexAttribArray(VERTEXATTR_VERTEX_INDEX);
    glVertexAttribPointer(VERTEXATTR_VERTEX_INDEX, VERTEXATTRSIZE_INDEX, GL_SHORT, GL_FALSE, 0, NULL);
}

} // namespace nw::lyt
} // namespace nw

//########################################
#elif defined(NW_PLATFORM_CAFE)
//########################################

#include <nw/lyt/lyt_ShaderConnection.h>

namespace nw
{
namespace lyt
{

//----------------------------------------
u32 ShaderSlotDescription::EstimateBufferSize()
{
    NW_ASSERT(shaderEnabled[DEFAULT_SHADER]);
    // TODO: メモリサイズ計算が合わないのを修正する。(ひとまず 128 を加算してあります)
    requiredBufferSize = gfnd::FrameHeapAllocator::MANAGER_WORK_AREA_SIZE + 128;

    shaderQuantity = std::count(shaderEnabled, shaderEnabled + ut::GetArrayLength(shaderEnabled), true);
    for (int i = 0; i < shaderQuantity; ++i)
    {
        requiredBufferSize = ut::RoundUp(requiredBufferSize, 4) + sizeof(ShaderConnection);
        requiredBufferSize = ut::RoundUp(requiredBufferSize, 4) + sizeof(GX2AttribStream) * maxAttributeStreamQuantity;
        requiredBufferSize = ut::RoundUp(requiredBufferSize, 4) + sizeof(gfnd::ShaderUniformId) * maxUniformQuantity;
#if defined(NW_DISPLAY_LIST_ENABLED)
        requiredBufferSize = ut::RoundUp(requiredBufferSize, gfnd::DisplayList::ALIGNMENT) + displayListCapacity;
#endif
    }

    return requiredBufferSize;
}

//----------------------------------------
void* ShaderSlot::Shutdown()
{
   std::fill_n(
        m_Connections,
        ut::GetArrayLength(m_Connections),
        static_cast<ShaderConnection*>(NULL));
    void* buffer = m_Buffer;
    m_Buffer = NULL;
    return buffer;
}

//----------------------------------------
void
ShaderSlot::Initialize(const ShaderSlotDescription& description, void* workBuffer)
{
    NW_ASSERT(m_Buffer == NULL);
    *const_cast<ShaderSlotDescription*>(&m_Description) = description;
    m_Buffer = workBuffer;

    NW_ASSERT(description.shaderEnabled[DEFAULT_SHADER]);
    NW_ASSERT(description.requiredBufferSize != 0);
    gfnd::FrameHeapAllocator allocator;
    allocator.Initialize(workBuffer, description.requiredBufferSize);

    NW_ASSERT(description.shaderQuantity == std::count(description.shaderEnabled, description.shaderEnabled + ut::GetArrayLength(description.shaderEnabled), true));
    for (int id = 0; id < ut::GetArrayLength(m_Connections); ++id)
    {
        ShaderConnection*& connection = m_Connections[id];
        if (description.shaderEnabled[id])
        {
            void* connectionMemory = allocator.Allocate(sizeof(ShaderConnection));
            connection = new(connectionMemory) ShaderConnection;

            connection->m_Attributes = static_cast<GX2AttribStream*>(
                allocator.Allocate(sizeof(GX2AttribStream) * description.maxAttributeStreamQuantity));
            NW_ASSERT_NOT_NULL(connection->m_Attributes);
            std::uninitialized_fill_n(connection->m_Attributes, description.maxAttributeStreamQuantity, GX2AttribStream());

            connection->m_UniformIds = static_cast<gfnd::ShaderUniformId*>(
                allocator.Allocate(sizeof(gfnd::ShaderUniformId) * description.maxUniformQuantity));
            NW_ASSERT_NOT_NULL(connection->m_UniformIds);
            std::uninitialized_fill_n(connection->m_UniformIds, description.maxUniformQuantity, GX2_UNIFORM_VAR_INVALID_OFFSET);

#if defined(NW_DISPLAY_LIST_ENABLED)
            void* memory = allocator.Allocate(description.displayListCapacity, gfnd::DisplayList::ALIGNMENT);
            connection->m_ShaderDisplayList.Initialize(memory, description.displayListCapacity);
#endif
        }
        else
        {
            if (SHADER_ID_QUANTITY <= id && description.shaderEnabled[id - SHADER_ID_QUANTITY])
            {
                connection = m_Connections[id - SHADER_ID_QUANTITY];
            }
            else
            {
                connection = m_Connections[DEFAULT_SHADER];
            }
        }
    }
}

//----------------------------------------
ShaderConnection*
ShaderSlot::Connect(ShaderId id, GX2VertexShader* vertexShader, GX2PixelShader* pixelShader)
{
    NW_ASSERT_MAXLT(id, ut::GetArrayLength(m_Connections));
    NW_ASSERT_NOT_NULL(vertexShader);
    NW_ASSERT_NOT_NULL(pixelShader);
    ShaderConnection* connection = m_Connections[id];

    connection->m_VertexShader = vertexShader;
    connection->m_PixelShader = pixelShader;

    connection->m_IsMultiTextureEnabled =
        (id == NULL_TEXTURE_SHADER || id == SINGLE_TEXTURE_SHADER)
        ? false
        : true;

    return connection;
}

//----------------------------------------
void
ShaderSlot::MakeDisplayList(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);

    connection->m_ShaderDisplayList.Clear();
    NW_MAKE_DISPLAY_LIST(cache, connection->m_ShaderDisplayList)
    {
        GX2SetShaders(
            &connection->m_FetchShader,
            connection->m_VertexShader,
            connection->m_PixelShader);
    }
}

//----------------------------------------
ShaderConnection*
ShaderSlot::GetConnection(int id)
{
    NW_ASSERT_MAXLT(id, ut::GetArrayLength(m_Connections));
    ShaderConnection* connection = m_Connections[id];
    return connection;
}

//----------------------------------------
GX2VertexShader*
ShaderSlot::GetVertexShader(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);
    return connection->m_VertexShader;
}

//----------------------------------------
GX2PixelShader*
ShaderSlot::GetPixelShader(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);
    return connection->m_PixelShader;
}

//----------------------------------------
GX2FetchShader*
ShaderSlot::GetFetchShader(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);
    return &connection->m_FetchShader;
}

//----------------------------------------
GX2AttribStream*
ShaderSlot::GetAttribStreams(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);
    return connection->m_Attributes;
}

//----------------------------------------
GX2AttribStream*
ShaderSlot::GetAttribStream(ShaderConnection* connection, u32 index)
{
    NW_ASSERT_NOT_NULL(connection);
    NW_ASSERT_MAXLT(index, m_Description.maxAttributeStreamQuantity);
    return &connection->m_Attributes[index];
}

//----------------------------------------
gfnd::ShaderUniformId*
ShaderSlot::GetUniformIds(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);
    return connection->m_UniformIds;
}

//----------------------------------------
const gfnd::ShaderUniformId*
ShaderSlot::GetUniformIds(const ShaderConnection* connection) const
{
    NW_ASSERT_NOT_NULL(connection);
    return connection->m_UniformIds;
}

//----------------------------------------
gfnd::ShaderUniformId
ShaderSlot::GetUniformId(const ShaderConnection* connection, Uniform uniformId) const
{
    NW_ASSERT_NOT_NULL(connection);
    NW_ASSERT_MAXLT(uniformId, m_Description.maxUniformQuantity);
    return connection->m_UniformIds[uniformId];
}

//----------------------------------------
bool
ShaderSlot::IsMultiTextureEnabled(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);
    return connection->m_IsMultiTextureEnabled;
}

//----------------------------------------
void
ShaderSlot::SetToDevice(ShaderConnection* connection)
{
    NW_ASSERT_NOT_NULL(connection);

    NW_MAKE_DISPLAY_LIST(cache, connection->m_ShaderDisplayList)
    {
        GX2SetShaders(
            &connection->m_FetchShader,
            connection->m_VertexShader,
            connection->m_PixelShader);
    }
    connection->m_ShaderDisplayList.Call();
}

//----------------------------------------
ShaderSetupHelper::ShaderSetupHelper()
{
}

//----------------------------------------
void ShaderSetupHelper::Setup(
    GraphicsResource* graphicsResource,
    const ut::MemoryRange& shaderBinary)
{
    //----------------------------------------
    // シェーダ管理に必要な初期化を行います。
    const int VERTEX_ATTRIBUTE_QUANTITY = 1;
    nw::lyt::ShaderSlotDescription description;
    description.maxUniformQuantity = nw::lyt::UNIFORM_MAX;
    description.maxAttributeStreamQuantity = VERTEX_ATTRIBUTE_QUANTITY;

    for (int shaderId = 0; shaderId < ut::GetArrayLength(description.shaderEnabled); ++shaderId)
    {
        // nw::lyt::ShaderSlotDescription::shaderEnabled[shaderId] を true にすると
        // インデックス shaderId に対応するシェーダを生成する必要があります。
        // インデックス shaderId は nw::lyt::ShaderId に対応しています。
        //
        // shaderEnabled[DEFAULT_SHADER] は必ず true にする必要があります。
        // shaderEnabled[shaderId] が false となっているシェーダは DEFAULT_SHADER のものが
        // 使用されるようになっています。
        description.shaderEnabled[shaderId] = true;
    }

    // シェーダ管理に必要なメモリのサイズを求めて確保します。
    nw::lyt::ShaderSlot& slot = graphicsResource->GetShaderSlot();
    slot.Initialize(description, Layout::AllocMemory(description.EstimateBufferSize(), 4));

    const void* data = reinterpret_cast<const void*>(shaderBinary.Begin());

    u32 vshCount = GFDGetVertexShaderCount(data);

    int shaderIdTable[SHADER_ID_QUANTITY * SHADER_VARIATION_QUANTITY];
    if (vshCount != ut::GetArrayLength(m_ShaderPtrs))
    {
        shaderIdTable[DEFAULT_SHADER] = 0;
        shaderIdTable[NULL_TEXTURE_SHADER] = 1;
        shaderIdTable[SINGLE_TEXTURE_SHADER] = 2;
        shaderIdTable[DOUBLE_TEXTURE1_SHADER] = 3;
        shaderIdTable[DOUBLE_TEXTURE2_SHADER] = 3;
        shaderIdTable[DOUBLE_TEXTURE3_SHADER] = 3;
        shaderIdTable[DOUBLE_TEXTURE4_SHADER] = 3;
        shaderIdTable[DOUBLE_TEXTURE5_SHADER] = 4;
        shaderIdTable[DOUBLE_TEXTURE6_SHADER] = 4;
        shaderIdTable[DOUBLE_TEXTURE7_SHADER] = 4;
        shaderIdTable[DOUBLE_TEXTURE8_SHADER] = 4;
        shaderIdTable[DOUBLE_INDIRECT_TEXTURE_SHADER] = 6;

        shaderIdTable[DEFAULT_SHADER + SHADER_ID_QUANTITY] = 0 + 9;
        shaderIdTable[NULL_TEXTURE_SHADER + SHADER_ID_QUANTITY] = 1 + 9;
        shaderIdTable[SINGLE_TEXTURE_SHADER + SHADER_ID_QUANTITY] = 2 + 9;
        shaderIdTable[DOUBLE_TEXTURE1_SHADER + SHADER_ID_QUANTITY] = 3 + 9;
        shaderIdTable[DOUBLE_TEXTURE2_SHADER + SHADER_ID_QUANTITY] = 3 + 9;
        shaderIdTable[DOUBLE_TEXTURE3_SHADER + SHADER_ID_QUANTITY] = 3 + 9;
        shaderIdTable[DOUBLE_TEXTURE4_SHADER + SHADER_ID_QUANTITY] = 3 + 9;
        shaderIdTable[DOUBLE_TEXTURE5_SHADER + SHADER_ID_QUANTITY] = 4 + 9;
        shaderIdTable[DOUBLE_TEXTURE6_SHADER + SHADER_ID_QUANTITY] = 4 + 9;
        shaderIdTable[DOUBLE_TEXTURE7_SHADER + SHADER_ID_QUANTITY] = 4 + 9;
        shaderIdTable[DOUBLE_TEXTURE8_SHADER + SHADER_ID_QUANTITY] = 4 + 9;
        shaderIdTable[DOUBLE_INDIRECT_TEXTURE_SHADER + SHADER_ID_QUANTITY] = 6 + 9;
    }
    else
    {
        for (int id = 0; id < ut::GetArrayLength(m_ShaderPtrs); ++id) { shaderIdTable[id] = id; }
    }

    //----------------------------------------
    // シェーダの生成です。
    for (int id = 0; id < ut::GetArrayLength(m_ShaderPtrs); ++id)
    {
        nw::lyt::ShaderId shaderId = static_cast<nw::lyt::ShaderId>(id);
        ShaderPtr& shader = m_ShaderPtrs[id];

        // 頂点シェーダを生成します。
        NW_ASSERT(0 < GFDGetVertexShaderCount(data));
        BOOL successVS = ReadVertexShader(&shader.vertexShader, shaderIdTable[id], data);
        NW_ASSERT(successVS);

        // ピクセルシェーダを生成します。
        NW_ASSERT(0 < GFDGetPixelShaderCount(data));
        BOOL successPS = ReadPixelShader(&shader.pixelShader, shaderIdTable[id], data);
        NW_ASSERT(successPS);

        // 生成したシェーダを登録します。
        nw::lyt::ShaderSlot& slot = graphicsResource->GetShaderSlot();
        nw::lyt::ShaderConnection* connection = slot.Connect(
            shaderId, shader.vertexShader, shader.pixelShader);

        // フェッチシェーダを生成します。
        {
            GX2AttribStream* attribute = slot.GetAttribStream(connection, 0);

            u32 location = static_cast<u32>(
                GX2GetVertexAttribVarLocation(shader.vertexShader, "aVertexIndex"));
            GX2InitAttribStream(
                attribute, location,
                0, 0,
                GX2_ATTRIB_FORMAT_32_32_FLOAT);

            GX2FetchShader* fetchShader = slot.GetFetchShader(connection);

            int fetchShaderSize = GX2CalcFetchShaderSize(slot.GetDescription().maxAttributeStreamQuantity);
            shader.fetchShaderBuffer = Layout::AllocMemory(fetchShaderSize, GX2_SHADER_ALIGNMENT);

            GX2InitFetchShader(
                fetchShader,
                shader.fetchShaderBuffer,
                slot.GetDescription().maxAttributeStreamQuantity,
                slot.GetAttribStreams(connection));
            GX2Invalidate(GX2_INVALIDATE_CPU, shader.fetchShaderBuffer, fetchShaderSize);
        }

        // シェーダーのユニフォームのIDをキャッシュします。
        // キャッシュしないといけないユニフォームの一覧は GraphicsResource::GetUniformList() で
        // 取ってくることができます。
        // slot.GetUniformIds(connection) で取得した配列にユーザー自身で設定することもできます。
        bool binded = shhelp::CacheUniforms(
            slot.GetUniformIds(connection), // ユニフォームのIDをキャッシュするための配列です。
            description.maxUniformQuantity,
            nw::lyt::GraphicsResource::GetUniformList(shaderId),
            slot.GetVertexShader(connection),
            slot.GetPixelShader(connection));
        NW_ASSERT(binded);

        slot.MakeDisplayList(connection);
    }
}

//----------------------------------------
void ShaderSetupHelper::Finalize()
{
    for (int id = 0; id < ut::GetArrayLength(m_ShaderPtrs); id++)
    {
        ShaderPtr& shaderPtr = m_ShaderPtrs[id];
        FreeVertexShader(shaderPtr.vertexShader);
        FreePixelShader(shaderPtr.pixelShader);
        if (shaderPtr.fetchShaderBuffer) { Layout::FreeMemory(shaderPtr.fetchShaderBuffer); }
    }
}

//----------------------------------------
bool ShaderSetupHelper::ReadVertexShader(GX2VertexShader** ppShader, u32 index, const void *data)
{
    if (data == NULL || ppShader == NULL) { return false; }
    if (index >= GFDGetVertexShaderCount(data)) { return false; }

    u32 headerSize = GFDGetVertexShaderHeaderSize(index, data);
    u32 programSize = GFDGetVertexShaderProgramSize(index, data);

    if (!headerSize || !programSize) { return false; }

    GX2VertexShader* header = static_cast<GX2VertexShader*>(
        Layout::AllocMemory(headerSize, PPC_IO_BUFFER_ALIGN));
    void* program = static_cast<void*>(
        Layout::AllocMemory(programSize, GX2_SHADER_ALIGNMENT));

    u32 ret = GFDGetVertexShader(header, program, index, data);
    if (ret)
    {
        GX2Invalidate(GX2_INVALIDATE_CPU,
                      header->shaderPtr,
                      header->shaderSize);
        *ppShader = header;
    }
    else
    {
        NW_LOG("Warning: Invalid Vertex Shader :%d", ret);
        if (header) { Layout::FreeMemory(header); }
        if (program) { Layout::FreeMemory(program); }
    }

    return ret;
}

//----------------------------------------
bool ShaderSetupHelper::ReadPixelShader(GX2PixelShader** ppShader, u32 index, const void *data)
{
    if (data == NULL || ppShader == NULL) { return FALSE; }
    if (index >= GFDGetPixelShaderCount(data)) { return FALSE; }

    u32 headerSize = GFDGetPixelShaderHeaderSize(index, data);
    u32 programSize = GFDGetPixelShaderProgramSize(index, data);

    if (!headerSize || !programSize) { return FALSE; }

    GX2PixelShader* header = static_cast<GX2PixelShader*>(
        Layout::AllocMemory(headerSize, PPC_IO_BUFFER_ALIGN));
    void* program = static_cast<void*>(
        Layout::AllocMemory(programSize, GX2_SHADER_ALIGNMENT));

    u32 ret = GFDGetPixelShader(header, program, index, data);
    if (ret)
    {
        GX2Invalidate(GX2_INVALIDATE_CPU,
                      header->shaderPtr,
                      header->shaderSize);
        *ppShader = header;
    }
    else
    {
        NW_LOG("Warning: Invalid Pixel Shader :%d", ret);
        if (header) { Layout::FreeMemory(header); }
        if (program) { Layout::FreeMemory(program); }
    }

    return ret;
}

//----------------------------------------
void ShaderSetupHelper::FreeVertexShader(GX2VertexShader* shader)
{
    if (shader)
    {
        if (shader->shaderPtr) { Layout::FreeMemory(shader->shaderPtr); }
        Layout::FreeMemory(shader);
    }
}

//----------------------------------------
void ShaderSetupHelper::FreePixelShader(GX2PixelShader* shader)
{
    if (shader)
    {
        if (shader->shaderPtr) { Layout::FreeMemory(shader->shaderPtr); }
        Layout::FreeMemory(shader);
    }
}

//----------------------------------------
void GraphicsResource::Setup(
        u32 charMax,
        const ut::MemoryRange& fontShaderBinary)
{
    {
        const u32 vtxBufferSize = nw::font::RectDrawer::GetWorkBufferSize(charMax, fontShaderBinary);
        m_FontDrawer.Initialize(
            Layout::AllocMemory(vtxBufferSize, GX2_VERTEX_BUFFER_ALIGNMENT),
            charMax,
            fontShaderBinary);
    }

    InitVBO();

    m_Initialized = true;
}

//----------------------------------------
gfnd::shader_helper::UniformParamRange
GraphicsResource::GetUniformList(ShaderId shaderId)
{
    namespace shhelp = nw::gfnd::shader_helper;
    static const shhelp::UniformParam uniforms[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };
    //----------------------------------------
    static const shhelp::UniformParam uniformsForNullTexture[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",

        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
    };
    static const shhelp::UniformParam uniformsForSingleTexture[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
    };
    static const shhelp::UniformParam uniformsForDoubleTexture[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
    };
    static const shhelp::UniformParam uniformsForDoubleIndirectTexture[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx0, "uIndirectMtx0",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx1, "uIndirectMtx1",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
    };
    //----------------------------------------
    static const shhelp::UniformParam uniformsWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };
    static const shhelp::UniformParam uniformsForNullTextureWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",

        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
    };
    static const shhelp::UniformParam uniformsForSingleTextureWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
    };
    static const shhelp::UniformParam uniformsForDoubleTextureWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
    };
    static const shhelp::UniformParam uniformsForDoubleIndirectTextureWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx0, "uIndirectMtx0",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx1, "uIndirectMtx1",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
    };

    switch (static_cast<s32>(shaderId))
    {
    case NULL_TEXTURE_SHADER: return ut::MakeRange(uniformsForNullTexture);
    case SINGLE_TEXTURE_SHADER: return ut::MakeRange(uniformsForSingleTexture);
    case DOUBLE_TEXTURE1_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_TEXTURE2_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_TEXTURE3_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_TEXTURE4_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_TEXTURE5_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_TEXTURE6_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_TEXTURE7_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_TEXTURE8_SHADER: return ut::MakeRange(uniformsForDoubleTexture);
    case DOUBLE_INDIRECT_TEXTURE_SHADER: return ut::MakeRange(uniformsForDoubleIndirectTexture);
    case DEFAULT_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsWithoutVertexColor);
    case NULL_TEXTURE_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForNullTextureWithoutVertexColor);
    case SINGLE_TEXTURE_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForSingleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE1_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE2_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE3_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE4_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE5_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE6_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE7_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_TEXTURE8_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleTextureWithoutVertexColor);
    case DOUBLE_INDIRECT_TEXTURE_SHADER + SHADER_ID_QUANTITY: return ut::MakeRange(uniformsForDoubleIndirectTextureWithoutVertexColor);
    default: return ut::MakeRange(uniforms);
    }
}

//----------------------------------------
gfnd::shader_helper::UniformParamRange
GraphicsResource::GetUniformList(const char* name, bool withoutVertexColor)
{
    namespace shhelp = nw::gfnd::shader_helper;
    static const shhelp::UniformParam uniforms[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };
    static const shhelp::UniformParam uniformsForTripleIndirectTexture[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx0, "uIndirectMtx0",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx1, "uIndirectMtx1",
        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };
    static const shhelp::UniformParam uniformsForTripleIndirectExTexture[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uColor, "uColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexColor, "uVertexColor",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSize, "uFrameSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uPaneSize, "uPaneSize",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx0, "uIndirectMtx0",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx1, "uIndirectMtx1",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };
    //----------------------------------------
    static const shhelp::UniformParam uniformsWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };
    static const shhelp::UniformParam uniformsForTripleIndirectTextureWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx0, "uIndirectMtx0",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx1, "uIndirectMtx1",
        shhelp::PIXEL_UNIFORM, UNIFORM_uTextureMode, "uTextureMode",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };
    static const shhelp::UniformParam uniformsForTripleIndirectExTextureWithoutVertexColor[] =
    {
        shhelp::VERTEX_UNIFORM, UNIFORM_uProjection, "uProjection",
        shhelp::VERTEX_UNIFORM, UNIFORM_uModelView, "uModelView[0]",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTransform, "uTransform",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_xz, "uTexMtx0_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx0_yw, "uTexMtx0_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_xz, "uTexMtx1_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx1_yw, "uTexMtx1_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_xz, "uTexMtx2_xz",
        shhelp::VERTEX_UNIFORM, UNIFORM_uTexMtx2_yw, "uTexMtx2_yw",
        shhelp::VERTEX_UNIFORM, UNIFORM_uFrameSpec, "uFrameSpec",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord0, "uVertexTexCoord0",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord1, "uVertexTexCoord1",
        shhelp::VERTEX_UNIFORM, UNIFORM_uVertexTexCoord2, "uVertexTexCoord2",
        shhelp::VERTEX_UNIFORM, UNIFORM_uGeneratingTexCoord, "uGeneratingTexCoord",
        shhelp::VERTEX_UNIFORM, UNIFORM_uRcpTexSize0, "uRcpTexSize0",

        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx0, "uIndirectMtx0",
        shhelp::PIXEL_UNIFORM, UNIFORM_uIndirectMtx1, "uIndirectMtx1",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateWidth, "uInterpolateWidth",
        shhelp::PIXEL_UNIFORM, UNIFORM_uInterpolateOffset, "uInterpolateOffset",

        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture0, "uTexture0",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture1, "uTexture1",
        shhelp::PIXEL_SAMPLER, UNIFORM_uTexture2, "uTexture2",
    };

    // nameをパースする
    s32 left = 0;
    s32 right = 0;
    const char* p = name;
    left = static_cast<s32>(*p - '0');
    p++;
    if (*p != '_')
    {
        left = left * 10 + static_cast<s32>(*p - '0');
        p++;
    }
    NW_ASSERT(*p == '_');
    p++;
    right = static_cast<s32>(*p - '0');
    p++;
    if (*p != '\0')
    {
        right = right * 10 + static_cast<s32>(*p - '0');
        p++;
    }
    NW_ASSERT(*p == '\0');

    if (withoutVertexColor)
    {
        if (left == 10 || left == 11)
        {
            return ut::MakeRange(uniformsForTripleIndirectExTextureWithoutVertexColor);
        }
        if (left == 9 || right == 9 || right == 11)
        {
            return ut::MakeRange(uniformsForTripleIndirectTextureWithoutVertexColor);
        }
        return ut::MakeRange(uniformsWithoutVertexColor);
    }
    else
    {
        if (left == 10 || left == 11)
        {
            return ut::MakeRange(uniformsForTripleIndirectExTexture);
        }
        if (left == 9 || right == 9 || right == 11)
        {
            return ut::MakeRange(uniformsForTripleIndirectTexture);
        }
        return ut::MakeRange(uniforms);
    }
}

//----------------------------------------
void
GraphicsResource::InitVBO()
{
    NW_ASSERT(m_Vertices == NULL);
    // GX2: バーテックスオブジェクト生成
    const int verticesSize = sizeof(f32) * 2 * 4;
    m_Vertices = static_cast<f32*>(Layout::AllocMemory(verticesSize, GX2_VERTEX_BUFFER_ALIGNMENT));
    m_Vertices[0] = 0.0f; m_Vertices[1] = 0.0f;
    m_Vertices[2] = 1.0f; m_Vertices[3] = 0.0f;
    m_Vertices[4] = 0.0f; m_Vertices[5] = 1.0f;
    m_Vertices[6] = 1.0f; m_Vertices[7] = 1.0f;
    GX2Invalidate(GX2_INVALIDATE_CPU, m_Vertices, verticesSize);

    NW_ASSERT(m_Indices == NULL);
    const int indeicesSize = sizeof(u16) * 4;
    m_Indices = static_cast<u16*>(Layout::AllocMemory(indeicesSize, GX2_INDEX_BUFFER_ALIGNMENT));
    m_Indices[0] = VERTEX_RT;
    m_Indices[1] = VERTEX_LT;
    m_Indices[2] = VERTEX_LB;
    m_Indices[3] = VERTEX_RB;
    GX2Invalidate(GX2_INVALIDATE_CPU, m_Indices, indeicesSize);
}

//----------------------------------------
void
GraphicsResource::ActiveVBO()
{
    // GX2: バーテックスオブジェクト有効化
    NW_ASSERT(m_Vertices);
    GX2SetAttribBuffer(
        0,
        sizeof(f32) * 2 * 4,
        sizeof(f32) * 2,
        m_Vertices);
}

//----------------------------------------
void
GraphicsResource::DrawVBO()
{
    NW_ASSERT(m_Indices);
    GX2DrawIndexed(
        GX2_PRIMITIVE_QUADS,
        VBO_INDEX_COUNT,
        GX2_INDEX_FORMAT_U16,
        m_Indices);
}

} // namespace nw::lyt
} // namespace nw

//########################################
#else
//########################################
#error "Not support platform."
#endif

namespace nw
{
namespace lyt
{

//----------------------------------------
GraphicsResource::GraphicsResource()
: m_pRectShaderBinary(NULL)
, m_RectShaderBinarySize(0)
, m_Initialized(false)
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
, m_GlProgram(0)
, m_VertexShader(0)
, m_FragmentShader(0)
#elif defined(NW_PLATFORM_CAFE)
, m_Vertices(NULL)
, m_Indices(NULL)
#endif
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    std::fill_n(m_UniformIds, static_cast<size_t>(UNIFORM_MAX), 0);
#endif
}

//----------------------------------------
GraphicsResource::~GraphicsResource()
{
    this->Finalize();
}

//----------------------------------------
void
GraphicsResource::Finalize()
{
    if (!m_Initialized)
    {
        return;
    }

    m_Initialized = false;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    glUseProgram(0);

    glDeleteProgram(m_GlProgram);
    m_GlProgram = 0;

    if ( m_FragmentShader )
    {
        glDeleteShader( m_FragmentShader );
        m_FragmentShader = NULL;
    }

    if ( m_VertexShader )
    {
        glDeleteShader( m_VertexShader );
        m_VertexShader = NULL;
    }

    glDeleteBuffers(this->VBO_MAX, m_GlVertexBufferObject);

#elif defined(NW_PLATFORM_CAFE)
    Layout::FreeMemory(m_Vertices);
    m_Vertices = NULL;
    Layout::FreeMemory(m_Indices);
    m_Indices = NULL;
    Layout::FreeMemory(m_ShaderSlot.Shutdown());
#endif

    if (NULL != m_pRectShaderBinary)
    {
        Layout::FreeMemory(m_pRectShaderBinary);
    }
    m_pRectShaderBinary = NULL;
    m_RectShaderBinarySize = 0;

    m_FontDrawer.Finalize();
    Layout::FreeMemory(m_FontDrawer.GetBuffer());
}

} // namespace nw::lyt
} // namespace nw
