﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_WINEXT_OSTIME_H_
#define NW_WINEXT_OSTIME_H_

#define OSMicrosecondsToNanoseconds(mics) ((mics) * 1000LL)
#define OSMillisecondsToNanoseconds(mils) ((mils) * 1000000LL)
#define OSSecondsToNanoseconds(secs)      ((secs) * 1000000000LL)

#define OSNanosecondsToMicroseconds(nano) ((nano) / 1000LL)
#define OSNanosecondsToMilliseconds(nano) ((nano) / 1000000LL)
#define OSNanosecondsToSeconds(nano)      ((nano) / 1000000000LL)

#ifndef _ASSEMBLER
#include <winext/types.h>

#if defined(__APPLE__)
#include <TargetConditionals.h>
#endif

namespace nw {
namespace internal {
namespace winext {

typedef s64 OSTime;
typedef s64 OSTimeSeconds;
typedef s64 OSTimeMilliseconds;
typedef s64 OSTimeMicroseconds;
typedef s64 OSTimeNanoseconds;
typedef u32 OSTick;

#if !defined(_WIN32) && !defined(ANDROID) && !defined(__APPLE__)
#define __OSBusClock    (OSGetSystemInfo()->busClockSpeed)
#define __OSCoreClock   (OSGetSystemInfo()->coreClockSpeed)

#define OS_BUS_CLOCK        __OSBusClock
#define OS_CORE_CLOCK       __OSCoreClock
#define OS_TIMER_CLOCK      (OS_BUS_CLOCK/4)
#endif

// Default Configuration
#ifndef OS_CORE_CLOCK
#define OS_CORE_CLOCK       729000000u
#endif
#ifndef OS_BUS_CLOCK
#define OS_BUS_CLOCK        243000000u
#endif
#ifndef OS_TIMER_CLOCK
#define OS_TIMER_CLOCK      (OS_BUS_CLOCK/4)
#endif

#define OSTicksToCycles( ticks )        (((ticks) * ((OS_CORE_CLOCK * 2) / OS_TIMER_CLOCK)) / 2)
#define OSTicksToSeconds( ticks )       ((ticks) / OS_TIMER_CLOCK)
#define OSSecondsToTicks( sec )         ((sec)  * OS_TIMER_CLOCK)

#if LEGACY_TIME_CONVERSIONS
    #define OSTicksToMilliseconds( ticks )  ((ticks) / (OS_TIMER_CLOCK / 1000))
    #define OSTicksToMicroseconds( ticks )  (((ticks) * 8) / (OS_TIMER_CLOCK / 125000))
    #define OSTicksToNanoseconds( ticks )   (((ticks) * 8000) / (OS_TIMER_CLOCK / 125000))

    #define OSMillisecondsToTicks( msec )   ((msec) * (OS_TIMER_CLOCK / 1000))
    #define OSMicrosecondsToTicks( usec )   (((usec) * (OS_TIMER_CLOCK / 125000)) / 8)
    #define OSNanosecondsToTicks( nsec )    (((nsec) * (OS_TIMER_CLOCK / 125000)) / 8000)
#else
    #define OSTicksToMilliseconds( ticks )  ((((u64)(ticks)) * 1000ULL)       / ((u64) OS_TIMER_CLOCK))
    #define OSTicksToMicroseconds( ticks )  ((((u64)(ticks)) * 1000000ULL)    / ((u64) OS_TIMER_CLOCK))
    #define OSTicksToNanoseconds( ticks )   ((((u64)(ticks)) * 32000ULL) / ((u64) OS_TIMER_CLOCK / 31250))

    #define OSMillisecondsToTicks( msec )   ((((u64)(msec)) * OS_TIMER_CLOCK)  / 1000)
    #define OSMicrosecondsToTicks( usec )   ((((u64)(usec)) * OS_TIMER_CLOCK)  / 1000000)
    #define OSNanosecondsToTicks( nsec )    ((((u64)(nsec)) * ((u64) OS_TIMER_CLOCK / 31250))  / 32000)
#endif



#define OSDiffTick(tick1, tick0)        ((s32) (tick1) - (s32) (tick0))

OSTick  OSGetTick( void );
OSTime  OSGetTime( void );
OSTime  OSGetSystemTime( void );

typedef struct OSCalendarTime
{
    int sec;    // seconds after the minute [0, 61]
    int min;    // minutes after the hour [0, 59]
    int hour;   // hours since midnight [0, 23]
    int mday;   // day of the month [1, 31]
    int mon;    // month since January [0, 11]
    int year;   // years since 0000
    int wday;   // days since Sunday [0, 6]
    int yday;   // days since January 1 [0, 365]

    int msec;   // milliseconds after the second [0,999]
    int usec;   // microseconds after the millisecond [0,999]
} OSCalendarTime;

OSTime OSCalendarTimeToTicks( OSCalendarTime* td );
void   OSTicksToCalendarTime( OSTime ticks, OSCalendarTime* td );

void OSWaitMicroseconds(OSTimeMicroseconds microseconds);

#ifdef __ghs__
static inline OSTime OSInlineGetSystemTime(void)
{
    u32 timeLower;
    u32 timeUpper1;
    u32 timeUpper2;

    OSTime resultTime;

    do
    {
        timeUpper1 = __MFTBU();
        timeLower  = __MFTB();
        timeUpper2 = __MFTBU();
    } while (timeUpper1 != timeUpper2);

    resultTime = (((OSTime) timeUpper1) << 32) | ((OSTime) timeLower);
    return resultTime;
}
#endif

#endif // _ASSEMBLER

} // namespace winext
} // namespace internal
} // namespace nw

#endif  // NW_WINEXT_OSTIME_H_

