﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef WINEXT_OSTHREAD_H__
#define WINEXT_OSTHREAD_H__

// Thread cancel state
#define OS_THREAD_CANCEL_ENABLED            0x00000000
#define OS_THREAD_CANCEL_DISABLED_USER      0x00000001

#define OS_THREAD_CANCEL_DISABLED_MUTEX     0x00010000
#define OS_THREAD_CANCEL_DISABLED_SPINLOCK  0x00020000
#define OS_THREAD_CANCEL_DISABLED_STACK     0x00040000
#define OS_THREAD_CANCEL_DISABLED_FASTMUTEX 0x00080000

#ifndef _ASSEMBLER

#include <winext/cafe/os/OSTime.h>

namespace nw {
namespace internal {
namespace winext {

#define OS_THREAD_SPECIFIC_MAX  16

struct OSThread;
typedef struct OSThreadSmallQueue OSThreadSmallQueue;
typedef struct OSThreadQueue    OSThreadQueue;
typedef struct OSThreadLink     OSThreadLink;
typedef s32                     OSPriority;     //  0 highest, 31 lowest

typedef struct OSMutex          OSMutex;
typedef struct OSMutexQueue     OSMutexQueue;
typedef struct OSMutexLink      OSMutexLink;
typedef struct OSCond           OSCond;
typedef struct OSAlarm          OSAlarm;
typedef struct OSTLSBlock       OSTLSBlock;

typedef struct OSFastMutex      OSFastMutex;
typedef struct OSFastMutexQueue OSFastMutexQueue;
typedef struct OSFastMutexLink  OSFastMutexLink;
typedef struct OSFastCond       OSFastCond;

struct OSThreadQueue
{
    OSThread*  head;
    OSThread*  tail;
    void *     parent_struct;   // For debugging, if OSThreadQueue is embedded inside larger structure
    u32        os_reserved1;
};

// SmallQueue is missing the parent_struct and os_reserved fields in order to save space
struct OSThreadSmallQueue
{
    OSThread*  head;
    OSThread*  tail;
};


struct OSThreadLink
{
    OSThread*  next;
    OSThread*  prev;
};

struct OSMutexQueue
{
    OSMutex*   head;
    OSMutex*   tail;
    void *     parent_struct;   // For debugging, if OSMutexQueue is embedded inside larger structure
    u32        os_reserved1;
};

struct OSMutexLink
{
    OSMutex*   next;
    OSMutex*   prev;
};

struct OSTLSBlock
{
    char*      address;         // start address of a thread specific copy of an RPL's TLS data
    int        status;          // debug agent's status of whether this address has been passed to MULTI
};

struct OSFastMutexQueue
{
    OSFastMutex *  head;
    OSFastMutex *  tail;
};

struct OSFastMutexLink
{
    OSFastMutex *  next;
    OSFastMutex *  prev;
};


#ifndef OSTHREADPROC_DECLARED
#define OSTHREADPROC_DECLARED
typedef int (*OSThread_Proc)(int intArg, void *ptrArg);
#endif

#if (defined __ghs__ || defined __MAKECORE__)

#include <time.h>

typedef struct _crt_overhead
{
    char            tmpnam_space[32];
    char            asctime_buff[30];
    char            *strtok_saved_pos;
    struct tm       gmtime_temp;
    void            *__eh_globals;
    void            *__eh_mem_manage[9];
    void            *__eh_store_globals[6];
    void            *__eh_store_globals_tdeh[76];
//    void            *__eh_init_block[2048];   8 kB !!!
} __crt_overhead;

#else

typedef struct _crt_overhead
{
    int dummy;
} __crt_overhead;

#endif

typedef enum
{
    OSTHREAD_TYPE_DRVR,   // device driver thread
    OSTHREAD_TYPE_APP_IO, // I/O thread of application
    OSTHREAD_TYPE_APP     // normal thread of application
} OSThreadType;

#define OS_THREAD_ID_APP_IO_START	((s16) 0x5000)
#define OS_THREAD_ID_DRV_START		((s16) 0x6000)
#define OS_THREAD_ID_NONE			((s16) 0x8000)

typedef void (*OSThreadCleanupCallback)(void* thread, void* stackEnd);
typedef void (*OSThreadDeallocator)(void* thread, void* stackEnd);

#define OSTHREAD_TXT_TAG              0x74487244
#define OSTHREAD_OS_RESERVED_BYTES    32

SET_STRUCT_ALIGN(8)
struct OSThread
{
    void* handle;
    int        intArg;
    void*      ptrArg;
    OSThread_Proc func;
};

SET_STRUCT_ALIGN(1)

// Thread states
enum OS_THREAD_STATE
{
    OS_THREAD_STATE_NONE     = 0,
    OS_THREAD_STATE_READY    = 1,
    OS_THREAD_STATE_RUNNING  = 2,
    OS_THREAD_STATE_WAITING  = 4,
    OS_THREAD_STATE_MORIBUND = 8
};

// Thread request flag
enum OS_THREAD_REQUEST_FLAG
{
    OS_THREAD_REQUEST_NONE    = 0x0,
    OS_THREAD_REQUEST_SUSPEND = 0x1,
    OS_THREAD_REQUEST_CANCEL  = 0x2
};

// Thread priorities
#define OS_APP_IO_PRIORITY_MIN      0   // highest
#define OS_APP_IO_PRIORITY_DEFAULT  16
#define OS_APP_IO_PRIORITY_MAX      31  // lowest
#define OS_APP_IO_PRIORITY_IDLE     OS_APP_IO_PRIORITY_MAX

#define OS_PRIORITY_MIN         0                   // highest
#define OS_PRIORITY_APP_DEFAULT 16
#define OS_PRIORITY_MAX         31                  // lowest
#define OS_PRIORITY_IDLE        OS_PRIORITY_MAX

//
// Thread attributes
//
// Threads lacking an affinity bit are affinitized to the
// core on which they are created
//

#define OS_THREAD_ATTR_AFFINITY_NONE    0x0007u        // affinity to run on every core
#define OS_THREAD_ATTR_AFFINITY_CORE0   0x0001u        // run only on core0
#define OS_THREAD_ATTR_AFFINITY_CORE1   0x0002u        // run only on core1
#define OS_THREAD_ATTR_AFFINITY_CORE2   0x0004u        // run only on core2
#define OS_THREAD_ATTR_DETACH           0x0008u        // detached
#define OS_THREAD_ATTR_PINNED_AFFINITY  0x0010u        // pinned (affinitized) to a single core
#define OS_THREAD_ATTR_CHECK_STACK_USE  0x0040u        // check for stack usage
#define OS_THREAD_ATTR_NAME_SENT        0x0080u        // debugger has seen the name
#define OS_THREAD_ATTR_LAST             (OS_THREAD_ATTR_DETACH | OS_THREAD_ATTR_PINNED_AFFINITY | OS_THREAD_ATTR_AFFINITY_NONE)

// Stack magic value
#define OS_THREAD_STACK_MAGIC            0xDEADBABE
#define OS_THREAD_STACK_USAGE_MAGIC      0xFEFEFEFE

// Time-slice
#define OS_THREAD_QUANTUM_INFINITE          0              // Default for run-to-completion threads
#define OS_THREAD_QUANTUM_MIN_MICROSECONDS  10
#define OS_THREAD_QUANTUM_MAX_MICROSECONDS  0xFFFFF        // (slightly more than 1 second)

void       OSInitThreadQueue   ( OSThreadQueue* queue );
void       OSInitThreadQueueEx ( OSThreadQueue* queue, void * parent_struct );
OSThread*  OSGetCurrentThread  ( void );
OSThread*  OSGetDefaultThread  ( u32 coreIdc);
BOOL       OSIsThreadSuspended ( OSThread* thread );
BOOL       OSIsThreadTerminated( OSThread* thread );
void       OSYieldThread       ( void );
BOOL       OSCreateThread(       OSThread*      thread,
                                 OSThread_Proc  entryPoint,
                                 int            intArg,
                                 void*          ptrArg,
                                 void*          stack,
                                 u32            stackSize,
                                 OSPriority     priority,
                                 u16            attr );
BOOL       OSCreateThreadType(   OSThread*      thread,
                                 OSThread_Proc  entryPoint,
                                 int            intArg,
                                 void*          ptrArg,
                                 void*          stack,
                                 u32            stackSize,
                                 OSPriority     priority,
                                 u16            attr,
                                 OSThreadType   type );
void       OSExitThread        ( int val );
BOOL       OSJoinThread        ( OSThread* thread, int * val );
void       OSCancelThread      ( OSThread* thread );
void       OSDetachThread      ( OSThread* thread );
s32        OSResumeThread      ( OSThread* thread );
s32        OSSuspendThread     ( OSThread* thread );
BOOL       OSSetThreadPriority ( OSThread* thread, OSPriority priority );
OSPriority OSGetThreadPriority ( OSThread* thread );
void       OSSleepThread       ( OSThreadQueue* queue );
void       OSWakeupThread      ( OSThreadQueue* queue );
BOOL       OSSetThreadAffinity ( OSThread* thread, u16 affinity);
u16        OSGetThreadAffinity ( OSThread* thread );
void       OSSetThreadName     ( OSThread* thread, const char* name);
const char* OSGetThreadName    ( OSThread const * thread );
void       OSTestThreadCancel  (void);
BOOL       OSSetThreadCancelState(BOOL cancelState);
void       OSContinueThread    ( OSThread* thread );
BOOL       OSSetThreadRunQuantum(OSThread *thread, u32 quantumMicrosec);

// Run the specified function on specified thread
// Only runs if thread is not already running.
// Runs with thread's specified priority.

BOOL       OSRunThread(OSThread*        thread,
                       OSThread_Proc    func,
                       int              intArg,
                       void *           ptrArg);

// Get/Set the data specific to the calling thread
void*      OSGetThreadSpecific ( s32 index );
void       OSSetThreadSpecific ( s32 index, void* ptr );

void       OSClearStack        ( u8 val );
u32        OSGetUserStackPointer(OSThread* thread);

BOOL OSSetThreadStackUsage(OSThread *thread);
void OSClearThreadStackUsage(OSThread *thread);
s32 OSCheckThreadStackUsage(OSThread *thread);

void       OSPrintActiveThreads( void );
long       OSCheckActiveThreads( void );

void       OSSleepTicks        ( OSTime ticks );

void       __OSBoostThreadPriority    ( OSThread* currentThread );
void       __OSDeboostThreadPrioriy   ( OSThread* currentThread );

OSTime OSGetThreadCoreTime(const OSThread *thread, u16 affinity);
#define OSGetThreadTime(thread) OSGetThreadCoreTime(thread, OS_THREAD_ATTR_AFFINITY_NONE)
#define OSGetCurrentThreadTime() OSGetThreadCoreTime(OSGetCurrentThread(), OS_THREAD_ATTR_AFFINITY_NONE)

#define OSSleepSeconds( sec )           OSSleepTicks( OSSecondsToTicks((OSTime)sec) )
// #define OSSleepMilliseconds( msec )     OSSleepTicks( OSMillisecondsToTicks((OSTime)msec) )
void OSSleepMilliseconds(u32 msec);
#define OSSleepMicroseconds( usec )     OSSleepTicks( OSMicrosecondsToTicks((OSTime)usec) )
#define OSSleepNanoseconds( nsec )      OSSleepTicks( OSNanosecondsToTicks((OSTime)nsec) )

OSThreadCleanupCallback OSSetThreadCleanupCallback(OSThread* thread, OSThreadCleanupCallback callback);
OSThreadDeallocator     OSSetThreadDeallocator(OSThread* thread, OSThreadDeallocator deallocator);

BOOL OSGetActiveThreadLink(OSThread *thread, OSThreadLink *threadLink);

#define OSPrintCurrentThreadState() \
{ \
    OSContext context = *OSGetCurrentContext(); \
    OSSaveContext(&context); \
    OSDumpContext(&context); \
}

// runtime initialization routines for GHS
// set these pointers to the values of __cpp_exception_init and __cpp_exception_cleanup
// this avoids a dependency between the RPX and the core libraries
extern void (*__cpp_exception_init_ptr)(void **);
extern void (*__cpp_exception_cleanup_ptr)(void **);

} // namespace winext
} // namespace internal
} // namespace nw

#endif // _ASSEMBLER

#endif  // WINEXT_OSTHREAD_H__

