﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_WINEXT_FS_H_
#define NW_WINEXT_FS_H_

#include <winext/cafe.h>

namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
 *
 *    Constants defined for this file
 *    -- #Defines's --
 *
 *---------------------------------------------------------------------------*/

#define FS_MAX_CLIENTS                  63
#define FS_MAX_MOUNTPOINTS              48

#define FS_CLIENT_BUFFER_SIZE           (5888)
#define FS_CMD_BLOCK_SIZE               (2688)

/* FS data buffer alignment */
#define FS_IO_BUFFER_ALIGN              64

/* Permission mode flag */
#define FS_MODE_IRGRP                   0x000040 /* read bit for owner's group */
#define FS_MODE_IWGRP                   0x000020 /* write bit for owner's group */
#define FS_MODE_IROTH                   0x000004 /* read bit for others */
#define FS_MODE_IWOTH                   0x000002 /* write bit for others */
#define FS_MODE_IRWG                    (FS_MODE_IRGRP | FS_MODE_IWGRP) /* read/write bits for owner's group */
#define FS_MODE_IRWO                    (FS_MODE_IROTH | FS_MODE_IWOTH) /* read/write bits for others */


/* Default mount paths */
#define FS_CONTENT_DIR                  "/vol/content"
#define FS_SAVE_DIR                     "/vol/save"

/* path to device files directory */
#define FS_DEVICEFILES_ROOT             "/dev"
#define FS_DEVICEFILES_ROOT_NAMESIZE    4

/* max length of path under mount point
 * should be the least value among all file systems */
#define FS_MAX_LOCALPATH_SIZE      511
#define FS_MAX_LOCALPATH_SIZE_PAD  1

/* max length of mount point path */
#define FS_MAX_MOUNTPATH_SIZE      128
#define FS_MAX_MOUNTPATH_SIZE_PAD  0

/* max length of full path (mount path + FS-local path) */
#define FS_MAX_FULLPATH_SIZE       (FS_MAX_LOCALPATH_SIZE + FS_MAX_MOUNTPATH_SIZE)
#define FS_MAX_FULLPATH_SIZE_PAD   (FS_MAX_LOCALPATH_SIZE_PAD + FS_MAX_MOUNTPATH_SIZE_PAD)

/* max length of path strings that user passes in arguments
 * should be the least value among all file systems */
#define FS_MAX_ARGPATH_SIZE        FS_MAX_FULLPATH_SIZE
#define FS_MAX_ARGPATH_SIZE_PAD    FS_MAX_FULLPATH_SIZE_PAD

/* max length of file/dir name
 * should be the least value among all file systems */
#define FS_MAX_ENTNAME_SIZE        256
#define FS_MAX_ENTNAME_SIZE_PAD    0

/* mode string in open file command parameters */
#define FS_MAX_MODE_SIZE           16
#define FS_MAX_MODE_SIZE_PAD       0


/* directory entry stat flag */
#define FS_STAT_ATTRIBUTES_SIZE             (48)        /* size of FS-specific attributes field */
#define FS_STAT_FLAG_IS_DIRECTORY           0x80000000  /* entry is directory */
#define FS_STAT_FLAG_IS_QUOTA               0x40000000  /* entry is quota */

#define FS_STAT_FLAG_SPRT_QUOTA_SIZE        0x20000000  /* FS supports .quota_size field */
#define FS_STAT_FLAG_SPRT_ENTID             0x10000000  /* FS supports .ent_id field */
#define FS_STAT_FLAG_SPRT_CTIME             0x08000000  /* FS supports .ctime field */
#define FS_STAT_FLAG_SPRT_MTIME             0x04000000  /* FS supports .mtime field */
#define FS_STAT_FLAG_SPRT_ATTRIBUTES        0x02000000  /* FS supports .attributes field */
#define FS_STAT_FLAG_SPRT_FILE_ALLOC_SIZE   0x01000000  /* FS supports .alloc_size field */
#define FS_STAT_FLAG_SPRT_DIR_SIZE          0x00100000  /* FS supports .size field for directory
                                                         * this field does not have meaning for file */

/* invalid handle value for FSOpenFile/FSOpenDir */
#define FS_INVALID_HANDLE_VALUE             -1

#if 0
/* Command priority for FSPriority */
#define FS_PRIORITY_DEFAULT                 16
#define FS_PRIORITY_HIGHEST                  0
#define FS_PRIORITY_LOWEST                  31
#endif


/*---------------------------------------------------------------------------*
 *
 *    Data types defined for this file
 *    -- Struct's, Typedef's, Enum's --
 *
 *---------------------------------------------------------------------------*/

/* Common primitives */
typedef s32                 FSError;
typedef s32                 FSStatus;
typedef s32                 FSDirHandle;
typedef s32                 FSFileHandle;
typedef u32                 FSSize;
typedef u64                 FSBSize;
typedef u32                 FSCount;
typedef u32                 FSFlag;
typedef u32                 FSRetFlag;
typedef u32                 FSMode;
typedef u32                 FSFilePosition;
typedef u64                 FSTime;
typedef u32                 FSEntryNum;
typedef u32                 FSUserId;
typedef u32                 FSGroupId;

typedef FSBSize             FSFreeSpaceSize;
typedef FSBSize             FSDirSize;

#if 0
typedef s32                 FSPriority;


#endif
/* FS Client context buffer */
typedef struct
{
    u8 buffer[FS_CLIENT_BUFFER_SIZE];
} FSClient;

/* FS comand block buffer */
typedef struct
{
    u8 buffer[FS_CMD_BLOCK_SIZE];
} FSCmdBlock;


/* Volume state */
typedef enum
{
    FS_VOLSTATE_INITIAL = 0, /* Internal use only */

    /* Ready */
    FS_VOLSTATE_READY,

    /*
     * Media is not attached.
     * State machine waits for re-insert if it is remountable.
     * If not, app should unmount the volume by itself.
     */
    FS_VOLSTATE_NO_MEDIA,        // There is no media

    /*
     * Something is wrong with media.
     * State machine waits for detach.
     */
    FS_VOLSTATE_INVALID_MEDIA,   // Unsupported media or state (cd-rom, reversed media)
    FS_VOLSTATE_DIRTY_MEDIA,     // Dirty / deformed media
    FS_VOLSTATE_WRONG_MEDIA,     // Media format is correct. but media is not the expected one.
    FS_VOLSTATE_MEDIA_ERROR,     // The other media error
    FS_VOLSTATE_DATA_CORRUPTED,  // Detected data corruption
    FS_VOLSTATE_WRITE_PROTECTED, // Write protected media

    /* Fatal */
    FS_VOLSTATE_JOURNAL_FULL,    // Journaling area is full. flushing data is needed.
    FS_VOLSTATE_FATAL,

    FS_VOLSTATE_INVALID
} FSVolumeState;


typedef enum
{
    FS_SOURCETYPE_EXTERNAL = 0, // Manual mounted external device
    FS_SOURCETYPE_HFIO,         // Host file IO
    FS_SOURCETYPE_MAX
} FSSourceType;


typedef struct
{
    FSSourceType          type;
    char                  path[FS_MAX_ARGPATH_SIZE];
} FSMountSource;

PACKED_STRUCT_BEGIN



/* File/Dir status */
typedef struct
{
    FSFlag      flag;           /* flag bitmap */

    FSMode      permission;     /* access permissions */

    FSUserId    owner_id;       /* owner title id */

    FSGroupId   group_id;       /* owner group id */

    FSSize      size;           /* file size in bytes.
                                   valid when entry is file. */
    FSSize      alloc_size;     /* allocated file size in bytes.
                                   valid when entry is file and FS_STAT_FLAG_SPRT_FILE_ALLOC_SIZE in flag is set */
    FSBSize     quota_size;     /* total size of quota/partition to which the entry belongs
                                   valid when FS_STAT_FLAG_SPRT_DIR_SIZE in flag is set */
    u32         ent_id;         /* entry ID. valid when FS_STAT_FLAG_SPRT_ENTID in flag is set */

    FSTime      ctime;          /* last created time
                                   valid when FS_STAT_FLAG_SPRT_CTIME in flag is set */
    FSTime      mtime;          /* last modified time
                                   valid when FS_STAT_FLAG_SPRT_MTIME in flag is set */

    u8          attributes[FS_STAT_ATTRIBUTES_SIZE];
                                /* FS-specific attribute field
                                   valid when FS_STAT_FLAG_SPRT_ATTRIBUTES in flag is set */
} PACKED_STRUCT_ATTRIBUTE FSStat;

/* Directory entry */
typedef struct
{
    FSStat      stat;
    char        name[FS_MAX_ENTNAME_SIZE + FS_MAX_ENTNAME_SIZE_PAD];
} PACKED_STRUCT_ATTRIBUTE FSDirEntry;

PACKED_STRUCT_END


/* Callback prototype for asynchronous command */
typedef void (*FSAsyncCallback)(
                    FSClient        *client,
                    FSCmdBlock      *block,
                    FSStatus        result,
                    void            *context
                    );


#if 0
typedef struct
{
    FSAsyncCallback        userCallback;
    void                   *userContext;
    OSMessageQueue*        ioMsgQueue;
} FSAsyncParams;
#endif


/* Callback prototype for error notification */
typedef void (*FSStateChangeCallback)(
                    FSClient       *client,
                    FSVolumeState  state,
                    void           *context
                    );


#if 0
typedef struct
{
    FSStateChangeCallback  userCallback;
    void                   *userContext;
    OSMessageQueue         *ioMsgQueue;
} FSStateChangeParams;
#endif



#if 0
PACKED_STRUCT_BEGIN

typedef struct
{
    void*               data; // FSAsyncResult or FSStateNoticeInfo
    u32                 reserved;
    u32                 reserved2;
    OSFunctionType      type; // OSFunctionType must be the 4th member,
                              // for compatiblity with OSDeviceMessage.
} PACKED_STRUCT_ATTRIBUTE FSMessage;

typedef struct
{
    FSAsyncParams       userParams;
    FSMessage           ioMsg;

    FSClient            *client;
    FSCmdBlock          *block;
    FSStatus            status;
} PACKED_STRUCT_ATTRIBUTE FSAsyncResult;

typedef struct
{
    FSStateChangeParams userParams;
    FSMessage           ioMsg;

    FSClient            *client;
    FSVolumeState       state;
} PACKED_STRUCT_ATTRIBUTE FSStateNoticeInfo;

PACKED_STRUCT_END
#endif


/*---------------------------------------------------------------------------*
 *
 *    -- Returned Error Codes --
 *
 *---------------------------------------------------------------------------*/

/* common error */
#define FS_STATUS_OK                    0

#define FS_STATUS_ERROR_BASE            (0)

#define FS_STATUS_CANCELLED             (FS_STATUS_ERROR_BASE-1)    /* Command cancelled */
#define FS_STATUS_END                   (FS_STATUS_ERROR_BASE-2)    /* Indicates end of file / directory entry */
#define FS_STATUS_MAX                   (FS_STATUS_ERROR_BASE-3)    /* Reached to max number of file / directory / client handles / mount points */
#define FS_STATUS_ALREADY_OPEN          (FS_STATUS_ERROR_BASE-4)    /* Target is already opened or locked by another transaction */
#define FS_STATUS_EXISTS                (FS_STATUS_ERROR_BASE-5)    /* Target path already exists or not empty */
#define FS_STATUS_NOT_FOUND             (FS_STATUS_ERROR_BASE-6)    /* Target path is not found */
#define FS_STATUS_NOT_FILE              (FS_STATUS_ERROR_BASE-7)    /* Specified path is directory or quota instead of a file. */
#define FS_STATUS_NOT_DIR               (FS_STATUS_ERROR_BASE-8)    /* Specified path is file instead of a directory or a quota. */
#define FS_STATUS_ACCESS_ERROR          (FS_STATUS_ERROR_BASE-9)    /* Attempted to access file with bad file mode */
#define FS_STATUS_PERMISSION_ERROR      (FS_STATUS_ERROR_BASE-10)   /* Did not have permission to complete operation */
#define FS_STATUS_FILE_TOO_BIG          (FS_STATUS_ERROR_BASE-11)   /* Request would push the file over the size limit (not the quota limit). */
#define FS_STATUS_STORAGE_FULL          (FS_STATUS_ERROR_BASE-12)   /* Request would cause one of the ancestor directories to exceed its quota
                                                                       / Or no free space left in storage */
#define FS_STATUS_JOURNAL_FULL          (FS_STATUS_ERROR_BASE-13)   /* Transaction journal is full, need to flush */
#define FS_STATUS_UNSUPPORTED_CMD       (FS_STATUS_ERROR_BASE-14)   /* Operation is not supported by file system */

/* for only manual mount device */
#define FS_STATUS_MEDIA_NOT_READY       (FS_STATUS_ERROR_BASE-15)   /* Medium is not ready */
#define FS_STATUS_INVALID_MEDIA         (FS_STATUS_ERROR_BASE-16)   /* Medium is invalid */
#define FS_STATUS_MEDIA_ERROR           (FS_STATUS_ERROR_BASE-17)   /* Medium is in some bad condition */
#define FS_STATUS_DATA_CORRUPTED        (FS_STATUS_ERROR_BASE-18)   /* Data is corrupted */
#define FS_STATUS_WRITE_PROTECTED       (FS_STATUS_ERROR_BASE-19)   /* Medium is write protected */


/* common fatal error */
#define FS_STATUS_FATAL_ERROR           (FS_STATUS_ERROR_BASE-1024) /* Internal resources ran short */


/* Error handling flag */
#define FS_RET_MAX                      0x0001
#define FS_RET_ALREADY_OPEN             0x0002
#define FS_RET_EXISTS                   0x0004
#define FS_RET_NOT_FOUND                0x0008
#define FS_RET_NOT_FILE                 0x0010
#define FS_RET_NOT_DIR                  0x0020
#define FS_RET_ACCESS_ERROR             0x0040
#define FS_RET_PERMISSION_ERROR         0x0080
#define FS_RET_FILE_TOO_BIG             0x0100
#define FS_RET_STORAGE_FULL             0x0200
#define FS_RET_UNSUPPORTED_CMD          0x0400
#define FS_RET_JOURNAL_FULL             0x0800

/* FS_STATUS_OK, FS_STATUS_END, FS_STATUS_CANCELLED and positive value can be returned always. */
#define FS_RET_NO_ERROR                 0x0000

#define FS_RET_ALL_ERROR                (u32)(-1)




/* FSError */
#define FS_ERROR_OK                       0

#define FS_ERROR_BASE                     (-196608)
/* FS service is not initialized */
#define FS_ERROR_NOT_INIT                 (FS_ERROR_BASE-1)
/* File system or device was too busy */
#define FS_ERROR_BUSY                     (FS_ERROR_BASE-2)
/* Command canceled */
#define FS_ERROR_CANCELED                 (FS_ERROR_BASE-3)
/* Indicates end of directory */
#define FS_ERROR_END_OF_DIRECTORY         (FS_ERROR_BASE-4)
/* Indicates end of file */
#define FS_ERROR_END_OF_FILE              (FS_ERROR_BASE-5)

/* Reached to max number of mount points */
#define FS_ERROR_MAX_MOUNTPOINTS          (FS_ERROR_BASE-16)
/* Reached to max number of volumes */
#define FS_ERROR_MAX_VOLUMES              (FS_ERROR_BASE-17)
/* Reached to max number of clients */
#define FS_ERROR_MAX_CLIENTS              (FS_ERROR_BASE-18)
/* Reached to max number of file handles */
#define FS_ERROR_MAX_FILES                (FS_ERROR_BASE-19)
/* Reached to max number of dir handles */
#define FS_ERROR_MAX_DIRS                 (FS_ERROR_BASE-20)
/* Target is already opened or locked by another transaction */
#define FS_ERROR_ALREADY_OPEN             (FS_ERROR_BASE-21)
/* Target path already exists */
#define FS_ERROR_ALREADY_EXISTS           (FS_ERROR_BASE-22)
/* Target path is not found */
#define FS_ERROR_NOT_FOUND                (FS_ERROR_BASE-23)
/* Target path is not empty */
#define FS_ERROR_NOT_EMPTY                (FS_ERROR_BASE-24)
/* Attempted to access file with bad file mode */
#define FS_ERROR_ACCESS_ERROR             (FS_ERROR_BASE-25)
/* Did not have permission to complete operation */
#define FS_ERROR_PERMISSION_ERROR         (FS_ERROR_BASE-26)
/* Cannot complete transaction due to corrupted data block */
#define FS_ERROR_DATA_CORRUPTED           (FS_ERROR_BASE-27)
/* Request would cause one of the ancestor directories to exceed its quota
 * Or no free space left in storage */
#define FS_ERROR_STORAGE_FULL             (FS_ERROR_BASE-28)
/* Transaction journal is full, need to flush */
#define FS_ERROR_JOURNAL_FULL             (FS_ERROR_BASE-29)

/* Operation is not available now */
#define FS_ERROR_UNAVAILABLE_CMD          (FS_ERROR_BASE-31)
/* Operation is not supported by FS */
#define FS_ERROR_UNSUPPORTED_CMD          (FS_ERROR_BASE-32)
/* Specified parameter is invalid */
#define FS_ERROR_INVALID_PARAM            (FS_ERROR_BASE-33)
/* Specified path is invalid */
#define FS_ERROR_INVALID_PATH             (FS_ERROR_BASE-34)
/* Specified buffer is invalid */
#define FS_ERROR_INVALID_BUFFER           (FS_ERROR_BASE-35)
/* Specified alignment is invalid */
#define FS_ERROR_INVALID_ALIGNMENT        (FS_ERROR_BASE-36)
/* Specified client handle is invalid */
#define FS_ERROR_INVALID_CLIENT_HANDLE    (FS_ERROR_BASE-37)
/* Specified file handle is invalid */
#define FS_ERROR_INVALID_FILE_HANDLE      (FS_ERROR_BASE-38)
/* Specified dir handle is invalid */
#define FS_ERROR_INVALID_DIR_HANDLE       (FS_ERROR_BASE-39)
/* Specified path is directory instead of a file. */
#define FS_ERROR_NOT_FILE                 (FS_ERROR_BASE-40)
/* Specified path is file instead of a directory. */
#define FS_ERROR_NOT_DIR                  (FS_ERROR_BASE-41)
/* Request would push the file over the size limit (not the quota limit). */
#define FS_ERROR_FILE_TOO_BIG             (FS_ERROR_BASE-42)
/* Attempted to access out of accessible area */
#define FS_ERROR_OUT_OF_RANGE             (FS_ERROR_BASE-43)
/* Internal resources ran short */
#define FS_ERROR_OUT_OF_RESOURCES         (FS_ERROR_BASE-44)

/* Medium is not ready to use, user has to put medium correctly */
#define FS_ERROR_MEDIA_NOT_READY          (FS_ERROR_BASE-64)
/* Medium is in some bad condition */
#define FS_ERROR_MEDIA_ERROR              (FS_ERROR_BASE-65)
/* Medium is in some bad condition */
#define FS_ERROR_WRITE_PROTECTED          (FS_ERROR_BASE-66)
/* Medium is invalid */
#define FS_ERROR_INVALID_MEDIA            (FS_ERROR_BASE-67)

#define FS_ERROR_FATAL                    (FS_ERROR_BASE-1024)


/*---------------------------------------------------------------------------*
 *
 *    -- Interface Function Prototypes --
 *
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
  Name:         FSInit

  Description:  Initialize FS library.

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void                FSInit( void );

/*---------------------------------------------------------------------------*
  Name:         FSShutdown

  Description:  Shutdown FS library.

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void                FSShutdown( void );

/*---------------------------------------------------------------------------*
  Name:         FSAddClient

  Description:  Register a new client.

  Arguments:    client          Pointer to client buffer.

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_MAX
 *---------------------------------------------------------------------------*/
FSStatus            FSAddClient(
                        FSClient                *client,
                        FSRetFlag               errHandling
                        );


/*---------------------------------------------------------------------------*
  Name:         FSDelClient

  Description:  Unregister a client.

  Arguments:    client          Pointer to client

  Returns:      Result code :
                  FS_STATUS_OK
 *---------------------------------------------------------------------------*/
FSStatus            FSDelClient(
                        FSClient                *client,
                        FSRetFlag               errHandling
                        );


/*---------------------------------------------------------------------------*
  Name:         FSGetClientNum

  Description:  Get number of client.

  Arguments:    None.

  Returns:      Number of registered clients.
                Non-negative value (>= 0)
 *---------------------------------------------------------------------------*/
u32                 FSGetClientNum( void );


/*---------------------------------------------------------------------------*
  Name:         FSInitCmdBlock

  Description:  Initialize command block.
                If command priority is set, it will be reset to the default value.

  Arguments:    block           Command block

  Returns:      None.
 *---------------------------------------------------------------------------*/
void                FSInitCmdBlock( FSCmdBlock *block );


/*---------------------------------------------------------------------------*
  Name:         FSCancelCommand

  Description:  Cancel specified command block which is still waiting.
                This cannot cancel already on going command.

  Arguments:    client          Pointer to client buffer
                block           Command block

  Returns:      None.

 *---------------------------------------------------------------------------*/
void                FSCancelCommand(
                        FSClient                *client,
                        FSCmdBlock              *block
                        );

/*---------------------------------------------------------------------------*
  Name:         FSCancelAllCommands

  Description:  Cancel all waiting command blocks for specified client
                handle.
                This cannot cancel already ongoing command.

  Arguments:    client          Pointer to client buffer

  Returns:      None.

 *---------------------------------------------------------------------------*/
void                FSCancelAllCommands(
                        FSClient                *client
                        );


/*---------------------------------------------------------------------------*
  Name:         FSSetUserData

  Description:  Set user data value to command block

  Arguments:    block           Command block
                userData        User data value

  Returns:      None.
 *---------------------------------------------------------------------------*/
void                FSSetUserData(
                        FSCmdBlock              *block,
                        void                    *userData
                        );


/*---------------------------------------------------------------------------*
  Name:         FSGetUserData

  Description:  Get user data value from command block

  Arguments:    block           Command block

  Returns:      User data value
                If command block struct pointer is invalid, return NULL.
 *---------------------------------------------------------------------------*/
void*               FSGetUserData(
                        FSCmdBlock              *block
                        );


/*---------------------------------------------------------------------------*
  Name:         FSGetVolumeState

  Description:  Get current state of specified volume.

  Arguments:    client          Pointer to client buffer

  Returns:      Volume status (FS_VOLSTATE_*)
                If client struct pointer is invalid, return FS_VOLSTATE_INVALID.
 *---------------------------------------------------------------------------*/
FSVolumeState       FSGetVolumeState(
                        FSClient                *client
                    );


#if 0
/*---------------------------------------------------------------------------*
  Name:         FSSetStateChangeNotification

  Description:  Set notification parameters when media or fatal error is occured
                and volume state is changed.

  Arguments:    ch              Client handle
                changeParams    Pointer to registered error notifications
                                You can unset notification by setting NULL here

  Returns:      None.
 *---------------------------------------------------------------------------*/
void                FSSetStateChangeNotification(
                        FSClient                *client,
                        FSStateChangeParams     *changeParams
                    );


/*---------------------------------------------------------------------------
  Name:        FSGetAsyncResult

  Description: Get a pointer to FSAsyncResult including the FSMessage.
               This function is used after the message notification is returned.

  Arguments:   ioMsg    Pointer to an FSMessage

  Returns:     pointer to an FSAsyncResult
  *---------------------------------------------------------------------------*/
FSAsyncResult*      FSGetAsyncResult(FSMessage* ioMsg);


/*---------------------------------------------------------------------------
  Name:        FSGetStateChangeInfo

  Description: Get a pointer to FSStateNoticeInfo included in the FSMessage.
               This function is used after the message notification is returned.

  Arguments:   ioMsg    Pointer to an FSMessage

  Returns:     pointer to an FSStateNoticeInfo
  *---------------------------------------------------------------------------*/
FSStateNoticeInfo*  FSGetStateChangeInfo(FSMessage* ioMsg);
#endif


/*---------------------------------------------------------------------------*
  Name:         FSGetMountSource(Async)

  Description:  Find and get mount source list.
                This function is for manual mounted external devices,
                hfio volume and so on.

  Arguments:    client          Pointer to client buffer
                block           Command block
                type            Source type
                source          Pointer to mountable source

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_END
                  FS_STATUS_MAX
                  FS_STATUS_CANCELLED

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSGetMountSource(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSSourceType            type,
                        FSMountSource           *source,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSGetMountSourceAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSSourceType            type,
                        FSMountSource           *source,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif


/*---------------------------------------------------------------------------*
  Name:         FSGetMountSourceNext(Async)

  Description:  Find and get next mount source list.
                This function is for manual mounted external devices,
                hfio volume and so on.

  Arguments:    client          Pointer to client buffer
                block           Command block
                source          Pointer to mountable source

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_END
                  FS_STATUS_MAX
                  FS_STATUS_CANCELLED

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSGetMountSourceNext(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSMountSource           *source,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSGetMountSourceNextAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSMountSource           *source,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif


/*---------------------------------------------------------------------------*
  Name:         FSMount(Async)

  Description:  Mount device or source path and return mount point path.
                This function is for manual mounted external devices,
                hfio volume and so on.
                No need to call for device stored application and save data since they are pre-mounted.

  Arguments:    client          Pointer to client buffer
                block           Command block
                source          Mountable source structure.
                target          Buffer for target mount point path.
                bytes           Length of target buffer.
                                This must be greater than or equal to
                                FS_MAX_MOUNTPATH_SIZE.


  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_EXISTS
                  FS_STATUS_MEDIA_ERROR

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSMount(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSMountSource           *source,
                        char                    *target,
                        u32                     bytes,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSMountAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSMountSource           *source,
                        char                    *target,
                        u32                     bytes,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSUnmount(Async)

  Description:  Unmount volume to specified mount point path.
                Actual device to mount is abstracted.
                This function is for manual mounted external devices,
                hfio volume and so on.
                No need to call for device stored application and save data since they are pre-mounted.

  Arguments:    client          Pointer to client buffer
                block           Command block
                target          Target mount point path.

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_MEDIA_ERROR

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSUnmount(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *target,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSUnmountAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *target,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSChangeDir(Async)

  Description:  Change directory

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Directory path.
                                Length must be less than FS_MAX_ARGPATH_SIZE.

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSChangeDir(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSChangeDirAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSOpenFile(Async)

  Description:  Open file

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Directory path.
                                Length must be less than FS_MAX_ARGPATH_SIZE.
                mode            Access mode
                fileHandle     Pointer to the handle of file stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_ALREADY_OPEN
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_NOT_FILE
                  FS_STATUS_MAX
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_PERMISSION_ERROR
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSOpenFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        const char              *mode,
                        FSFileHandle            *fileHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSOpenFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        const char              *mode,
                        FSFileHandle            *fileHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif


/*---------------------------------------------------------------------------*
  Name:         FSCloseFile(Async)

  Description:  Close file

  Arguments:    client          Pointer to client buffer
                block           Command block
                fileHandle      Handle of file stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSCloseFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSCloseFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSReadFile(Async)

  Description:  Read file

  Arguments:    client          Pointer to client buffer
                block           Command block
                dest            Pointer to data array to read into
                size            Size of one array member [bytes]
                count           Number of array members to read
                fileHandle      Handle of file stream

  Returns:      Result code :
                  Non-negative value (>= 0) Number of elements read.
                  FS_STATUS_CANCELLED
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSReadFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        void                    *dest,
                        FSSize                  size,
                        FSCount                 count,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSReadFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        void                    *dest,
                        FSSize                  size,
                        FSCount                 count,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSWriteFile(Async)

  Description:  Write file

  Arguments:    client          Pointer to client buffer
                block           Command block
                source          Pointer to data array to write into
                size            Size of one array member [bytes]
                count           Number of array members to read
                fileHandle      Handle of file stream

  Returns:      Result code :
                  Non-negative value (>= 0) Number of elements written.
                  FS_STATUS_CANCELLED
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_FILE_TOO_BIG
                  FS_STATUS_STORAGE_FULL
                  FS_STATUS_JOURNAL_FULL
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSWriteFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const void              *source,
                        FSSize                  size,
                        FSCount                 count,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSWriteFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const void              *source,
                        FSSize                  size,
                        FSCount                 count,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSReadFileWithPos(Async)

  Description:  Read file

  Arguments:    client          Pointer to client buffer
                block           Command block
                dest            Pointer to data array to read into
                size            Size of one array member [bytes]
                count           Number of array members to read
                fpos            Stating file position
                fileHandle      Handle of file stream

  Returns:      Result code :
                  Non-negative value (>= 0) Number of elements read.
                  FS_STATUS_CANCELLED
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSReadFileWithPos(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        void                    *dest,
                        FSSize                  size,
                        FSCount                 count,
                        FSFilePosition          fpos,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSReadFileWithPosAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        void                    *dest,
                        FSSize                  size,
                        FSCount                 count,
                        FSFilePosition          fpos,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSWriteFileWithPos(Async)

  Description:  Write file

  Arguments:    client          Pointer to client buffer
                block           Command block
                source          Pointer to data array to write into
                size            Size of one array member [bytes]
                count           Number of array members to read
                fpos            Stating file position
                fileHandle      Handle of file stream

  Returns:      Result code :
                  Non-negative value (>= 0) Number of elements written.
                  FS_STATUS_CANCELLED
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_FILE_TOO_BIG
                  FS_STATUS_STORAGE_FULL
                  FS_STATUS_JOURNAL_FULL
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSWriteFileWithPos(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const void              *source,
                        FSSize                  size,
                        FSCount                 count,
                        FSFilePosition          fpos,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSWriteFileWithPosAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const void              *source,
                        FSSize                  size,
                        FSCount                 count,
                        FSFilePosition          fpos,
                        FSFileHandle            fileHandle,
                        FSFlag                  flag,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSAppendFile(Async)

  Description:  Append file

  Arguments:    client          Pointer to client buffer
                block           Command block
                size            Size of one array member [bytes]
                count           Number of array members to append
                fileHandle      Handle of file stream

  Returns:      Result code :
                  Non-negative value (>= 0) Number of elements appended.
                  FS_STATUS_CANCELLED
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_FILE_TOO_BIG
                  FS_STATUS_STORAGE_FULL
                  FS_STATUS_JOURNAL_FULL
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSAppendFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSSize                  size,
                        FSCount                 count,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSAppendFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSSize                  size,
                        FSCount                 count,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSTruncateFile(Async)

  Description:  Truncate file at file position

  Arguments:    client          Pointer to client buffer
                block           Command block
                fileHandle      Handle of file stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_FILE_TOO_BIG
                  FS_STATUS_STORAGE_FULL
                  FS_STATUS_JOURNAL_FULL
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSTruncateFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSTruncateFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSFlushFile(Async)

  Description:  Flush file
                It is for only for filsystem supporting file caching feature.
                The other file systems are return FS_STATUS_OK.

  Arguments:    client          Pointer to client buffer
                block           Command block
                fileHandle      Handle of file stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_ACCESS_ERROR
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSFlushFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSFlushFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSGetPosFile(Async)

  Description:  Get file position

  Arguments:    client          Pointer to client buffer
                block           Command block
                fileHandle      Handle of file stream
                returnedFpos    Pointer to file position

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSGetPosFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSFilePosition          *returnedFpos,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSGetPosFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSFilePosition          *returnedFpos,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSSetPosFile(Async)

  Description:  Set file position

  Arguments:    client          Pointer to client buffer
                block           Command block
                fileHandle      Handle of file stream
                fpos            file position

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus           FSSetPosFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSFilePosition          fpos,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus           FSSetPosFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSFilePosition          fpos,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSGetStatFile(Async)

  Description:  Get file status

  Arguments:    client          Pointer to client buffer
                block           Command block
                fileHandle      Handle of file stream
                returnedStat    Pointer to file status

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSGetStatFile(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSStat                  *returnedStat,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSGetStatFileAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSStat                  *returnedStat,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSIsEof(Async)

  Description:  Check if the current file position is at the end of file (EOF).

  Arguments:    client          Pointer to client buffer
                block           Command block
                fileHandle      Handle of file stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_END
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSIsEof(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSIsEofAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSFileHandle            fileHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif


/*---------------------------------------------------------------------------*
  Name:         FSOpenDir(Async)

  Description:  Open directory and creates directory stream

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Directory path.
                                Length must be less than FS_MAX_ARGPATH_SIZE.
                dirHandle       Pointer to the handle of directory stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_NOT_DIR
                  FS_STATUS_MAX_HANDLES
                  FS_STATUS_PERMISSION_ERROR
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSOpenDir(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSDirHandle             *dirHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSOpenDirAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSDirHandle             *dirHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSCloseDir(Async)

  Description:  Close directory

  Arguments:    client          Pointer to client buffer
                block           Command block
                dirHandle       Handle of directory stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSCloseDir(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSDirHandle             dirHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSCloseDirAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSDirHandle             dirHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSReadDir(Async)

  Description:  Read directory entry name and status

  Arguments:    client          Pointer to client buffer
                block           Command block
                dirHandle       Handle of directory stream
                returndDirEntry information of entry

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_END
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSReadDir(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSDirHandle             dirHandle,
                        FSDirEntry              *returnedDirEntry,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSReadDirAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSDirHandle             dirHandle,
                        FSDirEntry              *returnedDirEntry,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSRewindDir(Async)

  Description:  Rewind directory stream

  Arguments:    client          Pointer to client buffer
                block           Command block
                dirHandle       Handle of directory stream

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSRewindDir(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSDirHandle             dirHandle,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSRewindDirAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        FSDirHandle             dirHandle,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSGetCwd(Async)

  Description:  Get current work directory path.

  Arguments:    client          Pointer to client buffer
                block           Command block
                returnedPath    Pointer to path buffer
                bytes           Length of path buffer.
                                This must be greater than or equal to
                                FS_MAX_FULLPATH_SIZE.

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSGetCwd(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        char                    *returnedPath,
                        u32                     bytes,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSGetCwdAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        char                    *returnedPath,
                        u32                     bytes,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSMakeDir(Async)

  Description:  Create directory.

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Directory path to create.

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_EXISTS
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_PERMISSION_ERROR
                  FS_STATUS_UNSUPPORTED_CMD
                  FS_STATUS_STORAGE_FULL
                  FS_STATUS_JOURNAL_FULL
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSMakeDir(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSMakeDirAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSRemove(Async)

  Description:  Remove entry.

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Directory path to open.

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_PERMISSION_ERROR
                  FS_STATUS_ALREADY_OPEN
                  FS_STATUS_UNSUPPORTED_CMD
                  FS_STATUS_STORAGE_FULL
                  FS_STATUS_JOURNAL_FULL
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSRemove(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSRemoveAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSRename(Async)

  Description:  Rename entry.

  Arguments:    client          Pointer to client buffer
                block           Command block
                oldPath         Old entry path.
                newPath         New entry path.

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_PERMISSION_ERROR
                  FS_STATUS_ALREADY_EXISTS
                  FS_STATUS_ALREADY_OPEN
                  FS_STATUS_UNSUPPORTED_CMD
                  FS_STATUS_STORAGE_FULL
                  FS_STATUS_JOURNAL_FULL
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSRename(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *oldPath,
                        const char              *newPath,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSRenameAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *oldPath,
                        const char              *newPath,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif


/*---------------------------------------------------------------------------*
  Name:         FSFlushQuota(Async)

  Description:  Flush quota
                It is for only for filsystem supporting journaling feature.
                The other file systems are return FS_STATUS_OK.

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Flushed quota path

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_UNSUPPORTED_CMD
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSFlushQuota(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSFlushQuotaAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSRollbackQuota(Async)

  Description:  Rollback quota
                It is for only for filsystem supporting journaling feature.
                The other file systems are return FS_STATUS_UNSUPPORTED_CMD.

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Rollbacked quota path

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_ALREADY_OPEN
                  FS_STATUS_UNSUPPORTED_CMD
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSRollbackQuota(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSRollbackQuotaAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSGetStat(Async)

  Description:  Get entry status

  Arguments:    client          Pointer to client buffer
                block           Command block
                path            Entry path
                returnedStat    Entry status

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)
                  FS_STATUS_FATAL_ERROR

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSGetStat(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSStat                  *returnedStat,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSGetStatAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSStat                  *returnedStat,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );
#endif

/*---------------------------------------------------------------------------*
  Name:         FSGetFreeSpaceSize(Async)

  Description:  Get free space size in quota

  Arguments:    client           Pointer to client buffer
                block            Command block
                path             Entry path
                returnedFreeSize Pointer to free size in quota

  Returns:      Result code :
                  FS_STATUS_OK
                  FS_STATUS_CANCELLED
                  FS_STATUS_NOT_FOUND
                  FS_STATUS_PERMISSION_ERROR
                  FS_STATUS_MEDIA_ERROR (For only manual mounted device)

                  (Async API: Immediate return values)
                  FS_STATUS_OK
                  FS_STATUS_FATAL_ERROR
 *---------------------------------------------------------------------------*/
FSStatus            FSGetFreeSpaceSize(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSFreeSpaceSize         *returnedFreeSize,
                        FSRetFlag               errHandling
                        );
#if 0
FSStatus            FSGetFreeSpaceSizeAsync(
                        FSClient                *client,
                        FSCmdBlock              *block,
                        const char              *path,
                        FSFreeSpaceSize         *returnedFreeSize,
                        FSRetFlag               errHandling,
                        const FSAsyncParams     *asyncParams
                        );



/*---------------------------------------------------------------------------*
  Name:         FSGetLastErrorCodeForViewer

  Description:  Calculate error code from last error and device type.
                This error code is used for error code viewer.

  Arguments:    client          Pointer to client buffer

  Returns:      Signed 32-bit error code for error viewer.
 *---------------------------------------------------------------------------*/
s32                 FSGetLastErrorCodeForViewer(
                        FSClient                *client
                        );

/*---------------------------------------------------------------------------*
  Name:         FSGetErrorCodeForViewer

  Description:  Calculate error code from last error and device type.
                This error code is used for error code viewer.

  Arguments:    client          Pointer to client buffer
                block           Pointer to command block

  Returns:      Signed 32-bit error code for error viewer.
 *---------------------------------------------------------------------------*/
s32                 FSGetErrorCodeForViewer(
                        FSClient                *client,
                        FSCmdBlock              *block
                        );


/*---------------------------------------------------------------------------*
  Name:         FSTimeToCalendarTime

  Description:  Convert FSTime to OSCalendarTime

  Arguments:    time             FSTime to be converted
                td               Pointer to OSCalendarTime struct

  Returns:      None.
 *---------------------------------------------------------------------------*/
void                FSTimeToCalendarTime(
                        FSTime          time,
                        OSCalendarTime  *td
                        );

/*---------------------------------------------------------------------------*
  Name:         FSSetCmdPriority

  Description:  Set priority for specified command block.
                Command block must be in free state.

  Arguments:    block       Command block to set priority.
                priority    New priority of the command. 0 is the highest priority, 31 is the lowest.
                            Default is 16.

  Returns:      FS_STATUS_OK            Successfully set new priority to the command block.
                FS_STATUS_FATAL_ERROR   block is invalid address or already in use,
                                        or priority is out of supported range.
 *---------------------------------------------------------------------------*/
FSStatus            FSSetCmdPriority(
                        FSCmdBlock              *block,
                        FSPriority              priority
                        );

/*---------------------------------------------------------------------------*
  Name:         FSGetCmdPriority

  Description:  Get current priority for specified command block.

  Arguments:    block       Command block to get priority.

  Returns:      Non-negative value (>= 0) Priority of the command.
                FS_STATUS_FATAL_ERROR     block is invalid address.
 *---------------------------------------------------------------------------*/
FSPriority          FSGetCmdPriority(
                        FSCmdBlock              *block
                        );
#endif


/*
 * for only development
 */



/*---------------------------------------------------------------------------*
  Name:         FSGetLastError(Async)

  Description:  Get detailed information of last error from client handle.

  Arguments:    client           Pointer to client buffer

  Returns:      error value.
 *---------------------------------------------------------------------------*/
FSError             FSGetLastError(
                        FSClient                *client
                        );

#if 0
/*---------------------------------------------------------------------------*
  Name:         FSSetEmulatedError

  Description:  Set up error code emulation for specified client.
                Next command queued to this client will fail with specified error code.

  Arguments:    client  Client to set emulated error
                error   Error code to emulate.
                        Specifying FS_ERROR_OK cancels the emulation.
                        Otherwise error must be negative.
                        Emulating FS_ERROR_MEDIA_NOT_READY is not supported.

  Returns:      FS_STATUS_OK            Successfully set emulated error code.
 *---------------------------------------------------------------------------*/
FSStatus            FSSetEmulatedError(
                        FSClient                *client,
                        FSError                 error
                        );

/*---------------------------------------------------------------------------*
  Name:         FSGetEmulatedError

  Description:  Get emulated error code set for specified client.

  Arguments:    client  Client to get emulated error

  Returns:      Negative value : Error code to emulate.
                FS_ERROR_OK    : No emulation is set.
                Positive value : Failed to get emulated error code.
                                 1: client is invalid address.
 *---------------------------------------------------------------------------*/
FSError             FSGetEmulatedError(
                        FSClient                *client
                        );
#endif

} // namespace winext
} // namespace internal
} // namespace nw

#endif // NW_WINEXT_FS_H_
