﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_STRING_H_
#define NW_UT_STRING_H_

#if defined(NW_PLATFORM_ANDROID)
#include <cstdio>
#endif

#include <cstring>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/ut_Memory.h>

namespace nw {
namespace ut {


//----------------------------------------
//! @name 文字列操作関連
//@{

//---------------------------------------------------------------------------
//! @brief        UTF-16 文字列を UTF-32 文字列に変換します。
//!
//! src 文字列を UTF-32 文字列に変換し、必ず nul の終端文字を追加します。
//! dstCount が src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @param[in]    dst 変換先のバッファのアドレスです。
//! @param[in]    src 変換元の文字列のアドレスです。
//! @param[in]    dstCount 変換先のバッファサイズです。
//! @param[in]    replacement 不正な文字だった場合に置換する文字です。
//!
//! @return       変換した文字数を返します。
//---------------------------------------------------------------------------
size_t wcs16towcs32(char32* dst, const char16* src, std::size_t dstCount, char32 replacement = ' ');

//---------------------------------------------------------------------------
//! @brief        UTF-32 文字列を UTF-16 文字列に変換します。
//!
//! src 文字列を UTF-16 文字列に変換し、必ず nul の終端文字を追加します。
//! dstCount が src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @param[in]    dst 変換先のバッファのアドレスです。
//! @param[in]    src 変換元の文字列のアドレスです。
//! @param[in]    dstCount 変換先のバッファサイズです。
//! @param[in]    replacement 不正な文字だった場合に置換する文字です。
//!
//! @return       変換した文字数を返します。
//---------------------------------------------------------------------------
size_t wcs32towcs16(char16* dst, const char32* src, std::size_t dstCount, char16 replacement = ' ');

//---------------------------------------------------------------------------
//! @brief        UTF-8 文字列を UTF-16 文字列に変換します。
//!
//! src 文字列を UTF-16 文字列に変換し、必ず nul の終端文字を追加します。
//! dstCount が src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! dst に NULL を指定した場合には、dstCount 引数は無視して変換後の文字数計算のみをおこないます。
//!
//! @param[in]    dst 変換先のバッファのアドレスです。NULL の場合には文字数の計算のみをおこないます。
//! @param[in]    src 変換元の文字列のアドレスです。
//! @param[in]    dstCount 変換先のバッファサイズです。
//! @param[in]    replacement 不正な文字だった場合に置換する文字です。
//!
//! @return       変換した文字数を返します。
//---------------------------------------------------------------------------
size_t mbstowcs16(char16* dst, const char* src, std::size_t dstCount, char16 replacement = ' ');

//---------------------------------------------------------------------------
//! @brief        UTF-16 文字列を UTF-8 文字列に変換します。
//!
//! src 文字列を UTF-8 文字列に変換し、必ず nul の終端文字を追加します。
//! dstCount が src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! dst に NULL を指定した場合には、dstCount 引数は無視して変換後の文字数計算のみをおこないます。
//!
//! @param[in]    dst 変換先のバッファのアドレスです。NULL の場合には文字数の計算のみをおこないます。
//! @param[in]    src 変換元の文字列のアドレスです。
//! @param[in]    dstCount 変換先のバッファサイズです。
//! @param[in]    replacement 不正な文字だった場合に置換する文字です。
//!
//! @return       変換した文字数を返します。
//---------------------------------------------------------------------------
size_t wcs16tombs(char* dst, const char16* src, std::size_t dstCount, char replacement = ' ');

//---------------------------------------------------------------------------
//! @brief        UTF8 文字列をバッファサイズで区切った場合に中途半端な文字列を切り詰めた
//!               文字列長を取得します。
//!
//! @param[in]    src        UTF8 文字列のアドレスです。
//! @param[in]    bufferSize 入力文字列のバッファサイズを取得します。
//!
//! @return       末尾の中途半端な文字列を切り詰めたバイト数を返します。
//---------------------------------------------------------------------------
size_t mbslen(const char* src, std::size_t bufferSize);

namespace internal{
//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列を srcCount 分コピーし、必ず nul の終端文字を追加します。
//! dstCount が srcCount 小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @tparam       文字の型です。
//!
//! @param[in]    dst コピー先のバッファのアドレスです。
//! @param[in]    dstCount コピー先のバッファサイズです。
//! @param[in]    src コピー元の文字列のアドレスです。
//! @param[in]    srcCount コピーする文字数です。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
template<typename TChar>
NW_INLINE size_t
strncpy_t(TChar* dst, std::size_t dstCount, const TChar* src, std::size_t srcCount)
{
    size_t length = 0;
    while ( length < ut::Min((dstCount - 1), srcCount) )
    {
        if (*src == std::char_traits<TChar>::to_char_type('\0'))
        {
            break;
        }

        *dst = *src;
        ++dst;
        ++src;
        ++length;
    }

    *dst = std::char_traits<TChar>::to_char_type('\0');

    return length;
}
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列をコピーし、必ず nul の終端文字を追加します。
//! dstCount が src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @param[in]    dst コピー先のバッファのアドレスです。
//! @param[in]    dstCount コピー先のバッファサイズです。
//! @param[in]    src コピー元の文字列のアドレスです。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
NW_INLINE size_t
strcpy(char* dst, std::size_t dstCount, const char* src)
{
    return ut::internal::strncpy_t(dst, dstCount, src, dstCount - 1);
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列をコピーし、必ず nul の終端文字を追加します。
//! dst の配列サイズ が src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @tparam       DstCount コピー先配列のサイズです。
//! @param[in]    dst コピー先の文字配列です。
//! @param[in]    src コピー元の文字列のアドレスです。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
template <size_t DstCount>
NW_FORCE_INLINE size_t
strcpy(char (&dst)[DstCount], const char* src)
{
    return ut::strcpy(dst, DstCount, src);
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列をコピーし、必ず nul の終端文字を追加します。
//! dstCount が src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @param[in]    dst コピー先のバッファのアドレスです。
//! @param[in]    dstCount コピー先のバッファサイズです。
//! @param[in]    src コピー元の文字列のアドレスです。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
NW_INLINE size_t
wcs16cpy(char16* dst, std::size_t dstCount, const char16* src)
{
    return ut::internal::strncpy_t(dst, dstCount, src, dstCount - 1);
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列をコピーし、必ず nul の終端文字を追加します。
//! dst の配列サイズが src の大きさより小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @tparam       DstCount コピー先配列のサイズです。
//! @param[in]    dst コピー先の文字配列です。
//! @param[in]    src コピー元の文字列のアドレスです。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
template <size_t DstCount>
NW_FORCE_INLINE size_t
wcs16cpy(char16 (&dst)[DstCount], const char16* src)
{
    return ut::wcs16cpy(dst, DstCount, src);
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列を srcCount 分コピーし、必ず nul の終端文字を追加します。
//! dstCount が srcCount 小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @param[in]    dst コピー先のバッファのアドレスです。
//! @param[in]    dstCount コピー先のバッファサイズです。
//! @param[in]    src コピー元の文字列のアドレスです。
//! @param[in]    srcCount コピーする文字数です。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
NW_INLINE size_t
strncpy(char* dst, std::size_t dstCount, const char* src, std::size_t srcCount)
{
    return ut::internal::strncpy_t(dst, dstCount, src, srcCount);
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列を srcCount 分コピーし、必ず nul の終端文字を追加します。
//! dst の配列サイズが srcCount 小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @tparam       DstCount コピー先配列のサイズです。
//! @param[in]    dst コピー先の文字配列です。
//! @param[in]    src コピー元の文字列のアドレスです。
//! @param[in]    srcCount コピーする文字数です。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
template <size_t DstCount>
NW_FORCE_INLINE size_t
strncpy(char (&dst)[DstCount], const char* src, std::size_t srcCount)
{
    return ut::strncpy(dst, DstCount, src, srcCount);
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列を srcCount 分コピーし、必ず nul の終端文字を追加します。
//! dstCount が srcCount 小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @param[in]    dst コピー先のバッファのアドレスです。
//! @param[in]    dstCount コピー先のバッファサイズです。
//! @param[in]    src コピー元の文字列のアドレスです。
//! @param[in]    srcCount コピーする文字数です。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
NW_INLINE size_t
wcs16ncpy(char16* dst, std::size_t dstCount, const char16* src, std::size_t srcCount)
{
    return ut::internal::strncpy_t(dst, dstCount, src, srcCount);
}

//---------------------------------------------------------------------------
//! @brief        文字列をコピーします。
//!
//! src 文字列を srcCount 分コピーし、必ず nul の終端文字を追加します。
//! dst の配列サイズ が srcCount 小さい場合は切り詰めてコピーしますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @tparam       DstCount コピー先配列のサイズです。
//! @param[in]    dst コピー先の文字配列です。
//! @param[in]    src コピー元の文字列のアドレスです。
//! @param[in]    srcCount コピーする文字数です。
//!
//! @return       コピーした文字数を返します。
//---------------------------------------------------------------------------
template <size_t DstCount>
NW_FORCE_INLINE size_t
wcs16ncpy(char16 (&dst)[DstCount], const char16* src, std::size_t srcCount)
{
    return ut::wcs16ncpy(dst, DstCount, src, srcCount);
}

//---------------------------------------------------------------------------
//! @brief        文字列を追加します。
//!
//! source 文字列を count 分コピーし、必ず null の終端文字を追加します。
//! dstCount が srcCount 小さい場合は切り詰めて追加されますが、
//! 必ず nul 終端文字を挿入します。
//!
//! @param[in]    dst 追加先のバッファのアドレスです。
//! @param[in]    dstCount コピー先のサイズです。
//! @param[in]    src 追加元の文字列のアドレスです。
//! @param[in]    srcCount 追加する文字数です。
//!
//! @return       追加した文字数を返します。
//---------------------------------------------------------------------------
size_t strncat(char* dst, std::size_t dstCount, const char* src, std::size_t srcCount);

//---------------------------------------------------------------------------
//! @brief        文字列の先頭から文字を検索します。
//!
//! @param[in]    src      検索対象文字列です。
//! @param[in]    srcCount 検索対象文字列の最大バッファ長です。
//! @param[in]    c        検索対象の文字です。
//!
//! @return       最初に発見された文字位置のポインタを返します。
//!               見つからない場合は NULL を返します。
//---------------------------------------------------------------------------
NW_INLINE char*
strnchr(char* src, std::size_t srcCount, int c)
{
    char* ptr = src;
    char* end_ptr = src + srcCount;

    while ( *ptr && ptr != end_ptr )
    {
        if (*ptr == c ) { return ptr; }
        ++ptr;
    }

    return NULL;
}

//---------------------------------------------------------------------------
//! @brief        文字列の先頭からマッチする文字を検索します。(const 版)
//!
//! @param[in]    src      検索対象文字列です。
//! @param[in]    srcCount 検索対象文字列の最大バッファ長です。
//! @param[in]    c        検索対象の文字です。
//!
//! @return       最初に発見された文字位置のポインタを返します。
//!               見つからない場合は NULL を返します。
//---------------------------------------------------------------------------
NW_INLINE const char*
strnchr(const char* src, std::size_t srcCount, int c)
{
    return strnchr( const_cast<char*>( src ), srcCount, c );
}


//---------------------------------------------------------------------------
//! @brief        指定されたデータを書式指定して書き込みます。
//!
//! source 文字列を count 分書き込み、必ず null の終端文字を追加します。
//!
//! @param[out]    buffer 書き込み先のバッファのアドレスです。
//! @param[in]     bufferSize 書き込み先のサイズです。
//! @param[in]     strCount 書き込む文字数です。特に指定がない場合は、(bufferSize - 1) を入力してください。
//! @param[in]     format 書式です。
//! @param[in]     vargs 引数リストです。
//!
//! @return       文字列の長さを返します。
//---------------------------------------------------------------------------
NW_INLINE int
vsnprintf(char* buffer, std::size_t bufferSize, std::size_t strCount, const char* format, va_list vargs)
{
#if defined(NW_COMPILER_MSVC)
#pragma warning(push)
#pragma warning(disable : 4996)
#endif

#if defined(NW_COMPILER_MSVC)
    int result = ::vsnprintf_s( buffer, bufferSize, strCount, format, vargs );
#elif defined(NW_COMPILER_GHS) || defined(NW_COMPILER_CLANG)
    int result = ::vsnprintf( buffer, Min(bufferSize, strCount + 1), format, vargs );
#else
    int result = std::vsnprintf( buffer, strCount, format, vargs );
#endif

    buffer[bufferSize - 1] = '\0';

    return result;

#if defined(NW_COMPILER_MSVC)
#pragma warning(pop)
#endif
}

//---------------------------------------------------------------------------
NW_INLINE int
vsnprintf(char* buffer, std::size_t bufferSize, const char* format, std::va_list vlist)
{
    return nw::ut::vsnprintf(buffer, bufferSize, bufferSize - 1, format, vlist);
}



//---------------------------------------------------------------------------
//! @brief        指定されたデータを書式指定して書き込みます。
//!
//! source 文字列を count 分書き込み、必ず null の終端文字を追加します。
//!
//! @param[out]    buffer 書き込み先のバッファのアドレスです。
//! @param[in]     bufferSize 書き込み先のサイズです。
//! @param[in]     strCount 書き込む文字数です。特に指定がない場合は、(bufferSize - 1) を入力してください。
//! @param[in]     format 書式です。
//! @param[in]     vargs 引数リストです。
//!
//! @return       文字列の長さを返します。
//---------------------------------------------------------------------------
#if 0
// MEMO: 32 ビットへ変換しない独自実装を追加。ただし、まだ検証が済むまで旧バージョンを利用します。
#ifdef NW_COMPILER_MSVC
NW_INLINE int
vsnw16printf(char16* buffer, std::size_t bufferSize, std::size_t strCount, const char16* format, va_list vargs)
{
#pragma warning(push)
#pragma warning(disable : 4996)
    int result = ::_vsnwprintf_s( buffer, bufferSize, strCount, format, vargs );
    buffer[bufferSize - 1] = 0;
    return result;
#pragma warning(pop)
}
#else
int
vsnw16printf(char16* buffer, std::size_t bufferSize, std::size_t strCount, const char16* format, va_list vargs);
#endif

#else
NW_INLINE int
vsnw16printf(char16* buffer, std::size_t bufferSize, std::size_t strCount, const char16* format, va_list vargs)
{
#ifdef NW_COMPILER_MSVC
#pragma warning(push)
#pragma warning(disable : 4996)
#endif

#ifdef NW_COMPILER_MSVC
    int result = ::_vsnwprintf_s( buffer, bufferSize, strCount, format, vargs );
    buffer[bufferSize - 1] = 0;
#elif defined(NW_COMPILER_GHS)
    NW_STATIC_ASSERT(sizeof(char16) == sizeof(wchar_t));
    int result = std::vswprintf( reinterpret_cast<wchar_t*>(buffer), std::min(bufferSize - 1, strCount), reinterpret_cast<const wchar_t*>(format), vargs );
    buffer[bufferSize - 1] = 0;
#elif defined(NW_COMPILER_CLANG)
    // TODO: vswprintf 対応
    const int BUFFER_SIZE = 1024;
    char dstBuffer[BUFFER_SIZE];
    char formatBuffer[BUFFER_SIZE];
    ut::wcs16tombs( formatBuffer, format, BUFFER_SIZE );
#if defined(NW_PLATFORM_ANDROID)
    int result = vsnprintf( dstBuffer, strCount, formatBuffer, vargs );
#else
    int result = std::vsnprintf( dstBuffer, strCount, formatBuffer, vargs );
#endif
    dstBuffer[bufferSize - 1] = 0;

    ut::mbstowcs16(buffer, dstBuffer, bufferSize);
#else
    NW_STATIC_ASSERT(sizeof(char32) == sizeof(wchar_t));
    const int BUFFER_SIZE = 1024;

    char32 dstBuffer[BUFFER_SIZE];
    char32 formatBuffer[BUFFER_SIZE];
    ut::wcs16towcs32(formatBuffer, format, BUFFER_SIZE);

    int result = std::vswprintf( reinterpret_cast<wchar_t*>(dstBuffer), strCount, reinterpret_cast<wchar_t*>(formatBuffer), vargs );
    dstBuffer[bufferSize - 1] = 0;

    ut::wcs32towcs16(buffer, dstBuffer, bufferSize);
#endif

    return result;

#ifdef NW_COMPILER_MSVC
#pragma warning(pop)
#endif
}
#endif

//---------------------------------------------------------------------------
//! @brief        バッファへ文字列を書式指定で書き込みます。
//!
//! source 文字列を count 分書き込み、必ず null の終端文字を追加します。
//!
//! @param[out]    buffer 書き込み先のバッファのアドレスです。
//! @param[in]     bufferSize 書き込み先のサイズです。
//! @param[in]     strCount 書き込む文字数です。特に指定がない場合は、(bufferSize - 1) を入力してください。
//! @param[in]     format 書式です。
//!
//! @return       文字列の長さを返します。
//---------------------------------------------------------------------------
NW_INLINE int
snprintf(char* buffer, std::size_t bufferSize, std::size_t strCount, const char* format, ...)
{
    std::va_list vargs;
    va_start(vargs, format);

    int result = vsnprintf(buffer, bufferSize, strCount, format, vargs);

    va_end(vargs);
    return result;
}

//---------------------------------------------------------------------------
NW_INLINE int
snprintf(char* buffer, std::size_t bufferSize, const char* format, ...)
{
    std::va_list vargs;
    va_start(vargs, format);

    int result = vsnprintf(buffer, bufferSize, bufferSize - 1, format, vargs);

    va_end(vargs);
    return result;
}

//---------------------------------------------------------------------------
//! @brief        指定されたデータを書式指定して書き込みます。
//!
//! source 文字列を count 分書き込み、必ず null の終端文字を追加します。
//!
//! @param[out]    buffer 書き込み先のバッファのアドレスです。
//! @param[in]     bufferSize 書き込み先のサイズです。
//! @param[in]     strCount 書き込む文字数です。特に指定がない場合は、(bufferSize - 1) を入力してください。
//! @param[in]     format 書式です。
//!
//! @return       文字列の長さを返します。
//---------------------------------------------------------------------------
NW_INLINE int
snw16printf(char16* buffer, std::size_t bufferSize, std::size_t strCount, const char16* format, ...)
{
    std::va_list vargs;
    va_start(vargs, format);

    int result = vsnw16printf(buffer, bufferSize, strCount, format, vargs);

    va_end(vargs);
    return result;
}

//---------------------------------------------------------------------------
//! @brief        文字列を小文字として比較します。
//!
//! @param[in]    string1 比較対象の文字列です。
//! @param[in]    string2 比較対象の文字列です。
//!
//! @return       比較結果を返します。
//---------------------------------------------------------------------------
int stricmp(const char *string1, const char *string2);

//---------------------------------------------------------------------------
//! @brief        文字列を小文字として比較します。
//!
//! @param[in]    string1 比較対象の文字列です。
//! @param[in]    string2 比較対象の文字列です。
//!
//! @return       比較結果を返します。
//---------------------------------------------------------------------------
int wcs16icmp(const char16 *string1, const char16 *string2);

//---------------------------------------------------------------------------
//! @brief        文字列を小文字として比較します。
//!
//! @param[in]    string1 比較対象の文字列です。
//! @param[in]    string2 比較対象の文字列です。
//!
//! @return       比較結果を返します。
//---------------------------------------------------------------------------
int wcs32icmp(const char32 *string1, const char32 *string2);

//---------------------------------------------------------------------------
//! @brief        16ビットワイド文字列の文字数を返します。
//!
//! @param[in]    src 文字列のアドレスです。
//!
//! @return       文字数を返します。サロゲートペアは複数文字として計算されます。
//---------------------------------------------------------------------------
size_t wcs16len(const char16* src);

template <typename TChar> class StringBaseImpl;

//---------------------------------------------------------------------------
//! @brief        バッファへ文字列を書式指定で書き込みます。
//!
//! @param[out]    buffer 書き込み先の StringBaseImpl のアドレスです。
//! @param[in]     format 書式です。
//!
//! @return       文字列の長さを返します。
//---------------------------------------------------------------------------
template <typename TChar>
NW_INLINE int snprintf(StringBaseImpl<TChar>* buffer, const TChar* format, ...);

//@}


//----------------------------------------
//! @name String クラス
//@{

//---------------------------------------------------------------------------
//! @brief        範囲チェック付き文字バッファの基底クラスです。
//!
//! @sa           nw::ut::snprintf
//---------------------------------------------------------------------------
template <typename TChar>
class StringBaseImpl
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ StringBaseImpl() : m_CharArray( NULL ), m_BufferSize( 0 ) {}

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ virtual ~StringBaseImpl() {}

    //---------------------------------------------------------------------------
    //! @brief        C の NUL 終端文字列へのポインタを取得します。
    //!
    //! @return       内部で持っている C 文字列へのポインタを返します。
    //---------------------------------------------------------------------------
    const TChar* c_str() const { return m_CharArray; }

    //---------------------------------------------------------------------------
    //! @brief        インデックス番目の文字を取得します。
    //!               バッファ範囲外の場合には ASSERT とします。
    //!
    //! @param[in]    idx     取得する文字のインデックスです。
    //!
    //! @return       インデックス番目の文字を返します。
    //---------------------------------------------------------------------------
    const TChar at( uint idx ) const
    {
        NW_ASSERT_NOT_NULL( m_CharArray );
        NW_ASSERT_MINMAXLT( idx, 0, m_BufferSize );

        return m_CharArray[ idx ];
    }

    //---------------------------------------------------------------------------
    //! @brief        インデックス版目の文字を取得する為の operator です。
    //!               バッファ範囲外の場合には ASSERT とします。
    //!
    //! @param[in]    idx     取得する文字のインデックスです。
    //!
    //! @return       インデックス番目の文字を返します。
    //---------------------------------------------------------------------------
    const TChar operator[]( s32 idx ) const { return this->at( idx ); }

    //---------------------------------------------------------------------------
    //! @brief        リサイズなしで扱えるバッファサイズを文字数で返します。
    //!
    //! @return       バッファに格納できる文字数です。
    //---------------------------------------------------------------------------
    uint capacity() const { return m_BufferSize; }

    //---------------------------------------------------------------------------
    //! @brief        NUL 終端文字の文字長を取得します。
    //!
    //! @return       文字数です。
    //---------------------------------------------------------------------------
    uint length() const
    {
        if ( m_CharArray == NULL ) { return 0; }
        return StringBaseImpl::StrLen_( m_CharArray );
    }

    //---------------------------------------------------------------------------
    //! @brief        NUL 終端文字の文字長を取得します。
    //!               length() と同じものですが、STL で利用できように用意しています。
    //! @return       文字列の長さです。
    //---------------------------------------------------------------------------
    uint size() const { return this->length(); }

    //---------------------------------------------------------------------------
    //! @brief        文字列が空かどうかを取得します。
    //!
    //! @return       文字列が空の場合には true、それ以外の場合には false を返します。
    //---------------------------------------------------------------------------
    bool empty() const { return m_CharArray == NULL || *m_CharArray == 0; }

protected:
    TChar* m_CharArray;     //!< @briefprivate
    uint   m_BufferSize;    //!< @briefprivate

    //! @briefprivate
    //! @details 文字列をコピーします。
    //! @param[out] dst コピーした文字列のバッファです。
    //! @param[in] src コピー元になる文字列です。
    //! @param[in] length コピーする文字列の長さです。
    static void StrCopy_(TChar* dst, const TChar* src, uint length)
    {
    #ifdef NW_COMPILER_MSVC
    #pragma warning(push)
    #pragma warning(disable : 4996)
    #endif
        std::char_traits<TChar>::copy(dst, src, length);
    #ifdef NW_COMPILER_MSVC
    #pragma warning(pop)
    #endif

        dst[ length ] = 0;
    }

    //! @briefprivate
    //! @details 文字列の長さを取得します。
    //! @param[in] str 長さを取得する文字列です。
    //! @return 文字列の長さです。
    static uint StrLen_(const TChar* str)
    {
        return std::char_traits<TChar>::length(str);
    }

    //! @briefprivate
    //! @details 文字列を切り取ります。
    //! @param[out] dst 切り取った文字列のバッファです。
    //! @param[in] src 切り取る元になる文字列です。
    //! @param[in] srcLen 切り取る文字列の長さです。
    static void StrCat_(TChar* dst, const TChar* src, uint srcLen)
    {
        uint offset = StrLen_( dst );

        memcpy( dst + offset, src, srcLen * sizeof(TChar) );

        dst[ offset + srcLen ] = 0;
    }

private:
    friend int snprintf<TChar>(StringBaseImpl<TChar>* buffer, const TChar* format, ...);
};

//---------------------------------------------------------------------------
//! @brief        範囲チェック付き文字バッファの char 版です。
//---------------------------------------------------------------------------
typedef StringBaseImpl<char>   StringBase;

//---------------------------------------------------------------------------
//! @brief        範囲チェック付き文字バッファの char16 版です。
//---------------------------------------------------------------------------
typedef StringBaseImpl<char16> StringBase16;

//---------------------------------------------------------------------------
//! @briefprivate
//! @details       バッファへ文字列を書式指定で書き込みます。
//!
//! @param[out]    buffer 書き込み先の StringBaseImpl のアドレスです。
//! @param[in]     format 書式です。
//!
//! @return        文字列の長さを返します。
template <>
NW_INLINE int
snprintf<char>(StringBaseImpl<char>* buffer, const char* format, ...)
{
    std::va_list vargs;
    va_start(vargs, format);

    int result = vsnprintf(buffer->m_CharArray, buffer->m_BufferSize, buffer->m_BufferSize - 1, format, vargs);

    va_end(vargs);
    return result;
}

//---------------------------------------------------------------------------
//! @briefprivate
//! @details       バッファへ文字列を書式指定で書き込みます。
//!
//! @param[out]    buffer 書き込み先の StringBaseImpl のアドレスです。
//! @param[in]     format 書式です。
//!
//! @return        文字列の長さを返します。
template <>
NW_INLINE int
snprintf<char16>(StringBaseImpl<char16>* buffer, const char16* format, ...)
{
    std::va_list vargs;
    va_start(vargs, format);

    int result = snw16printf(buffer->m_CharArray, buffer->m_BufferSize, buffer->m_BufferSize - 1, format, vargs);

    va_end(vargs);
    return result;
}

//---------------------------------------------------------------------------
//! @brief        可変サイズの文字バッファクラスです。
//---------------------------------------------------------------------------
template <typename TChar>
class VariableStringImpl : public StringBaseImpl<TChar>
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //! @param[in]    allocator アロケータです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit VariableStringImpl( ut::IAllocator* allocator )
      : StringBaseImpl<TChar>()
      , m_Allocator( allocator )
    {
    }

    /* ctor */ VariableStringImpl( const TChar* str, ut::IAllocator* allocator )
      : StringBaseImpl<TChar>()
      , m_Allocator( allocator )
    {
        this->CopyFrom( str );
    }

    /* ctor */ VariableStringImpl( const TChar* str, u32 len, ut::IAllocator* allocator )
      : StringBaseImpl<TChar>()
      , m_Allocator( allocator )
    {
        this->CopyFrom( str, len );
    }

    /* ctor */ VariableStringImpl( const StringBaseImpl<TChar>& str, ut::IAllocator* allocator )
      : StringBaseImpl<TChar>()
      , m_Allocator( allocator )
    {
        this->CopyFrom( str.c_str() );
    }

    /* ctor */ VariableStringImpl( const VariableStringImpl<TChar>& str )
      : StringBaseImpl<TChar>()
      , m_Allocator( str.GetAllocator() )
    {
        this->CopyFrom( str.c_str() );
    }

    //---------------------------------------------------------------------------
    //! @brief        アロケータを取得します。
    //!
    //! @return       アロケータへのポインタを返します。
    //---------------------------------------------------------------------------
    ut::IAllocator* GetAllocator() const
    {
        return m_Allocator;
    }


    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ virtual ~VariableStringImpl()
    {
        this->FreeBuffer_( this->m_CharArray );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列をコピーします。
    //!
    //! @param[in]    str     コピー元の文字列です。
    //---------------------------------------------------------------------------
    void CopyFrom(const TChar* str)
    {
        uint len = 0;

        if ( str )
        {
            len = StringBaseImpl<TChar>::StrLen_( str );
        }

        this->CopyFrom( str, len );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列をコピーします。
    //!
    //! @param[in]    str     コピー元の文字列です。
    //! @param[in]    len     コピーする最大文字数です。
    //---------------------------------------------------------------------------
    void CopyFrom(const TChar* str, uint len)
    {
        if ( str == NULL )
        {
            NW_ASSERT( len == 0 );

            this->reserve( 0 );
        }

        this->ReserveImpl_( len + 1, false );

        StringBaseImpl<TChar>::StrCopy_( this->m_CharArray, str, len );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列をコピーします。
    //!
    //! @param[in]    str     コピー元の文字列です。
    //---------------------------------------------------------------------------
    void CopyFrom(const StringBaseImpl<TChar>& str)
    {
        this->CopyFrom( str.c_str() );
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーオペレータです。
    //!
    //! @param[in]    str  コピー元の文字列です。
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    VariableStringImpl& operator=(const VariableStringImpl<TChar>& str)
    {
        this->CopyFrom( str.c_str() );
        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーオペレータです。
    //!
    //! @param[in]    s    コピー元の文字列です。
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    VariableStringImpl& operator=(const TChar* s)
    {
        this->CopyFrom( s );
        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファへ文字列をコピーします。
    //!
    //! @param[out]   dst     コピー先のバッファアドレスです。
    //! @param[in]    dstSize コピー先のバッファアドレスに格納可能な文字数です。
    //! @param[in]    count   コピーする文字数です。
    //! @param[in]    offset  コピーする文字列のオフセットを指定します。
    //! @return       コピーした文字数です。
    //---------------------------------------------------------------------------
    uint CopyTo(TChar* dst, uint dstSize, uint count, uint offset = 0)
    {
        NW_ASSERT_NOT_NULL( dst );

        uint strLen = this->length();

        if (strLen <= offset)
        {
            return 0;
        }

        strLen = ut::Min( strLen - offset, count );

        if ( dstSize < strLen )
        {
            strLen = dstSize - 1;
        }

        StringBaseImpl<TChar>::StrCopy_( dst, this->m_CharArray + offset, strLen );
        return strLen;
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファへ文字列をコピーします。
    //!
    //! @param[out]   dst     コピー先のバッファアドレスです。
    //! @param[in]    dstSize コピー先のバッファアドレスに格納可能な文字数です。
    //! @return       コピーした文字数です。
    //---------------------------------------------------------------------------
    uint CopyTo(TChar* dst, uint dstSize)
    {
        return this->CopyTo( dst, dstSize, this->length() );
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファへ文字列をコピーします。
    //!
    //! @param[out]   dst     コピー先のバッファアドレスです。
    //! @param[in]    count   コピーする文字数です。
    //! @param[in]    offset  コピーする文字列のオフセットを指定します。
    //! @return       コピーした文字数です。
    //---------------------------------------------------------------------------
    uint copy(TChar* dst, uint count, uint offset = 0)
    {
        return this->CopyTo( dst, count + 1, count, offset );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列の加算演算子です。
    //! @param[in]    str 追加する文字列です。
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    const VariableStringImpl&    operator += ( const TChar* str ) { return this->append( str ); }
    const VariableStringImpl&    operator += ( const VariableStringImpl& str ) { return this->append( str ); }

    //---------------------------------------------------------------------------
    //! @brief        TChar 型へのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const TChar* () const { return this->c_str(); }

    //---------------------------------------------------------------------------
    //! @brief        バッファをあらかじめ確保します。
    //!
    //! @param[in]    size    最低限必要なバッファの文字数を指定します。
    //!                       内部的にはこれ以上のサイズのバッファが確保されます。
    //---------------------------------------------------------------------------
    void reserve( uint size )
    {
        ReserveImpl_( size, true );
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファを解放して NULL にします。
    //---------------------------------------------------------------------------
    void clear()
    {
        if ( this->m_CharArray )
        {
            this->FreeBuffer_( this->m_CharArray );
            this->m_CharArray = NULL;
            this->m_BufferSize = 0;
        }
    }


    //---------------------------------------------------------------------------
    //! @brief        文字列の最後に引数で指定した文字列を追加します。
    //!
    //! @param[in]    str     追加する文字列です。
    //!
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    VariableStringImpl& append( const TChar* str )
    {
        uint len = 0;

        if ( str )
        {
            len = StringBaseImpl<TChar>::StrLen_( str );
        }

        return this->append( str, len );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列の最後に引数で指定した文字列を追加します。
    //!
    //! @param[in]    str     追加する文字列です。
    //! @param[in]    len     追加する文字列長です。
    //!
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    VariableStringImpl& append( const TChar* str, uint len )
    {
        if ( ! str )
        {
            return *this;
        }

        uint strLen = len + 1;

        if ( this->m_CharArray )
        {
            strLen += this->length();
        }

        this->reserve( strLen );

        StringBaseImpl<TChar>::StrCat_( this->m_CharArray, str, len );

        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列の最後に引数で指定した文字列を追加します。
    //!
    //! @param[in]    str     追加する文字列です。
    //!
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    VariableStringImpl& append( const StringBaseImpl<TChar>& str )
    {
        return this->append( str.c_str() );
    }

private:
    //---------------------------------------------------------------------------
    //! @brief        アロケータからメモリを確保します。
    //!
    //! @param[in]    size    確保するメモリサイズです。
    //!
    //! @return       確保したバッファへのポインタを返します。
    //---------------------------------------------------------------------------
    TChar* AllocateBuffer_( u32 size )
    {
        NW_ASSERT_NOT_NULL( m_Allocator );

        return static_cast<TChar*>( m_Allocator->Alloc( size ) );
    }

    //---------------------------------------------------------------------------
    //! @brief        アロケータへメモリを返します。
    //!
    //! @param[in]    buffer   解放するバッファアドレスです。
    //---------------------------------------------------------------------------
    void FreeBuffer_( TChar* buffer )
    {
        if ( buffer )
        {
            NW_ASSERT_NOT_NULL( m_Allocator );
            m_Allocator->Free( buffer );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファを確保します。
    //!
    //! @param[in]    size    最低限必要なバッファの文字数を指定します。
    //!                       内部的にはこれ以上のサイズのバッファが確保されます。
    //! @param[in]    copyStr 元データをコピーするか指定します。
    //---------------------------------------------------------------------------
    void ReserveImpl_( uint size, bool copyStr )
    {
        if ( size == 0 )
        {
            size = 1;
        }

        // サイズが足りている場合はとりあえず何もしない。
        if ( this->m_BufferSize >= size )
        {
            return;
        }

        uint newBufferSize = ut::RoundUp( size, 32 );
        TChar* newBuffer = this->AllocateBuffer_( newBufferSize * sizeof(TChar) );

        if ( this->m_CharArray )
        {
            if ( copyStr )
            {
                StringBaseImpl<TChar>::StrCopy_( newBuffer, this->m_CharArray, newBufferSize );
            }

            this->FreeBuffer_( this->m_CharArray );
        }
        else
        {
            *newBuffer = 0; // NUL終端
        }

        this->m_CharArray  = newBuffer;
        this->m_BufferSize = newBufferSize;
    }

    ut::IAllocator*  m_Allocator;
};

//---------------------------------------------------------------------------
//! @brief        可変サイズの文字バッファクラスの char 版です。
//---------------------------------------------------------------------------
typedef VariableStringImpl<char>   VariableString;

//---------------------------------------------------------------------------
//! @brief        可変サイズの文字バッファクラスの char16 版です。
//---------------------------------------------------------------------------
typedef VariableStringImpl<char16> VariableString16;

//---------------------------------------------------------------------------
//! @brief        固定サイズの文字バッファクラスです。
//---------------------------------------------------------------------------
template <typename TChar>
class FixedStringImpl : public StringBaseImpl<TChar>
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @details      initStr なしのコンストラクタではバッファに入っている文字列をそのまま設定します。
    //!
    //! @param[in]    buffer     バッファアドレスです。
    //! @param[in]    bufferLen  バッファに格納可能な文字数です。
    //---------------------------------------------------------------------------
    /* ctor */ FixedStringImpl( TChar* buffer, uint bufferLen )
      : StringBaseImpl<TChar>()
    {
        this->m_CharArray = buffer;
        this->m_BufferSize = bufferLen;
        this->m_CharArray[bufferLen - 1] = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[out]   buffer     バッファアドレスです。
    //! @param[in]    bufferLen  バッファに格納可能な文字数です。
    //! @param[in]    initStr    初期化する文字列です。
    //---------------------------------------------------------------------------
    /* ctor */ FixedStringImpl( TChar* buffer, uint bufferLen, const TChar* initStr  )
      : StringBaseImpl<TChar>()
    {
        this->m_CharArray = buffer;
        this->m_BufferSize = bufferLen;
        ut::internal::strncpy_t(this->m_CharArray, bufferLen, initStr, bufferLen - 1);
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列をコピーします。
    //!
    //! @param[in]    str     コピー元の文字列です。
    //---------------------------------------------------------------------------
    void CopyFrom(const TChar* str)
    {
        uint len = 0;

        if ( str )
        {
            len = StringBaseImpl<TChar>::StrLen_( str );
        }

        this->CopyFrom( str, len );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列をコピーします。
    //!
    //! @param[in]    str     コピー元の文字列です。
    //! @param[in]    len     コピーする最大文字数です。
    //---------------------------------------------------------------------------
    void CopyFrom(const TChar* str, uint len)
    {
        if ( str == NULL )
        {
            NW_ASSERT( len == 0 );

            *this->m_CharArray = 0;

            return;
        }

        len = ut::Min(len, this->m_BufferSize - 1);

        StringBaseImpl<TChar>::StrCopy_( this->m_CharArray, str, len );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列をコピーします。
    //!
    //! @param[in]    str     コピー元の文字列です。
    //---------------------------------------------------------------------------
    void CopyFrom(const StringBaseImpl<TChar>& str)
    {
        this->CopyFrom( str.c_str() );
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーオペレータです。
    //!
    //! @param[in]    str  コピー元の文字列です。
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    FixedStringImpl& operator=(const FixedStringImpl<TChar>& str)
    {
        this->CopyFrom( str.c_str() );
        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        コピーオペレータです。
    //!
    //! @param[in]    s    コピー元の文字列です。
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    FixedStringImpl& operator=(const TChar* s)
    {
        this->CopyFrom( s );
        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファへ文字列をコピーします。
    //!
    //! @param[out]   dst     コピー先のバッファアドレスです。
    //! @param[in]    dstSize コピー先のバッファアドレスに格納可能な文字数です。
    //! @param[in]    count   コピーする文字数です。
    //! @param[in]    offset  コピーする文字列のオフセットを指定します。
    //! @return       コピーした文字数です。
    //---------------------------------------------------------------------------
    uint CopyTo(TChar* dst, uint dstSize, uint count, uint offset = 0)
    {
        NW_ASSERT_NOT_NULL( dst );

        uint strLen = this->length();

        if (strLen <= offset)
        {
            return 0;
        }

        strLen = ut::Min( strLen - offset, count );

        if ( dstSize < strLen )
        {
            strLen = dstSize - 1;
        }

        StringBaseImpl<TChar>::StrCopy_( dst, this->m_CharArray + offset, strLen );
        return strLen;
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファへ文字列をコピーします。
    //!
    //! @param[out]   dst     コピー先のバッファアドレスです。
    //! @param[in]    dstSize コピー先のバッファアドレスに格納可能な文字数です。
    //! @return       コピーした文字数です。
    //---------------------------------------------------------------------------
    uint CopyTo(TChar* dst, uint dstSize)
    {
        return this->CopyTo( dst, dstSize, this->length() );
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファへ文字列をコピーします。
    //!
    //! @param[out]   dst     コピー先のバッファアドレスです。
    //! @param[in]    count   コピーする文字数です。
    //! @param[in]    offset  コピーする文字列のオフセットを指定します。
    //! @return       コピーした文字数です。
    //---------------------------------------------------------------------------
    uint copy(TChar* dst, uint count, uint offset = 0)
    {
        return this->CopyTo( dst, count + 1, count, offset );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列の加算演算子です。
    //! @param[in]    str 追加する文字列です。
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    const FixedStringImpl&    operator += ( const TChar* str ) { return this->append( str ); }
    const FixedStringImpl&    operator += ( const FixedStringImpl& str ) { return this->append( str ); }

    //---------------------------------------------------------------------------
    //! @brief        TChar 型へのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const TChar* () const { return this->c_str(); }

    //---------------------------------------------------------------------------
    //! @brief        文字列の最後に引数で指定した文字列を追加します。
    //!
    //! @param[in]    str     追加する文字列です。
    //!
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    FixedStringImpl& append( const TChar* str )
    {
        uint len = 0;

        if ( str )
        {
            len = StringBaseImpl<TChar>::StrLen_( str );
        }

        return this->append( str, len );
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列の最後に引数で指定した文字列を追加します。
    //!
    //! @param[in]    str     追加する文字列です。
    //! @param[in]    len     追加する文字列長です。
    //!
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    FixedStringImpl& append( const TChar* str, uint len )
    {
        if ( ! str )
        {
            return *this;
        }

        if ( this->length() + len >= this->m_BufferSize )
        {
            len = this->m_BufferSize - this->length() - 1;
        }

        StringBaseImpl<TChar>::StrCat_( this->m_CharArray, str, len );

        return *this;
    }

    //---------------------------------------------------------------------------
    //! @brief        文字列の最後に引数で指定した文字列を追加します。
    //!
    //! @param[in]    str     追加する文字列です。
    //!
    //! @return       this ポインタへの参照を返します。
    //---------------------------------------------------------------------------
    FixedStringImpl& append( const StringBaseImpl<TChar>& str )
    {
        return this->append( str.c_str() );
    }
};

//---------------------------------------------------------------------------
//! @brief        固定サイズの文字バッファクラスの char 版です。
//---------------------------------------------------------------------------
typedef FixedStringImpl<char>   FixedString;

//---------------------------------------------------------------------------
//! @brief        固定サイズの文字バッファクラスの char16 版です。
//---------------------------------------------------------------------------
typedef FixedStringImpl<char16> FixedString16;

//@}


} // namespace ut
} // namespace nw

#endif //  NW_UT_STRING_H_
