﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_SIGNAL_H_
#define NW_UT_SIGNAL_H_

#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_MoveArray.h>

namespace nw
{
namespace ut
{

//----------------------------------------
//! @name シグナル関連
//@{

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        複数の呼び出しを行い、最後の返り値を返す関数オブジェクトです。
//---------------------------------------------------------------------------
template<typename TResult>
struct LastValueResult
{
    typedef TResult ResultType;
    template<typename TInputIterator, typename TInvoker>
    ResultType operator()(TInputIterator first, TInputIterator last, const TInvoker& invoker)
    {
        ResultType result;
        while (first != last)
        {
            result = invoker(*first);
            ++first;
        }
        return result;
    }
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        複数の呼び出しを行い、最後の返り値を返す関数オブジェクトです。
//---------------------------------------------------------------------------
template<>
struct LastValueResult<void>
{
    template<typename TInputIterator, typename TInvoker>
    void operator()(TInputIterator first, TInputIterator last, const TInvoker& invoker)
    {
        while (first != last)
        {
            invoker(*first);
            ++first;
        }
    }
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details      スロットを生成します。
//!
//! @tparam       TSlot スロットの型です。
//! @tparam       TFunction 関数ポインタまたは、関数オブジェクトの型です。
//!
//! @param[in]    allocator アロケータです。
//! @param[in]    function 関数または、関数オブジェクトです。
//!
//! @return       生成したスロットです。
//---------------------------------------------------------------------------
template<typename TSlot, typename TFunction>
NW_INLINE TSlot*
CreateSlot(
    ut::IAllocator* allocator,
    TFunction function
)
{
    NW_ASSERT_NOT_NULL(allocator);

    void* memory = allocator->Alloc(sizeof(TSlot));
    NW_ASSERT_NOT_NULL(memory);

    return new(memory) TSlot(allocator, function);
}

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details      スロットを破棄する関数オブジェクトです。
//!
//! @tparam       TSlot スロットの型です。
//---------------------------------------------------------------------------
template<typename TSlot>
struct SlotDestroyer : public std::unary_function<TSlot&, void>
{
    void operator()(TSlot& slot) const
    {
        slot->Destroy();
    }
};

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details      コンテナ要素の全てのスロットを破棄するための関数です。
//!
//! @tparam       TArray 削除するスロットの配列の型です。
//!
//! @param[in]    array 削除するスロットの配列です。
//---------------------------------------------------------------------------
template<typename TArray>
NW_INLINE void
DestroyAllSlots(
    TArray& array
)
{
    std::for_each(array.begin(), array.end(), SlotDestroyer<typename TArray::value_type>());
    array.clear();
}

//@}

// ----------------------------------------------------------------------------
// 自動生成したファイルをインクルードします。
// ----------------------------------------------------------------------------
#include <nw/ut/generated/ut_Signal.hi>

} // namespace ut
} // namespace nw

#endif // NW_UT_SIGNAL_H_
