﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_RUNTIMETYPEINFO_H_
#define NW_UT_RUNTIMETYPEINFO_H_

#include <stddef.h>
#include <nw/types.h>
#include <nw/ut/ut_TypeTraits.h>

namespace nw {
namespace ut {

//--------------------------------------------------------------------
//! @brief   ベースクラスの型情報定義マクロです。
//!
//! @details ダイナミックキャストを可能にする為に埋め込むべき実行時型情報を
//!          メンバに含めるマクロです。クラスの宣言中に記述してください。
//!
//!          nw::ut::DynamicCast 等の機能が使用できるようになります。
//--------------------------------------------------------------------
#define NW_UT_RUNTIME_TYPEINFO_ROOT()                                               \
    /*! @briefprivate */                                                        \
    /*! @details クラススタティックな型情報を取得します。 */                          \
    static  const nw::ut::internal::RuntimeTypeInfo*    GetRuntimeTypeInfoStatic()  \
    {                                                                               \
        static const nw::ut::internal::RuntimeTypeInfo s_TypeInfo(NULL);            \
        return &s_TypeInfo;                                                         \
    }                                                                               \
    /*! @briefprivate */                                                        \
    /*! @details virtual な型情報を取得します。 */                                    \
    virtual const nw::ut::internal::RuntimeTypeInfo*    GetRuntimeTypeInfo() const  \
    {                                                                               \
        return GetRuntimeTypeInfoStatic();                                          \
    }

//--------------------------------------------------------------------
//! @brief   サブクラスの型情報定義マクロです。
//!
//! @details ダイナミックキャストを可能にする為に埋め込むべき実行時型情報を
//!          メンバに含めるマクロです。クラスの宣言中に記述してください。
//!
//!          nw::ut::DynamicCast 等の機能が使用できるようになります。
//!
//! @param   base 親クラスのクラス名を指定します。
//--------------------------------------------------------------------
#define NW_UT_RUNTIME_TYPEINFO(base)                                                \
    /*! @briefprivate */                                                        \
    /*! @details クラススタティックな型情報を取得します。 */                          \
    static  const nw::ut::internal::RuntimeTypeInfo*    GetRuntimeTypeInfoStatic()  \
    {                                                                               \
        static const nw::ut::internal::RuntimeTypeInfo s_TypeInfo( base::GetRuntimeTypeInfoStatic() ); \
        return &s_TypeInfo;                                                         \
    }                                                                               \
    /*! @briefprivate */                                                        \
    /*! @details virtual な型情報を取得します。 */                                    \
    virtual const nw::ut::internal::RuntimeTypeInfo*    GetRuntimeTypeInfo() const  \
    {                                                                               \
        return GetRuntimeTypeInfoStatic();                                          \
    }


namespace internal {

//--------------------------------------------------------------------
//! @brief 実行時型情報構造体
//--------------------------------------------------------------------
struct RuntimeTypeInfo
{
    const RuntimeTypeInfo*  m_ParentTypeInfo; //!< ベースクラスの型情報へのポインタです。

    //--------------------------------------------------------------------
    //! @brief コンストラクタです。
    //! @param[in] parent 親の実行時型情報です。
    //--------------------------------------------------------------------
    /* ctor */ explicit  RuntimeTypeInfo( const RuntimeTypeInfo* parent ) : m_ParentTypeInfo( parent ) {}

    bool                 IsDerivedFrom( const RuntimeTypeInfo* s_TypeInfo ) const
    {
        const RuntimeTypeInfo *self = this;

        while ( self )
        {
            if ( self == s_TypeInfo )
            {
                return true;
            }
            self = self->m_ParentTypeInfo;
        }
        return false;
    }
};

} // namespace internal

//----------------------------------------
//! @name 基礎
//@{

//---------------------------------------------------------------------------
//! @brief        ダウンキャスト用テンプレート関数です。
//!
//! @tparam       TToPtr  キャスト先のポインタ型を指定します。
//! @param[out]   obj     キャストをおこなうインスタンスです。
//!
//! @return       ダウンキャストに成功した場合はキャスト先クラスのポインタ、失敗した場合は NULL を返します。
//---------------------------------------------------------------------------
template<typename TToPtr, typename TFrom>
NW_INLINE TToPtr
DynamicCast(TFrom* obj)
{
    const internal::RuntimeTypeInfo* typeInfoU = ut::PeelPointer<TToPtr>::Type::GetRuntimeTypeInfoStatic();
    if ( obj && obj->GetRuntimeTypeInfo()->IsDerivedFrom( typeInfoU ) )
    {
        return static_cast<TToPtr>(obj);
    }
    return NULL;
}

//---------------------------------------------------------------------------
//! @brief        ダウンキャスト可能かどうかの型チェックの ASSERT 付きスタティックキャストです。
//!               入力が NULL の場合は、ASSERT せず NULL を返します。
//!
//! @tparam       TToPtr  キャスト先のポインタ型を指定します。
//! @param[out]   obj     キャストをおこなうインスタンスです。
//!
//! @return       キャスト結果を返します。
//---------------------------------------------------------------------------
template<typename TToPtr, typename TFrom>
NW_INLINE TToPtr
StaticCast(TFrom* obj)
{
    NW_ASSERT( obj == NULL || DynamicCast<TToPtr>( obj ) != NULL );

    return static_cast<TToPtr>(obj);
}


//--------------------------------------------------------------------------
//! @brief        型の比較をおこないます。
//!
//! @tparam       To       比較する型です。
//! @param[in]    instance 型を比較するインスタンスです。
//!
//! @return       テンプレート引数に指定した型と同一のインスタンスであれば真を返します。
//---------------------------------------------------------------------------
template <typename To, typename From>
NW_INLINE bool
IsTypeOf(const From* instance)
{
    if (instance == NULL) { return false; }
    return instance->GetRuntimeTypeInfo() == To::GetRuntimeTypeInfoStatic();
}

//---------------------------------------------------------------------------
//! @brief        オブジェクトが指定した型にキャスト可能かどうかを取得します。
//!
//! @tparam       To       比較対象の型です。
//! @param[in]    instance 型をチェックするインスタンスです。
//!
//! @return       テンプレート引数に指定した型にキャスト可能であれば真を返します。
//---------------------------------------------------------------------------
template <typename To, typename From>
NW_INLINE bool
IsDerivedFrom(const From* instance)
{
    const internal::RuntimeTypeInfo* typeInfoU = To::GetRuntimeTypeInfoStatic();
    return instance && instance->GetRuntimeTypeInfo()->IsDerivedFrom( typeInfoU );
}

//---------------------------------------------------------------------------
//! @brief        インスタンスが同じ型かどうかを比較します。
//!               NULL の場合には結果は false になります。
//!
//! @param[in]    ptrA   比較するインスタンス１です。
//! @param[in]    ptrB   比較するインスタンス２です。
//!
//! @return       同一のクラスのインスタンスであれば true、そうでなければ false を返します。
//---------------------------------------------------------------------------
template< typename TypeA, typename TypeB >
NW_INLINE bool
IsSameType( const TypeA* ptrA, const TypeB* ptrB )
{
    if (ptrA == NULL || ptrB == NULL)
    {
        return false;
    }

    return ptrA->GetRuntimeTypeInfo() == ptrB->GetRuntimeTypeInfo();
}


//---------------------------------------------------------------------------
//! @brief        インスタンスが同じ型かどうかを比較します。
//!               両方が NULL の場合にも結果は true になります。
//!
//! @param[in]    ptrA   比較するインスタンス１です。
//! @param[in]    ptrB   比較するインスタンス２です。
//!
//! @return       同一のクラスのインスタンスであれば true、そうでなければ false を返します。
//---------------------------------------------------------------------------
template< typename TypeA, typename TypeB >
NW_INLINE bool
IsSameTypeOrNull( const TypeA* ptrA, const TypeB* ptrB )
{
    if (ptrA == NULL)
    {
        return ptrB == NULL;
    }

    return ptrA->GetRuntimeTypeInfo() == ptrB->GetRuntimeTypeInfo();
}


//@}

} // namespace ut
} // nemespace nw

// NW_UT_RUNTIMETYPEINFO_H_
#endif
