﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_MEMORY_H_
#define NW_UT_MEMORY_H_

#include <nw/types.h>
#include <nw/assert.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>
#include <nw/ut/ut_Inlines.h>
#include <functional>
#include <iterator>
#include <algorithm>
#include <string>

namespace nw {
namespace ut {

//---------------------------------------------------------------------------
//! @brief        メモリコピーをおこないます。
//!
//! @param[out]   dstp    コピー先のアドレスです。
//! @param[in]    srcp    コピー元のアドレスです。
//! @param[in]    size    コピーサイズです。
//---------------------------------------------------------------------------
NW_FORCE_INLINE void
MemCpy(void* dstp, const void* srcp, size_t size)
{
    std::memcpy(dstp, srcp, size);
}


//---------------------------------------------------------------------------
//! @brief        メモリアロケータのインターフェースです。
//---------------------------------------------------------------------------
class IAllocator
{
public:
    NW_UT_RUNTIME_TYPEINFO_ROOT();

    //! アライメント指定を省略した場合のデフォルト値です。
    static const int DEFAULT_ALIGNMENT = 4;

    //! キャッシュのアライメントです。
    static const int CACHE_LINE_ALIGNMENT = 64;

    //---------------------------------------------------------------------------
    //! @brief デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~IAllocator() {}

    //----------------------------------------
    //! @name メモリの確保と解放
    //@{

    //---------------------------------------------------------------------------
    //! @brief メモリを確保します。
    //! @param[in] size 確保するサイズです。
    //! @param[in] alignment アライメントです。
    //---------------------------------------------------------------------------
    virtual void* Alloc(size_t size, u32 alignment) = 0;

    //---------------------------------------------------------------------------
    //! @brief メモリを解放します。
    //! @param[in] memory 解放するメモリのアドレスです。
    //---------------------------------------------------------------------------
    virtual void  Free(void* memory) = 0;

    //@}

    //----------------------------------------
    //! @name ユーティリティ
    //@{

    //---------------------------------------------------------------------------
    //! @brief アライメント指定を省略してメモリを確保します。
    //!
    //! @details
    //!  派生クラスで Alloc(size, alignment)のオーバーライドを定義するとベースクラスの
    //!  オーバーロードが隠されるので注意してください。
    //!  サブクラスへの参照から直接利用するには、 using IAllocator::Alloc; を記述します。
    //!
    //! @param[in]    size      アロケートするメモリサイズです。
    //!
    //! @return       確保したメモリを返します。
    //---------------------------------------------------------------------------
    void* Alloc(size_t size) { return this->Alloc( size, DEFAULT_ALIGNMENT ); }

    //---------------------------------------------------------------------------
    //! @brief 指定個数分のオブジェクトに必要なメモリを確保します。
    //!
    //! @details バイトサイズではなく、オブジェクトの個数を引数とする事に注意してくさだい。
    //!
    //! @tparam    TObject   確保したポインタをキャストする型を指定します。
    //! @param[in] count     確保するオブジェクトの個数です。
    //! @param[in] alignment 確保するメモリのアライメントを指定します。
    //!
    //! @return    確保したメモリを TObject のポインタにキャストして返します。
    //---------------------------------------------------------------------------
    template<typename TObject>
    TObject* Alloc(int count, u32 alignment = DEFAULT_ALIGNMENT)
    {
        NW_ASSERT(count >= 0);
        return static_cast<TObject*>(this->Alloc(sizeof(TObject) * count, alignment));
    }

    //---------------------------------------------------------------------------
    //! @brief メモリを確保し、指定したバイトデータでクリアします。
    //!
    //! @param[in] size      アロケートするメモリサイズです。
    //! @param[in] data      メモリ領域をクリアするバイトデータです。
    //! @param[in] alignment 確保するメモリのアライメントを指定します。
    //!
    //! @return       確保したメモリを返します。
    //---------------------------------------------------------------------------
    void* AllocAndClear(size_t size, u8 data, u32 alignment = DEFAULT_ALIGNMENT)
    {
        u8* memory = static_cast<u8*>(this->Alloc(size, alignment));

        if (memory != NULL)
        {
            std::fill_n(NW_CHECKED_ARRAY_ITERATOR(memory, size), size,  data);
        }
        return memory;
    }

    //---------------------------------------------------------------------------
    //! @brief 指定個数分のオブジェクトに必要なメモリを確保し、指定したバイトデータでクリアします。
    //!
    //! @details バイトサイズではなく、オブジェクトの個数を引数とする事に注意してくさだい。
    //!
    //! @tparam    TObject   確保したポインタをキャストする型を指定します。
    //! @param[in] count     確保するオブジェクトの個数です。
    //! @param[in] data      メモリ領域をクリアするバイトデータです。
    //! @param[in] alignment 確保するメモリのアライメントを指定します。
    //!
    //! @return    確保したメモリを TObject のポインタにキャストして返します。
    //---------------------------------------------------------------------------
    template<typename TObject>
    TObject* AllocAndClear(int count, u8 data, u32 alignment = DEFAULT_ALIGNMENT)
    {
        NW_ASSERT(count >= 0);
        u8* memory = static_cast<u8*>(this->Alloc(sizeof(TObject) * count, alignment));

        if (memory != NULL)
        {
            std::fill_n(memory, count, data);
        }

        return static_cast<TObject*>(memory);
    }

    //---------------------------------------------------------------------------
    //! @brief 指定個数分のオブジェクトに必要なメモリを確保し、コンストラクタを呼び出します。
    //!
    //! @details バイトサイズではなく、オブジェクトの個数を引数とする事に注意してくさだい。
    //!
    //! @tparam    TObject   確保したポインタをキャストする型を指定します。
    //! @param[in] count     確保するオブジェクトの個数です。
    //! @param[in] alignment 確保するメモリのアライメントを指定します。
    //!
    //! @return    確保したメモリを TObject のポインタにキャストして返します。
    //---------------------------------------------------------------------------
    template<typename TObject>
    TObject* AllocAndConstruct(int count, u32 alignment = DEFAULT_ALIGNMENT)
    {
        NW_ASSERT(count >= 0);
        TObject* objects = static_cast<TObject*>(this->Alloc(sizeof(TObject) * count, alignment));

        if (objects != NULL)
        {
            for (int i = 0; i < count; ++i)
            {
                this->Construct(&objects[i]);
            }
        }

        return objects;
    }

    //---------------------------------------------------------------------------
    //! @brief 指定個数分のオブジェクトに必要なメモリを確保し、指定オブジェクトをコピーします。
    //!
    //! @details バイトサイズではなく、オブジェクトの個数を引数とする事に注意してくさだい。
    //!
    //! @tparam    TObject   確保したポインタをキャストする型を指定します。
    //! @param[in] count     確保するオブジェクトの個数です。
    //! @param[in] object    コピー元のオブジェクトです。
    //! @param[in] alignment 確保するメモリのアライメントを指定します。
    //!
    //! @return    確保したメモリを TObject のポインタにキャストして返します。
    //---------------------------------------------------------------------------
    template<typename TObject>
    TObject* AllocAndFill(int count, const TObject& object, u32 alignment = DEFAULT_ALIGNMENT)
    {
        NW_ASSERT(count >= 0);
        TObject* objects = static_cast<TObject*>(this->Alloc(sizeof(TObject) * count, alignment));

        if (objects != NULL)
        {
            std::fill_n(NW_CHECKED_ARRAY_ITERATOR(objects, count), count, object);
        }

        return objects;
    }

    //---------------------------------------------------------------------------
    //! @brief 指定個数分のオブジェクトのデストラクタを呼び出し、メモリを解放します。
    //!
    //! @param[out] objects 解放するオブジェクト配列のポインタです。
    //! @param[in]  count   解放するオブジェクトの個数です。
    //---------------------------------------------------------------------------
    template<typename TObject>
    void DestructAndFree(TObject* objects, int count)
    {
        NW_ASSERT(count >= 0);
        if (objects == NULL) { return; }

        for (int i = 0; i < count; ++i)
        {
            this->Destruct(&objects[i]);
        }

        Free(objects);
    }

    //---------------------------------------------------------------------------
    //! @brief オブジェクトのデストラクタを呼び出し、メモリを解放します。
    //!
    //! @param[out] object 解放するオブジェクトのポインタです。
    //---------------------------------------------------------------------------
    template<typename TObject>
    void DestructAndFree(TObject* object)
    {
        if (object == NULL) { return; }
        this->Destruct(object);
        Free(object);
    }

    //@}

private:
    //---------------------------------------------------------------------------
    //! @brief        割当て済みの領域にコンストラクタを呼び出します。
    //!
    //! @param[in,out]   object  コンストラクトをおこなうオブジェクトです。
    //---------------------------------------------------------------------------
    template<typename TObject>
    void Construct(TObject* object)
    {
        new(static_cast<void*>(object)) TObject;
    }

    //---------------------------------------------------------------------------
    //! @brief        初期化済みの領域のデストラクタを呼び出します。
    //!
    //! @param[in,out]  object  デスクトラクトをおこなうオブジェクトです。
    //---------------------------------------------------------------------------
    template<typename TObject>
    void Destruct(TObject* object)
    {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_UNUSED_VARIABLE(object); // VisualStudio の C4100 コンパイラバグの回避
#endif
        object->~TObject();
    }
};

//----------------------------------------
//! @name メモリ確保／解放
//@{

//---------------------------------------------------------------------------
//! @brief        必要な分だけメモリを確保して、文字列をコピーします。
//!
//! @tparam       TChar 文字の型です。
//!
//! @param[in]    allocator メモリアロケータです。
//! @param[in]    str コピーする文字列です。
//! @param[in]    maxLength コピー時に制限される最大文字数です。
//!
//! @return       コピーした文字列を返します。
//---------------------------------------------------------------------------
template<typename TChar>
NW_INLINE TChar*
AllocAndCopyString(IAllocator* allocator, const TChar* str, uint maxLength)
{
    if (str == NULL) { return NULL; }

    u32 length = std::char_traits<TChar>::length(str);
    length = (length < maxLength)? length : maxLength;

    uint bufferLength = length + 1;
    TChar* copyStr = reinterpret_cast<TChar*>(allocator->Alloc(bufferLength * sizeof(TChar)));
    NW_ASSERT_NOT_NULL(copyStr);
    NW_CHAR_TRAITS_COPY(TChar, copyStr, bufferLength, str, length);
    copyStr[length] = 0;

    return copyStr;
}

//---------------------------------------------------------------------------
//! @brief        メモリを解放後に NULL を設定するためのインライン関数です。
//!
//! @tparam       TMemory 解放するメモリの型です。
//!
//! @param[in]    memory 解放するメモリです。
//! @param[in]    allocator アロケータです。
//---------------------------------------------------------------------------
template<typename TMemory>
NW_INLINE void
SafeFree(TMemory*& memory, IAllocator* allocator)
{
    if (memory == NULL)
    {
        return;
    }

    allocator->Free(static_cast<void*>(memory));
    memory = NULL;
}

//---------------------------------------------------------------------------
//! @brief        デストラクタ呼出後、メモリを解放して NULL を設定するためのインライン関数です。
//!
//! @tparam       TObject 解放するオブジェクトの型です。
//!
//! @param[in]    object 解放するオブジェクトです。
//! @param[in]    allocator アロケータです。
//---------------------------------------------------------------------------
template<typename TObject>
NW_INLINE void
SafeFreeWithDestruct(TObject*& object, IAllocator* allocator)
{
    if (object == NULL)
    {
        return;
    }

    allocator->DestructAndFree(object);
    object = NULL;
}

//---------------------------------------------------------------------------
//! @brief        SafeFree でメモリを解放するためのデリーターです。
//!
//! @tparam       TMemory 削除するメモリの型です。
//---------------------------------------------------------------------------
template<typename TMemory>
struct SafeFreeFunctor : public std::unary_function<TMemory&, void>
{
    explicit SafeFreeFunctor(IAllocator* allocator) : m_Allocator(allocator) {}

    //! @brief メモリを破棄します。
    //! @param[in] memory 破棄するメモリです。
    void operator()(TMemory& memory) const
    {
        SafeFree(memory, m_Allocator);
    }

    IAllocator* m_Allocator;
};

//---------------------------------------------------------------------------
//! @brief        SafeFree でコンテナ要素の全てのメモリを解放するための関数です。
//!
//! @tparam       TArray 解放するコンテナの型です。
//!
//! @param[in]    array 削除するメモリのコンテナです。
//! @param[in]    allocator アロケータです。
//---------------------------------------------------------------------------
template<typename TArray>
inline void
SafeFreeAll(
    TArray& array, IAllocator* allocator
)
{
    std::for_each(array.begin(), array.end(),
                  SafeFreeFunctor<typename TArray::value_type>(allocator));
    array.clear();
}

//@}

namespace internal {

//---------------------------------------------------------------------------
//! @brief        アラインメントを考慮したメモリサイズ計算を行います。
//!
//! :private
//---------------------------------------------------------------------------
class MemorySizeCalculator
{
public:
    static const size_t MAX_ALIGNMENT = IAllocator::CACHE_LINE_ALIGNMENT;

    explicit MemorySizeCalculator(size_t alignment)
    : m_Size(0),
      m_Alignment(alignment),
      m_MaxAlignment(alignment)
    {
        NW_ASSERT(alignment <= MAX_ALIGNMENT);
        NW_TASSERTMSG(alignment != 0 && (alignment & alignment - 1) == 0, "alignment must be power of 2");
    }

    MemorySizeCalculator& operator+=(size_t size)
    {
        m_MaxAlignment = ut::Max(m_MaxAlignment, m_Alignment);
        m_Size = ut::RoundUp(m_Size, static_cast<u32>(m_Alignment)) + size;
        return *this;
    }

    void ChangeAlignment(size_t alignment)
    {
        NW_ASSERT(alignment <= MAX_ALIGNMENT);
        NW_TASSERTMSG(alignment != 0 && (alignment & alignment - 1) == 0, "alignment must be power of 2");
        m_Alignment = alignment;
    }

    void Add(size_t size, size_t immediateAlignment)
    {
        size_t oldAlignment = m_Alignment;

        ChangeAlignment(immediateAlignment);
        *this += size;
        ChangeAlignment(oldAlignment);
    }

    MemorySizeCalculator& operator*=(int operand)
    {
        m_Size = ut::RoundUp(m_Size, static_cast<u32>(m_Alignment)) * operand;
        return *this;
    }

    size_t GetAlignment() const {
        return m_Alignment;
    }

    size_t GetMaxAlignment() const {
        return m_MaxAlignment;
    }

    size_t GetSizeWithPadding(size_t alignment) const {
        return m_Size + m_MaxAlignment - alignment;
    }

private:
    size_t m_Size;
    size_t m_Alignment;

    size_t m_MaxAlignment;
};

} // namespace internal

} // namespace ut
} // namespace nw


/* NW_UT_MEMORY_H_ */
#endif
