﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_IO_STREAM_H_
#define NW_UT_IO_STREAM_H_

#include <nw/ut/ut_RuntimeTypeInfo.h>

namespace nw   {
namespace ut   {

//-------------------------------------------------
//! @brief データストリーム基底クラス
//-------------------------------------------------
class IoStream
{
//------ publicメンバ --------------------------
public:
    NW_UT_RUNTIME_TYPEINFO_ROOT()      // ダウンキャスト用の実行時型情報を埋め込みます

    //! 非同期コールバック関数
    typedef void (*IoStreamCallback)( s32 result, IoStream* stream, void* arg );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ virtual  ~IoStream() {}

    //---------------------------------------------------------------------------
    //! @brief        ストリームのクローズ
    //---------------------------------------------------------------------------
    virtual void        Close()                               = 0;

    //---------------------------------------------------------------------------
    //! @brief       ストリームからデータを読み込みます(同期処理)
    //!
    //! @param[in]   buf         読み込み先バッファへのポインタ
    //! @param[in]   length      読み込むデータ長
    //!
    //! @return      実際に読み込まれたデータサイズを返します。
    //---------------------------------------------------------------------------
    virtual s32         Read( void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief       ストリームからデータを読み込みます(非同期処理)
    //!
    //! @param[in]   buf         読み込み先バッファへのポインタ
    //! @param[in]   length      読み込むデータ長
    //! @param[in]   callback    読み込み処理完了時のコールバック
    //! @param[in]   arg         読み込み処理完了時のコールバック引数
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には 偽 を返します。
    //!              現状は常に偽を返すため、使用時はオーバーライドして実装頂く必要があります。
    //---------------------------------------------------------------------------
    virtual bool        ReadAsync(
                            void*               buf,
                            u32                 length,
                            IoStreamCallback    callback,
                            void*               arg );

    //---------------------------------------------------------------------------
    //! @brief       ストリームにデータを書き込みます。(同期版)
    //!
    //! @param[in]   buf         書き込み元バッファ
    //! @param[in]   length      書き込みデータのバイト数
    //!
    //! @return      実際に書き込まれたバイト数を返します。
    //!               書き込みに失敗した場合には負のエラーコードを返します。
    //---------------------------------------------------------------------------
    virtual s32         Write( const void* buf, u32 length );

    //---------------------------------------------------------------------------
    //! @brief       ストリームにデータを書き込みます。(非同期版)
    //!
    //! @param[in]   buf         書き込み元バッファへのポインタ
    //! @param[in]   length      書き込むデータ長
    //! @param[in]   callback    非同期処理完了時のコールバック
    //! @param[in]   arg         非同期処理完了時のコールバック引数
    //!
    //! @return      コマンドが正常に発行された場合には真を返します。
    //!              発行されなかった場合には偽を返します。
    //!              現状は常に偽を返すため、使用時はオーバーライドして実装頂く必要があります。
    //---------------------------------------------------------------------------
    virtual bool        WriteAsync(
                            const void*         buf,
                            u32                 length,
                            IoStreamCallback    callback,
                            void*               arg );

    //---------------------------------------------------------------------------
    //! @brief       非同期処理の完了を待ちます
    //!
    //! @return      非同期処理の結果を返します。
    //!               コマンドが失敗した場合には負のエラーコードを返します。
    //!               コマンドが成功した場合にはそれ以外の値を返します。
    //---------------------------------------------------------------------------
    virtual s32          WaitAsync();

    //---------------------------------------------------------------------------
    //! @brief       非同期処理の完了を待ちます
    //!
    //! @return      処理中の場合には真を返します。
    //!               処理中でない場合には偽を返します。
    //---------------------------------------------------------------------------
    virtual bool        IsBusy() const;

    //---------------------------------------------------------------------------
    //! @brief 非同期処理の可否を取得します。
    //! @return 非同期処理が可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanAsync () const                      = 0;

    //---------------------------------------------------------------------------
    //! @brief リード処理の可否を取得します。
    //! @return リード処理が可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanRead  () const                      = 0;

    //---------------------------------------------------------------------------
    //! @brief ライト処理の可否を取得します。
    //! @return ライト処理が可能であれば true を返します。
    //---------------------------------------------------------------------------
    virtual bool        CanWrite () const                      = 0;

    //---------------------------------------------------------------------------
    //! @brief        ストリームがオープンできているかを取得します。
    //!
    //! @return       ストリームがオープンできている場合には true を返します。
    //---------------------------------------------------------------------------
    virtual bool        IsAvailable()    const                 = 0;

    // アクセスの際の必要アライメントを取得します
    //---------------------------------------------------------------------------
    //! @brief        アクセスの際のファイルオフセットの必要アラインを取得します。
    //!
    //! @return       アクセスの際のファイルオフセットの必要アラインです。
    //---------------------------------------------------------------------------
    virtual u32         GetOffsetAlign() const { return 1; }

    //---------------------------------------------------------------------------
    //! @brief        アクセスサイズの必要アラインを取得します。
    //!
    //! @return       ファイルのリードサイズの必要アラインです。
    //---------------------------------------------------------------------------
    virtual u32         GetSizeAlign  () const { return 1; }

    //---------------------------------------------------------------------------
    //! @brief        送受信用バッファの必要アラインを取得します。
    //!
    //! @return       送受信用バッファの必要アラインを返します。
    //---------------------------------------------------------------------------
    virtual u32         GetBufferAlign() const { return 1; }

//------ protectedメンバ --------------------------
protected:
    //---  コンストラクタ -----------
    /* ctor */          IoStream()
                          : m_Callback ( NULL  ),
                            m_Arg      ( NULL  )
                        {}

    s32                 m_AsyncResult;   // 非同期処理の結果格納
    IoStreamCallback    m_Callback;      // 非同期処理コールバック
    void*               m_Arg;           // コールバック引数
};

}   /* namespace ut     */
}   /* namespace nw   */



/* NW_UT_IO_STREAM_H_ */
#endif
