﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_UT_HEAP_BASE_H_
#define NW_UT_HEAP_BASE_H_

#include <nw/ut/ut_LinkList.h>

#include <stddef.h>         // for offsetof()

namespace nw {
namespace ut {

//! @briefprivate
//! @details 継承する子クラスをリストで管理するためのクラス
class HeapNode
{
  public:
    HeapNode() {}
    virtual ~HeapNode() {}

  public:
    LinkListNode mLinkNode;
};


//! @briefprivate
//! @details 拡張ヒープ、フレームヒープ、ユニットヒープの基底
class HeapBase : public HeapNode
{
  public:
    //
    //  type
    //

    typedef ut::LinkList< HeapBase, offsetof(HeapNode,mLinkNode)> HeapList;


    //
    //  constant value
    //

    // ヒープからメモリを割り当てるときのデフォルトアライメントサイズ
    static const int DEFAULT_ALIGNMENT  = 4;

    // 拡張ヒープ・フレームヒープ・ユニットヒープのシグネチャ
    static const u32 EXPHEAP_SIGNATURE  = 'EXPH';
    static const u32 FRMHEAP_SIGNATURE  = 'FRMH';
    static const u32 UNTHEAP_SIGNATURE  = 'UNTH';

    // メモリ確保時のオプション
    static const int OPT_0_CLEAR        = ( 1 << 0 );   // メモリ確保時に、メモリをゼロクリア
    static const int OPT_DEBUG_FILL     = ( 1 << 1 );   // ヒープ作成時・メモリ確保・解放時にメモリ充填
    static const int OPT_THREAD_SAFE    = ( 1 << 2 );   // 排他制御の有無

    // このビットが立っているとエラー出力
    static const int ERROR_PRINT            = ( 1 << 0 );


    //
    //  enum
    //

    enum FillType {
        HEAP_FILL_NOUSE,    // デバッグフィル未使用時
        HEAP_FILL_ALLOC,    // デバッグフィル確保時
        HEAP_FILL_FREE,     // デバッグフィル解放時
        HEAP_FILL_MAX
    };

    enum HeapType {
        HEAP_TYPE_EXP,      // 拡張ヒープ
        HEAP_TYPE_FRM,      // フレームヒープ
        HEAP_TYPE_UNIT,     // ユニットヒープ
        HEAP_TYPE_UNKNOWN   // 未知のヒープ
    };


    //
    //  function
    //

    static HeapBase* FindContainHeap( const void* memBlock );   // TODO: static ???
    static HeapBase* FindParentHeap( const HeapBase* heap );    // TODO: static ???

    void* GetHeapStartAddress() { return this; }
    void* GetHeapEndAddress()   { return mHeapEnd; }

    s32 GetTotalSize()          { return ((s32)(mHeapEnd) - (s32)(this)); }
    s32 GetTotalUsableSize()    { return ((s32)(mHeapEnd) - (s32)(mHeapStart)); }

//    void Dump();

    u32 SetFillValue( FillType type, u32 val );
    u32 GetFillValue( FillType type );

    HeapType GetHeapType();

  protected:
    static const int MIN_ALIGNMENT = 4;

    void Initialize( u32 signature, void* start, void* end, u16 optFlag );
    void Finalize();

    u32 GetSignature() const { return mSignature; }
    void* GetHeapStart() const { return mHeapStart; }
    void* GetHeapEnd() const { return mHeapEnd; }

    void LockHeap();
    void UnlockHeap();

    void FillFreeMemory( void* address, u32 size );
    void FillNoUseMemory( void* address, u32 size );
    void FillAllocMemory( void* address, u32 size );

    void*       mHeapStart;     // ヒープ先頭アドレス
    void*       mHeapEnd;       // ヒープ末尾 (+1) アドレス

  private:
    static HeapBase* FindContainHeap( HeapList* pList, const void* memBlock );
    static HeapList* FindListContainHeap( HeapBase* pHeapBase );

    u16  GetOptionFlag();
    void SetOptionFlag( u16 optFlag );

    u32         mSignature;
    HeapList    mChildList;
#ifdef NW_UT_HEAP_MULTITHREADED
    OSMutex     mMutex;
#endif
    u32         mAttribute;     // 属性 [8:オプションフラグ]
};

} // nw::ut
} // nw


#endif // NW_UT_HEAP_BASE_H_

