﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_SPY_MODULE_H_
#define NW_SND_SPY_SPY_MODULE_H_

#include <nw/snd/spy/sndspy_SpyDataID.h>
#include <nw/snd/spy/sndspy_SpyDataInfo.h>

#include <nw/snd/spy/sndspy_Config.h>
#ifdef NW_SND_SPY_ENABLE

namespace nw {
namespace snd {
namespace spy {

class SpyController;
class SpyDataInfo;

//! @brief SpyController が提供するインフラを使って、データを送信するためのクラスです。
class SpyModule
{
protected:
    SpyModule()
        : m_Prev(NULL)
        , m_Next(NULL)
        , m_Controller(NULL)
        , m_RegistCount(0)
        , m_IsRequested(false)
    {
    }

public:
    //! @brief 対象の SpyDataInfo を返します。
    virtual const SpyDataInfo& GetDataInfo() const = 0;

    //! @brief 対象の SpyController を返します。
    //!
    //! SpyController との関連付けを保持するために使用します。
    //!
    SpyController* GetController() const
    {
        return m_Controller;
    }

    //! @brief SpyController にインストールされているか調べます。
    //!
    //! @return
    //! インストール済みなら true を返します。
    //!
    bool IsInstalled() const
    {
        return m_Controller != NULL;
    }

    //! @brief SoundSpy からのデータ要求の状態を取得します。
    //!
    //! セッションの開始時、および終了時に false に初期化されます。
    //!
    //! @return
    //! データが要求されていれば true を返します。
    //!
    bool IsRequested() const
    {
        return m_IsRequested;
    }

protected:
    //! @brief 対象の SpyDataInfo を返します。
    virtual SpyDataInfo& GetDataInfo() = 0;

    //! @brief セッションの開始を通知します。
    //!
    //! 導出クラスでは、最初に基底クラスの実装を呼びだしてください。
    //!
    //! この関数をオーバーライドすることで、
    //! セッション中の初めてのデータ送信を区別するための
    //! フラグを初期化するなどの処理をおこなうことができます。
    //!
    virtual void OnSessionStarted()
    {
        m_IsRequested = false;
    }

    //! @brief SoundSpy からのデータ要求を通知します。
    //!
    //! SoundSpy からデータ要求が来たときに、各データインフォに、
    //! データ要求の有無が通知されます。
    //!
    //! セッションの終了時には bRequested = false で呼び出されます。
    //!
    //! 導出クラスでは、最初に基底クラスの実装を呼びだしてください。
    //!
    //! @param[in] isRequested  データが要求されているときは true が指定されます。
    //!
    virtual void OnRequested(bool isRequested)
    {
        m_IsRequested = isRequested;
    }

private:
    //! @brief リンクリストの前の要素を返します。
    //!
    //! SpyController が、SpyDataInitializer の保持に使用します。
    //!
    SpyModule* GetPrev() const
    {
        return m_Prev;
    }

    //! @brief リンクリストの前の要素を設定します。
    //!
    //! SpyModule が、SpyDataInitializer の保持に使用します。
    //!
    void SetPrev(SpyModule* prev)
    {
        m_Prev = prev;
    }

    //! @brief リンクリストの次の要素を返します。
    //!
    //! SpyController が、SpyModule の保持に使用します。
    //!
    SpyModule* GetNext() const
    {
        return m_Next;
    }

    //! @brief リンクリストの次の要素を設定します。
    //!
    //! SpyController が、SpyModule の保持に使用します。
    //!
    void SetNext(SpyModule* next)
    {
        m_Next = next;
    }

    //! @brief SpyController と関連付けます。
    //!
    //! @param[in] controller  関連付ける SpyController を指定します。
    //!
    void Attach(SpyController& controller);

    //! @brief SpyController との関連付けを解除します。
    void Detach();

    void AllocateDataID(SpyDataID dataID);

private:
    SpyModule* m_Prev;
    SpyModule* m_Next;

    SpyController* m_Controller;

    u8 m_RegistCount;
    volatile bool m_IsRequested;

    friend class nw::snd::spy::SpyController;
};

//!
//! @brief SpyModule の簡単な実装を提供するクラスです。
//!
class SpySimpleModule : public SpyModule
{
public:
    SpySimpleModule(const char* dataName, const u32 dataVersion)
        : m_DataInfo(dataName, dataVersion)
    {
    }

    //! @brief 対象の SpyDataInfo を返します。
    virtual const SpyDataInfo& GetDataInfo() const
    {
        return m_DataInfo;
    }

protected:
    //! @brief 対象の SpyDataInfo を返します。
    virtual SpyDataInfo& GetDataInfo()
    {
        return m_DataInfo;
    }

private:
    SpySimpleDataInfo m_DataInfo;
};

} // namespace spy
} // namespace snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE

#endif // NW_SND_SPY_SPY_MODULE_H_
