﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SPY_FND_THREAD_H_
#define NW_SND_SPY_FND_THREAD_H_

#include <nw/snd/spy/fnd/basis/sndspyfnd_Config.h>
#include <nw/snd/spy/fnd/basis/sndspyfnd_Time.h>

namespace nw {
namespace snd {
namespace spy {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  スレッドを制御するクラスです。
//---------------------------------------------------------------------------
class Thread
{
public: // 定数の定義
    static const u32 INVALID_ID;                        //!< 無効なIDです。
    static const u32 DEFAULT_THREAD_PRIORITY = 16;      //!< プライオリティのデフォルト値です。
    static const u32 MIN_THREAD_PRIORITY = 0;           //!< プライオリティの最小値です。
    static const u32 MAX_THREAD_PRIORITY = 31;          //!< プライオリティの最大値です。

private: // 定数の定義
#if defined(NW_PLATFORM_CAFE)
    static const u32 OSTHREAD_ALIGNMENT = 8;            //!< OSThread 構造体のアライメントです。
#endif

public: // 型の定義
#if defined(NW_PLATFORM_CAFE)
    typedef OSThread* Handle;
#elif defined(NW_PLATFORM_CTR)
    typedef nn::os::Thread Handle;
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    typedef void* Handle;
#endif

    //! @brief  スレッドの Affinity Mask です。
    enum AFFINITY_MASK
    {
        CORE_DEFAULT = 0,
        CORE_ALL = 0xffffffff,

        CORE_0  = 1 << 0,
        CORE_1  = 1 << 1,
        CORE_2  = 1 << 2,
        CORE_3  = 1 << 3,
        CORE_4  = 1 << 4,
        CORE_5  = 1 << 5,
        CORE_6  = 1 << 6,
        CORE_7  = 1 << 7,
        CORE_8  = 1 << 8,
        CORE_9  = 1 << 9,
        CORE_10 = 1 << 10,
        CORE_11 = 1 << 11,
        CORE_12 = 1 << 12,
        CORE_13 = 1 << 13,
        CORE_14 = 1 << 14,
        CORE_15 = 1 << 15,
        CORE_16 = 1 << 16,
        CORE_17 = 1 << 17,
        CORE_18 = 1 << 18,
        CORE_19 = 1 << 19,
        CORE_20 = 1 << 20,
        CORE_21 = 1 << 21,
        CORE_22 = 1 << 22,
        CORE_23 = 1 << 23,
        CORE_24 = 1 << 24,
        CORE_25 = 1 << 25,
        CORE_26 = 1 << 26,
        CORE_27 = 1 << 27,
        CORE_28 = 1 << 28,
        CORE_29 = 1 << 29,
        CORE_30 = 1 << 30,
        CORE_31 = 0x80000000
    };

    //! @brief  スレッドハンドラです。
    class Handler
    {
    public:
        virtual ~Handler() { }

    public:
        virtual u32 Run(void* param) = 0;
    };

    //! @brief  スレッドハンドラのデリゲートです。
    //!         スレッド処理を指定メソッドに委譲します。
    template<class TTarget>
    class HandlerDelegate : public Handler
    {
    public:
        typedef u32 (TTarget::*HandlerPtr)(void* param);

        HandlerDelegate() : m_Target(NULL), m_HandlerFunc(NULL) { }

        //---------------------------------------------------------------------------
        //! @brief         初期化します。
        //!
        //! @param[in]     target       handlerFunc を所有するインスタンスを指定します。
        //! @param[in]     handlerFunc  &TTarget::HandlerFunc のように書きます。
        //---------------------------------------------------------------------------
        void Initialize(TTarget* target, HandlerPtr handlerFunc)
        {
            NW_ASSERT_NOT_NULL(target);
            NW_ASSERT_NOT_NULL(handlerFunc);
            m_Target = target;
            m_HandlerFunc = handlerFunc;
        }

        virtual u32 Run(void* param)
        {
            NW_ASSERT_NOT_NULL(m_Target);
            NW_ASSERT_NOT_NULL(m_HandlerFunc);
            return (m_Target->*this->m_HandlerFunc)(param);
        }

    private:
        TTarget*   m_Target;
        HandlerPtr m_HandlerFunc;
    };

    //! @brief  スレッド実行のパラメータです。
    struct RunArgs
    {
        //! @brief  コンストラクタです。
        RunArgs();

        //! @brief  パラメータの有効性を調べます。
        //! @return パラメータが有効なら true を返します。
        bool IsValid() const;

        const char*   name;             //!< スレッドの名前です。
        void*         stack;            //!< スレッドのスタックです。
        u32           stackSize;        //!< スレッドのスタックサイズです。
        AFFINITY_MASK affinityMask;     //!< スレッドの Affinity Mask です。
        u32           priority;         //!< スレッドのプライオリティです。
        void*         param;            //!< スレッドに渡すユーザーパラメータです。

        Handler*      handler;          //!< スレッドハンドラです。
    };

    //! @brief  スレッドの状態を示します。
    enum State
    {
        STATE_NOT_RUN = 0,  //!< スレッドは一度も実行されていません。
        STATE_RUNNING,      //!< スレッドは実行中です。
        STATE_EXITED,       //!< スレッドは終了済みです。
        STATE_RELEASED      //!< スレッドのリソースが解放されており再利用可能な状態です。
    };

private:
    class ThreadMain;

public:
    Thread();
    ~Thread();

public:
    //! @brief  スレッドを実行します。
    //! @param[in] args パラメータです。
    //! @return スレッドの実行に成功した場合は true を返します。
    bool Run(const RunArgs& args);

    //! @brief  スレッドが終了するまで待機します。
    void WaitForExit();

    //! @brief  スレッドの所有権を開放し、再度実行可能な状態にします。
    void Release();

    //! @brief  プライオリティを取得します。
    //! @return プライオリティを返します。
    s32 GetPriority() const;

    //! @brief  プライオリティを設定します。
    //! @param[in] value プライオリティです。
    void SetPriority(s32 value);

    //! @brief  スレッドの状態を取得します。
    //! @return スレッドの状態を返します。
    State GetState() const;

    //! @brief  カレントスレッドをスリープします。
    //! @param[in] timeSpan スリープから復帰するまでの時間です。
    static void Sleep(const TimeSpan& timeSpan);

private:
    bool Create(Handle& handle, u32& id, const RunArgs& args);
    void Detach();
    void SetName(const char* name);
    void SetAffinityMask(AFFINITY_MASK value);
    void Resume();
    void Join();
    bool IsTerminated() const;

    void SetState(State value);

    void OnRun();
    void OnExit();

private:
    u32         m_State;        //!< スレッドの状態です。
    Handle      m_Handle;       //!< スレッドハンドルです。
    u32         m_ID;           //!< スレッドIDです。
    u32         m_Priority;     //!< スレッドのプライオリティです。
    const char* m_Name;         //!< スレッドの名前です。
    void*       m_Param;        //!< ユーザーパラメータです。

    Handler*    m_Handler;      //!< スレッドハンドラです。

#if defined(NW_PLATFORM_CAFE)
    //! @brief  アライメントのためのスレッドハンドルバッファです。
    u8 m_HandleBuffer[sizeof(OSThread) + OSTHREAD_ALIGNMENT];
#endif
};

} // namespace nw::snd::spy::internal::fnd
} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_FND_THREAD_H_
