﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_WaveSoundPlayer.h
 *
 * @file snd_WaveSoundPlayer.h
 */

#ifndef NW_SND_WAVE_SOUND_PLAYER_H_
#define NW_SND_WAVE_SOUND_PLAYER_H_

#include <nw/snd/snd_BasicSoundPlayer.h>
#include <nw/snd/snd_DisposeCallback.h>
#include <nw/snd/snd_CurveLfo.h>
#include <nw/snd/snd_WaveFileReader.h>
#include <nw/snd/snd_WaveSoundFileReader.h>   // WaveSoundInfo, WaveSoundNoteInfo
#include <nw/snd/snd_Channel.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_LoaderManager.h>
#include <nw/snd/snd_Task.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    ウェーブサウンドデータのパラメータセットです。
//!
//!           この情報は @ref WaveSoundHandle::ReadWaveSoundDataInfo から取得できます。
//!
//! @see WaveSoundHandle::ReadWaveSoundDataInfo
//!
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
struct WaveSoundDataInfo
{
    //---------------------------------------------------------------------------
    //! @brief    ウェーブサウンドデータがループするなら true、
    //!           終端で終了するなら false となります。
    //---------------------------------------------------------------------------
    bool loopFlag;

    //---------------------------------------------------------------------------
    //! @brief    ウェーブサウンドデータのサンプリングレートです。
    //---------------------------------------------------------------------------
    int sampleRate;

    //---------------------------------------------------------------------------
    //! @brief    ウェーブサウンドデータがループする時のループ開始位置を、
    //!           ウェーブサウンドの先頭からのサンプル数で表します。
    //---------------------------------------------------------------------------
    u32 loopStart;

    //---------------------------------------------------------------------------
    //! @brief    ウェーブサウンドデータがループする時のループ終了位置を、
    //!           ウェーブサウンドの先頭からのサンプル数で表します。
    //!           ループしない時は、データの終端をサンプル数で表します。
    //!           (再生される最後のサンプルの次のサンプルを指します)
    //---------------------------------------------------------------------------
    u32 loopEnd;

    //---------------------------------------------------------------------------
    //! @brief    ウェーブサウンドデータがループする時のループ開始位置を、
    //!           ウェーブサウンドの先頭からのサンプル数で表します。
    //!
    //!           0.3.0 より loopStart の値が、ハードウェアの制約による
    //!           補正がかかる前の値を返すように変更されました。
    //!           以前の値を取得したい場合にはこちらの値を使用してください。
    //!           ただし、このメンバの使用は非推奨で、将来的に削除予定です。
    //---------------------------------------------------------------------------
    u32 compatibleLoopStart;

    //---------------------------------------------------------------------------
    //! @brief    ウェーブサウンドデータがループする時のループ終了位置を、
    //!           ウェーブサウンドの先頭からのサンプル数で表します。
    //!           ループしない時は、データの終端をサンプル数で表します。
    //!
    //!           0.3.0 より loopEnd の値が、ハードウェアの制約による
    //!           補正がかかる前の値を返すように変更されました。
    //!           以前の値を取得したい場合にはこちらの値を使用してください。
    //!           ただし、このメンバの使用は非推奨で、将来的に削除予定です。
    //---------------------------------------------------------------------------
    u32 compatibleLoopEnd;

    //! @briefprivate
    void Dump()
    {
        NW_LOG("WaveSoundDataInfo::%s loop?(%d) rate(%d) loopStart(%d) loopEnd(%d) compatibleLoopStart(%d) compatibleLoopEnd(%d)\n",
                __FUNCTION__, loopFlag, sampleRate, loopStart, loopEnd, compatibleLoopStart, compatibleLoopEnd);
    }
};

class SoundDataManager;
class SoundPlayer;

namespace internal {

class PlayerHeap;

namespace driver {

class WaveSoundLoader;
typedef LoaderManager<WaveSoundLoader> WaveSoundLoaderManager;

// プレイヤーヒープへのロードをつかさどる
class WaveSoundLoader
{
public:
    struct LoadInfo
    {
        const SoundArchive* soundArchive;
        const SoundDataManager* soundDataManager;
        const LoadItemInfo* loadInfoWsd;
        SoundPlayer* soundPlayer;

        LoadInfo(
            const SoundArchive* arc,
            const SoundDataManager* mgr,
            const LoadItemInfo* wsd,
            SoundPlayer* player) :
        soundArchive(arc),
        soundDataManager(mgr),
        loadInfoWsd(wsd),
        soundPlayer(player)
        {}
    };

    struct Data
    {
        const void* wsdFile;   // メモリ上の bXwsd ファイル
        const void* waveFile;

        Data() : wsdFile(NULL), waveFile(NULL) {}
        void Initialize() { wsdFile = waveFile = NULL; }
    };

    struct Arg
    {
        const SoundArchive* soundArchive;
        const SoundDataManager* soundDataManager;
        SoundPlayer* soundPlayer;
        LoadItemInfo loadInfoWsd;
        s32 index;    // .bXwsd ファイル中にいくつめのサウンドかを示す

        Arg() : soundDataManager(NULL), soundArchive(NULL), soundPlayer(NULL), index(0)
        {}
    };

    class DataLoadTask : public Task
    {
    public:
        void Initialize();
        /* override */ void Execute();
        bool TryAllocPlayerHeap();
        void FreePlayerHeap();

        Arg m_Arg;
        Data m_Data;
        PlayerHeap* m_pPlayerHeap;
        bool m_IsLoadSuccess;
        u8 padding[3];
    };

    bool IsInUse();
    void Initialize(const Arg& arg);
    void Finalize();
    bool TryWait();
    bool IsLoadSuccess() const { return m_Task.m_IsLoadSuccess; }

    const void* GetWsdFile() const { return m_Task.m_Data.wsdFile; }
    const void* GetWaveFile() const { return m_Task.m_Data.waveFile; }

private:
    DataLoadTask m_Task;

public:
    ut::LinkListNode m_LinkForLoaderManager; // for WaveSoundLoaderManager
};

class WaveSoundPlayer : public BasicSoundPlayer, public DisposeCallback, public SoundThread::PlayerCallback
{
    /* ------------------------------------------------------------------------
            constant variable
       ------------------------------------------------------------------------ */
public:
    static const int PAUSE_RELEASE_VALUE    = 127;
    static const int MUTE_RELEASE_VALUE     = 127;
    static const int DEFAULT_PRIORITY       = 64;

    enum StartOffsetType
    {
        START_OFFSET_TYPE_SAMPLE,
        START_OFFSET_TYPE_MILLISEC
    };

    struct StartInfo
    {
        s32 index;
        StartOffsetType startOffsetType;
        s32 startOffset;
        s32 delayTime;
    };

    struct WaveSoundCallbackArg
    {
        const void* wsdFile;    // メモリ上のウェーブサウンドファイル
        int wsdIndex;           // bcwsd 内での当該 WSD インデックス
        int noteIndex;          // 現状ではつねにゼロ
        u32 callbackData;       // SoundArchivePlayer::PrepareWaveSoundImpl にて fileId が入る
                                // (が、CTR では無視される)
        const PlayerHeapDataManager* dataMgr;
    };

    class WaveSoundCallback
    {
      public:
        virtual ~WaveSoundCallback() {}

        virtual bool GetWaveSoundData(
            WaveSoundInfo* info,
            WaveSoundNoteInfo* noteInfo,
            WaveInfo* waveData,
            const WaveSoundCallbackArg& arg
        ) const = 0;
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    WaveSoundPlayer();
    virtual ~WaveSoundPlayer();

    /* override */ void Initialize();
    /* override */ void Finalize();

    bool IsPrepared() const
    {
        if (m_ResState >= RES_STATE_ASSIGNED)
        {
            return true;
        }
        return false;
    }

    void SetLoaderManager(WaveSoundLoaderManager* manager)
    {
        m_pLoaderManager = manager;
    }

    struct PrepareArg
    {
        const void* wsdFile;
        const void* waveFile;
        s8 waveType;
        bool useContextInfo;
        u8 padding[2];
        internal::WaveFileReader::OffsetContextInfo contextInfo;

        PrepareArg()
            : wsdFile(NULL)
            , waveFile(NULL)
            , waveType(WAVE_TYPE_NWWAV)
            , useContextInfo(false)
        {}
    };
    void Prepare(const StartInfo& info, const PrepareArg& arg);
    void RequestLoad(const StartInfo& info, const WaveSoundLoader::Arg& arg);

    virtual void Start();
    virtual void Stop();
    virtual void Pause( bool flag );

    //------------------------------------------------------------------
    // プレイヤーパラメータ

    void SetPanRange( f32 panRange );
    void SetChannelPriority( int priority );
    void SetReleasePriorityFix( bool fix );

    f32 GetPanRange() const { return m_PanRange; }
    int GetChannelPriority() const { return m_Priority; }

    virtual void InvalidateData( const void* start, const void* end );

    s32 GetPlaySamplePosition(bool isOriginalSamplePosition) const;
    const void* GetWaveFile() const { return m_pWaveFile; }

public:
    void DebugUpdate() { if ( m_ActiveFlag ) { Update(); } }

private:
    virtual void OnUpdateFrameSoundThread() { Update(); }
    virtual void OnShutdownSoundThread() { Stop(); }
    void PrepareForPlayerHeap(const PrepareArg& arg);
    bool TryAllocLoader();
    void FreeLoader();

    bool m_WavePlayFlag; // チャンネルスタートしたかどうか
    bool m_ReleasePriorityFixFlag;

    u8 m_Priority;
    s8 m_WaveType;
    f32 m_PanRange;

    const void* m_pWsdFile;
    const void* m_pWaveFile;
    int m_WaveSoundIndex;
    StartOffsetType m_StartOffsetType;
    int m_StartOffset;
    s32 m_DelayCount;

    CurveLfoParam m_LfoParam;
    WaveSoundInfo m_WaveSoundInfo;
    Channel* m_pChannel;

    void FinishPlayer();
    void Update();
    bool IsChannelActive() const { return ( m_pChannel != NULL ) && m_pChannel->IsActive(); }
    bool StartChannel();
    void CloseChannel();
    void UpdateChannel();
    static void ChannelCallbackFunc(
        Channel* dropChannel,
        Channel::ChannelCallbackStatus status,
        void* userData
    );


    enum ResState
    {
        RES_STATE_INVALID,          // 未初期化
        RES_STATE_RECV_LOAD_REQ,    // ロードリクエストを受けた
        RES_STATE_APPEND_LOAD_TASK, // ロードタスクを投げた
        RES_STATE_ASSIGNED          // ロードされ、WaveSoundPlayer にセットされた
    };
    u8 m_ResState; // enum ResState
    bool m_IsInitialized;
    bool m_IsRegisterPlayerCallback;
    bool m_UseContextInfo;
    internal::WaveFileReader::OffsetContextInfo m_ContextInfo;

    WaveSoundLoaderManager* m_pLoaderManager;
    WaveSoundLoader* m_pLoader;
    WaveSoundLoader::Arg m_LoaderArg;
}; // class WaveSoundPlayer

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_WAVE_SOUND_PLAYER_H_ */

