﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_WAVE_ARCHIVE_FILE_H_
#define NW_SND_WAVE_ARCHIVE_FILE_H_

#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/snd/snd_Util.h>

namespace nw {
namespace snd {
namespace internal {

//
// MEMO: WaveArchiveFile (.bxwar ＝ 波形アーカイブファイル)
//  * ヘッダ、INFO ブロックは最初に読む
//  * FILE ブロックをまるごと読むこともある
//  * FILE ブロックのうち、必要な波形ファイルのみ読むことも鳴る
//

struct WaveArchiveFile
{
    //
    // ヘッダー
    //
    struct InfoBlock;
    struct FileBlock;

    static const int BLOCK_SIZE = 2;

    // メモ : WaveArchiveFile::FileHeader はヘッダだけ個別にロードするので、
    //        Util::SoundFileHeader を継承できない。
#if defined(NW_PLATFORM_CAFE)
    struct FileHeader : public ut::BinaryFileHeader
#else
    struct FileHeader : public BinaryFileHeader
#endif
    {
        // データ
        Util::ReferenceWithSize toBlocks[ BLOCK_SIZE ];

        // アクセサ (一括ロードの場合に有効)
        const InfoBlock* GetInfoBlock() const;
        const FileBlock* GetFileBlock() const;

        // アクセサ (個別ロードのときに利用される)
        u32 GetInfoBlockSize() const;
        u32 GetFileBlockSize() const;
        u32 GetInfoBlockOffset() const;
        u32 GetFileBlockOffset() const;

      private:
        const Util::ReferenceWithSize* GetReferenceBy( u16 typeId ) const;
    };

    //
    // INFO ブロック
    //
    struct InfoBlockBody
    {
        // データ
        Util::Table<Util::ReferenceWithSize> table;

        // アクセサ
        inline u32 GetWaveFileCount() const { return table.count; }
        u32 GetSize( u32 index ) const
        {
            NW_ASSERT( index < GetWaveFileCount() );
            return table.item[ index ].size;
        }
        u32 GetOffsetFromFileBlockBody( u32 index ) const
        {
            NW_ASSERT( index < GetWaveFileCount() );
            return table.item[ index ].offset;
        }

        static const u32 INVALID_OFFSET = 0xffffffff;

    };
    struct InfoBlock
    {
        ut::BinaryBlockHeader   header;
        InfoBlockBody           body;
    };

    //
    // FILE ブロック
    //
    struct FileBlockBody
    {
    };
    struct FileBlock
    {
        ut::BinaryBlockHeader   header;
        FileBlockBody           body;
    };
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_WAVE_ARCHIVE_FILE_H_ */

