﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_STREAM_LOADER_H_
#define NW_SND_STREAM_LOADER_H_

#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_Config.h>
#include <nw/snd/snd_StreamSoundFileLoader.h>
#include <nw/snd/snd_CachedFsFileStream.h>
#include <nw/snd/snd_Task.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_LoaderManager.h>
#include <nw/snd/snd_SoundArchiveFilesHook.h>

namespace nw {
namespace snd {



//---------------------------------------------------------------------------
//! @briefprivate
//! @brief StreamRegionCallback の返り値です。
//! @see StreamRegionCallback
//! @date 2012/03/06 初版
//---------------------------------------------------------------------------
enum StreamRegionCallbackResult {
    STREAM_REGION_CALLBACK_FINISH,  //!< サウンドを終了することを示します
    STREAM_REGION_CALLBACK_CONTINUE //!< 次のリージョンにジャンプすることを示します
};

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief StreamRegionCallback で入出力するパラメータです。
//! @see StreamRegionCallback
//! @date 2012/03/06 初版
//---------------------------------------------------------------------------
struct StreamRegionCallbackParam {
    //! リージョン番号です。現在再生中にリージョン番号が格納されています。
    //! 次の再生するリージョン番号を書き込みます。
    s32 regionNo;
    const char* regionName;     //!< @briefprivate
    s32 regionNameSize;         //!< @briefprivate
};

//---------------------------------------------------------------------------
//! @briefprivate
//! @brief ストリームサウンドのリージョンジャンプで利用されるコールバック関数の型定義です。
//!
//! @param[in,out] param    前のリージョン番号が格納されており、
//!                         次のリージョン番号を格納するためのパラメータです。
//! @param[in] arg          コールバック関数のユーザー引数です。
//!
//! @return サウンドを終了するか、次のリージョンにジャンプするかを返します。
//!
//! @see StreamRegionCallbackParam
//! @see StreamRegionCallbackResult
//! @see SoundStartable::StartInfo
//---------------------------------------------------------------------------
typedef StreamRegionCallbackResult (*StreamRegionCallback)(
        StreamRegionCallbackParam* param, void* arg );



namespace internal {

struct DriverCommandStreamSoundLoadHeader;
struct DriverCommandStreamSoundLoadData;

struct TrackDataInfo
{
    u8 volume;
    u8 pan;
    u8 span;
    u8 flags;   // StreamSoundFile::SurroundMode が入る
    u8 mainSend;
    u8 fxSend[ AUX_BUS_NUM ];
    u8 lpfFreq;
    u8 biquadType;
    u8 biquadValue;
    u8 channelCount;
    u8 channelIndex[ WAVE_CHANNEL_MAX ];
};

struct TrackDataInfos
{
    TrackDataInfo track[STRM_TRACK_NUM];
};

struct StreamDataInfoDetail
{
    SampleFormat sampleFormat;
    u32 sampleRate;
    bool loopFlag;
    u32 loopStart;
    u32 sampleCount;
    u32 originalLoopStart;
    u32 originalLoopEnd;

    u32 blockSampleCount;
    u32 blockSize;
    u32 lastBlockSize;
    u32 lastBlockSampleCount;

    u32 channelCount;
    u32 trackCount;
    TrackDataInfo trackInfo[STRM_TRACK_NUM];

    void SetStreamSoundInfo(const StreamSoundFile::StreamSoundInfo& info);

    u32 GetLastBlockIndex() const
    {
        return static_cast<int>( ( sampleCount - 1 ) / blockSampleCount );
    }

    u32 GetLoopStartInBlock() const
    {
        return loopStart % blockSampleCount;
    }

    u32 GetLoopStartBlockIndex(u32 loopStartInBlock) const
    {
        return (loopStart - loopStartInBlock) / blockSampleCount;
    }
};

struct LoadDataParam
{
    u32 blockIndex;
    u32 samples;
    u32 sampleBegin;
    u32 sampleOffset;
    bool adpcmContextEnable;
    AdpcmContext adpcmContext[STRM_CHANNEL_NUM];
    u32 loopCount;
    bool lastBlockFlag;

    LoadDataParam()
    {
        Initialize();
    }

    void Initialize()
    {
        blockIndex = 0;
        samples = 0;
        sampleBegin = 0;
        sampleOffset = 0;
        adpcmContextEnable = false;
        loopCount = 0;
        lastBlockFlag = false;
    }
};

struct FileStreamHookParam
{
    FileStreamHookParam()
    : pSoundArchiveFilesHook(NULL)
    , itemLabel(NULL)
    {}

    bool IsHookEnabled() const { return pSoundArchiveFilesHook != NULL; }

    internal::SoundArchiveFilesHook* pSoundArchiveFilesHook;
    const char* itemLabel;
};


namespace driver {

class StreamDataDecoder
{
public:
    enum DecodeType
    {
        DECODE_TYPE_NORMAL,
        DECODE_TYPE_LOOP,
        DECODE_TYPE_IDLING,
        DECODE_TYPE_NUM
    };

    struct DataInfo
    {
        u32 channelCount;
        u32 sampleRate;
        u32 blockSampleCount;
        u32 blockSize;
    };

public:
    virtual ~StreamDataDecoder() {}

    virtual void* AllocWorkBuffer() = 0;
    virtual void  FreeWorkBuffer(void* buffer) = 0;
    virtual bool  ReadDataInfo( internal::FileStream* fileStream, DataInfo* info, void* workBuffer ) = 0;
    virtual bool  Skip( internal::FileStream* fileStream ) = 0;
    virtual bool  Decode( internal::FileStream* fileStream, int channelCount, DecodeType decodeType, s16* decodedDataArray[], void* workBuffer ) = 0;
};

class StreamSoundPlayer;
class StreamSoundLoader;
typedef LoaderManager<StreamSoundLoader> StreamSoundLoaderManager;

class StreamSoundLoader
{
public:
    // ストリームバッファ 1 ブロックの基本サイズ [byte]
    // (通常はこのサイズ分をロードして再生する)
    static const u32 DATA_BLOCK_SIZE_BASE = 8 * 1024;

    // 上記のサンプル数を、一番圧縮率の低い PCM16 でバイト数に換算した値
    static const u32 DATA_BLOCK_SIZE_MARGIN = DATA_BLOCK_SIZE_MARGIN_SAMPLES * 2;

    // 1 ブロックのサイズ [byte]
    static const u32 DATA_BLOCK_SIZE_MAX = DATA_BLOCK_SIZE_BASE + DATA_BLOCK_SIZE_MARGIN;

    static const int FILE_STREAM_BUFFER_SIZE = 512;

    static const u32 LOAD_BUFFER_CHANNEL_NUM = 2; // 同時にロードを行うチャンネルの数
    static const u32 LOAD_BUFFER_SIZE = DATA_BLOCK_SIZE_MAX * LOAD_BUFFER_CHANNEL_NUM;

    StreamSoundLoader();
    ~StreamSoundLoader();

    void Initialize();
    void Finalize();

    static void SetStreamDataDecoder( StreamDataDecoder* decoder )
    {
        s_StreamDataDecoder = decoder;
    }

    void Update();
    void ForceFinish();

    bool IsBusy() const;
    bool IsInUse();

    void RequestLoadHeader();
    void RequestLoadData( void* bufferAddress[], u32 bufferBlockIndex, u32 startOffsetSamples, u32 prefetchOffsetSamples, int priority );
    void RequestClose();
    void CancelRequest();

public:
    StreamSoundFileLoader m_FileLoader;
    StreamSoundPlayer* m_PlayerHandle;
    internal::FileStream* m_pFileStream;
    StreamDataInfoDetail* m_DataInfo;
    StreamFileType m_FileType;
    FileStreamHookParam m_FileStreamHookParam;

    u32 m_ChannelCount;
    u16 m_RequestIndex;
    bool m_LoopFlag;
    bool m_IsStreamOpenFailureHalt;
    u32 m_LoopStart;
    u32 m_LoopEnd;

    StreamRegionCallback m_StreamRegionCallbackFunc;
    void* m_StreamRegionCallbackArg;
    char m_FilePath[FILE_PATH_MAX];
    void* m_pCacheBuffer;
    size_t m_CacheSize;
#if defined(NW_PLATFORM_CAFE)
    FSClient* m_pFsClient;
    FsFileStream::FsCommandBlockPool* m_pFsCommandBufferPool;
    FSPriority m_FsPriority;
#endif

private:
    class StreamHeaderLoadTask : public Task
    {
    public:
        /* override */ void Execute()
        {
            NW_ASSERT_NOT_NULL(m_pLoader);
            bool result = m_pLoader->Open();

            if (!result)
            {
                if (m_pLoader->m_IsStreamOpenFailureHalt)
                {
                    nw::ut::TPanic(NW_FILE_NAME, __LINE__, "StreamSoundLoader::Open is Failed\n");
                }
                else
                {
                    m_pLoader->ForceFinish();
                    return;
                }
            }

            m_pLoader->LoadHeader();
        }
        StreamSoundLoader* m_pLoader;
    };

    class StreamDataLoadTask : public Task
    {
    public:
        /* override */ void Execute()
        {
            NW_ASSERT_NOT_NULL(m_pLoader);
            m_pLoader->LoadData( m_BufferAddress, m_BufferBlockIndex, m_StartOffsetSamples, m_PrefetchOffsetSamples );
        }

        void* m_BufferAddress[ STRM_CHANNEL_NUM ];
        s32 m_BufferBlockIndex;
        u32 m_StartOffsetSamples;
        u32 m_PrefetchOffsetSamples;

        StreamSoundLoader* m_pLoader;
        ut::LinkListNode m_Link;
    };

    class StreamCloseTask : public Task
    {
    public:
        /* override */ void Execute()
        {
            NW_ASSERT_NOT_NULL(m_pLoader);
            m_pLoader->Close();
        }
        StreamSoundLoader* m_pLoader;
    };

    typedef ut::LinkList<StreamDataLoadTask, offsetof(StreamDataLoadTask,m_Link)> StreamDataLoadTaskList;

    struct CurrentRegion
    {
        CurrentRegion()
        : current(0)
        , begin(0)
        , end(0)
        {
        }

        u32 current;
        u32 begin;
        u32 end;

        bool IsIn(u32 value) { return current + value < end; }
        bool IsInWithBorder(u32 value) { return current + value <= end; }
        bool IsEnd() const { return current == end; }
        u32 Rest() const { return end - current; }
    };

    struct BlockInfo
    {
        BlockInfo()
        : size(0)
        , samples(0)
        , startOffsetSamples(0)
        , startOffsetSamplesAlign(0)
        , startOffsetByte(0)
        , copyByte(0)
        {
        }

        u32 size;
        u32 samples;
        u32 startOffsetSamples;
        u32 startOffsetSamplesAlign;
        u32 startOffsetByte;
        u32 copyByte;

        u32 GetStartOffsetInFrame() const { return startOffsetSamples - startOffsetSamplesAlign; }
    };

private:
    void WaitFinalize();

    bool Open();
    void Close();

    void LoadHeader();
    bool LoadHeader1( DriverCommandStreamSoundLoadHeader* command );
    bool LoadHeader2( DriverCommandStreamSoundLoadHeader* command );

    void LoadData( void* bufferAddress[], u32 bufferBlockIndex, u32 startOffsetSamples, u32 prefetchOffsetSamples );
    bool LoadData1( DriverCommandStreamSoundLoadData* command, void* bufferAddress[], u32 bufferBlockIndex, u32 startOffsetSamples, u32 prefetchOffsetSamples );
    bool LoadData2( DriverCommandStreamSoundLoadData* command, void* bufferAddress[], u32 bufferBlockIndex, u32 startOffsetSamples, u32 prefetchOffsetSamples );

    void SetStreamSoundInfo2(const StreamDataDecoder::DataInfo& info);

    bool ApplyStartOffset(u32 startOffsetSamples, u32* loopCount);
    bool MoveNextRegion(u32* loopCount);

    void ChangeRegionInfo(u32 regionNo);
    bool SetupRegionInfo();
    bool UpdateRegion();

    bool ReadTrackInfoFromStreamSoundFile(StreamSoundFileReader& reader);

    bool IsLoopStartFilePos(u32 loadingDataBlockIndex);
    void UpdateLoadingDataBlockIndex();
    void UpdateLoadingDataBlockIndex2();

    u32 GetLoadChannelCount(u32 loadStartChannel);
    bool LoadStreamBuffer(u8* buffer, const BlockInfo& blockInfo, u32 loadChannelCount);
    void CalculateBlockInfo(BlockInfo& blockInfo);
    bool LoadOneBlockData(void* bufferAddress[], const BlockInfo& blockInfo, u32 destAddressOffset, bool firstBlock, bool updateAdpcmContext);

    bool LoadAdpcmContextForStartOffset();
    void UpdateAdpcmInfoForStartOffset(const void* blockBegin, u32 channel, const BlockInfo& blockInfo);
    bool SetAdpcmInfo(
        StreamSoundFileReader& reader,
        u32 channelCount,
        AdpcmParam* adpcmParam[]);

    bool DecodeStreamData(StreamDataDecoder::DecodeType decodeType, void* bufferAddress[]);

private:
    u32 m_LoadingDataBlockIndex;    // 全データのうち、現在ロード中のブロックの通し番号
    u32 m_LastBlockIndex;
    u32 m_LoopStartBlockIndex;
    u32 m_DataStartFilePos;
    u32 m_LoopStartFilePos;
    u32 m_LoopStartBlockSampleOffset;

    bool m_LoopJumpFlag;

    bool m_LoadFinishFlag;

    bool m_IsRegionInfoEnabled;

    u8 padding;

    u32 m_CurrentRegionNo;
    CurrentRegion m_Region;

    StreamHeaderLoadTask m_StreamHeaderLoadTask;
    StreamCloseTask m_StreamCloseTask;
    StreamDataLoadTaskList m_StreamDataLoadTaskList;
    InstancePool<StreamDataLoadTask> m_StreamDataLoadTaskPool;
    u8 m_StreamDataLoadTaskArea[STRM_DATA_LOAD_TASK_MAX*sizeof(StreamDataLoadTask)];
    void* m_pDecodeWorkBuffer;

    SampleFormat m_SampleFormat;
    struct AdpcmInfo
    {
        AdpcmParam param;
        AdpcmContext beginContext;
        AdpcmContext loopContext;
    };
    AdpcmInfo m_AdpcmInfo[STRM_CHANNEL_NUM];
    u32 m_AdpcmContextForStartOffsetFrame;
    AdpcmContext m_AdpcmContextForStartOffset[STRM_CHANNEL_NUM];

    u32 m_FileStreamBuffer[ FILE_STREAM_BUFFER_SIZE/sizeof(u32) ];

    static u8 s_LoadBuffer[ LOAD_BUFFER_SIZE ];
    static StreamDataDecoder* s_StreamDataDecoder;

public:
    ut::LinkListNode m_LinkForLoaderManager;  // for StreamSoundLoaderManager
};

} // nw::snd::internal::driver
} // nw::snd::internal
} // nw::snd
} // nw

#endif /* NW_SND_STREAM_LOADER_H_ */

