﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundStartable.h
 *
 * @file snd_SoundStartable.h
 */

#ifndef NW_SND_SOUND_STARTABLE_H_
#define NW_SND_SOUND_STARTABLE_H_

#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_StreamSoundPlayer.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_WaveFileReader.h>

namespace nw {
namespace snd {

class SoundHandle;


// 再生可能なことを表すインターフェイスです。
// 純粋仮想関数 detail_SetupSoundImpl() 及び detail_ConvertLabelStringToItemId() を
// 継承したクラスで実装することにより、そのクラスで
// StartSound(), HoldSound(), PrepareSound() 関数が使用できるようになります。

//---------------------------------------------------------------------------
//! @brief        サウンドの再生が可能なことを表す抽象クラスです。
//!
//! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundStartable
{
public:
    //---------------------------------------------------------------------------
    //! @brief    サウンド再生処理結果を表すクラスです。
    //!
    //!           START_ERR_NOT_ENOUGH_INSTANCE は、
    //!           再生しようとしたサウンドに必要な数のインスタンスが
    //!           作られていないことを表します。
    //!           サウンドアーカイブを使って再生している場合は、
    //!           サウンドアーカイブ中に設定されている
    //!           インスタンス数を増やす必要があります。
    //!
    //! @see SoundStartable::StartSound.
    //! @see SoundStartable::HoldSound
    //! @see SoundStartable::PrepareSound
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    class StartResult
    {
    public:
        //---------------------------------------------------------------------------
        //! @brief    サウンド再生処理結果を表すコードです。
        //!
        //! @date 2012/03/09 エラーコードの説明を追加
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        enum ResultCode
        {
            //! 再生は成功しました。
            START_SUCCESS = 0,

            //! プレイヤーで同時再生可能なサウンド数を超え、かつ、
            //! 他のサウンドよりプレイヤー優先度がが低かったため、再生に失敗しました。
            START_ERR_LOW_PRIORITY,

            //! 指定したラベル文字列に対応するサウンドが見つからないため、再生に失敗しました。
            START_ERR_INVALID_LABEL_STRING,

            //! 指定したサウンド ID に対応するサウンドが見つからないため、再生に失敗しました。
            START_ERR_INVALID_SOUNDID,

            //! (現在は使われていません)
            START_ERR_NOT_DATA_LOADED,

            //! シーケンスデータが未ロードでプレイヤーヒープも存在しないため、再生に失敗しました。
            START_ERR_NOT_SEQ_LOADED,

            //! バンクデータが未ロードでプレイヤーヒープも存在しないため、再生に失敗しました。
            START_ERR_NOT_BANK_LOADED,

            //! ウェーブサウンドデータが未ロードでプレイヤーヒープも存在しないため、再生に失敗しました。
            START_ERR_NOT_WSD_LOADED,

            //! 波形アーカイブデータが未ロードでプレイヤーヒープも存在しないため、再生に失敗しました。
            START_ERR_NOT_WARC_LOADED,

            //! プレイヤーヒープの準備が整っていないため、再生に失敗しました。
            START_ERR_NOT_ENOUGH_PLAYER_HEAP,

            //! (現在は使われていません)
            START_ERR_CANNOT_OPEN_FILE,

            //! @ref SoundArchivePlayer::Initialize が呼ばれていない、あるいは、SoundArchivePlayer
            //! に設定した SoundArchive に対する初期化 (@ref FsSoundArchive::Open,
            //! @ref MemorySoundArchive::Initialize) が呼ばれていないため、
            //! 再生に失敗しました。
            START_ERR_NOT_AVAILABLE,

            //! (現在は使われていません)
            START_ERR_CANNOT_ALLOCATE_TRACK,

            //! 優先度が低いため、サウンドインスタンスの確保に失敗し、再生に失敗しました。
            //! サウンドインスタンスの数は、SoundMaker の [プロジェクトの設定] -
            //! [サウンドアーカイブ] タブの「◯◯サウンドの数」で設定できます。
            //! また、優先度を高くするには、プレイヤー優先度を高くしてください。
            START_ERR_NOT_ENOUGH_INSTANCE,

            //! 不正なパラメータが指定されたため、再生に失敗しました。
            //! 現在は、@ref StartInfo の actorPlayerId が不正な値 (＝0～3以外の値) の場合に、
            //! 本エラーが発生します。
            START_ERR_INVALID_PARAMETER,

            //! @ref StartInfo の seqSoundInfo で指定されている startLocationLabel が不正なため、
            //! 再生に失敗しました。
            START_ERR_INVALID_SEQ_START_LOCATION_LABEL,

            //! @ref SoundActor あるいは @ref Sound3DActor が初期化前のため、
            //! 再生に失敗しました。
            START_ERR_ACTOR_NOT_INITIALIZED,

            //! (現在は使われていません)
            START_ERR_INVALID_WARC_ID,

            //! ロード済みのバンクデータが不正なため、波形アーカイブデータが見つからず、再生に失敗しました。
            START_ERR_INVALID_BANK_DATA,

            START_ERR_INVALID_STRM_FILE_ID,
            START_ERR_INVALID_STRM_FILE_PATH,
            START_ERR_INVALID_RENDERER_TYPE,

#if defined(NW_PLATFORM_CAFE)
            // ストリームバッファとプリフェッチデータが、512MByte 境界を跨ぐように配置されています。
            // これらのデータは、メモリ上で同じ 512MByte 境界内に配置されていなければ、再生ができません。
            // 詳細は SDK のドキュメントの「ストリーミングの概要」を参照してください。
            START_ERR_INVALID_VOICE_OFFSET_BOUNDARY,
#endif

            //! サウンドライブラリ内部では使われていません。
            //! "START_ERR_USER" ～ "START_ERR_UNKNOWN - 1" は、
            //! ユーザーが自由に利用することができます。
            START_ERR_USER = 128,

            //! なんらかの理由で、再生に失敗しました。
            START_ERR_UNKNOWN = 255
        };

        //! @briefprivate
        StartResult() : m_Code( START_ERR_UNKNOWN ) {}

        //! @briefprivate
        //! @param code :private
        explicit StartResult( ResultCode code ) : m_Code( code ) {}

        //---------------------------------------------------------------------------
        //! @brief    再生成功したかどうかを取得します。
        //!
        //! @return   再生成功したら true, 失敗したら false を返します。
        //!
        //! @see SoundStartable::StartSound
        //! @see SoundStartable::HoldSound
        //! @see SoundStartable::PrepareSound
        //!
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        bool IsSuccess() const { return m_Code == START_SUCCESS; }

        //---------------------------------------------------------------------------
        //! @brief    再生処理結果コードを取得します。
        //!
        //! @return   再生処理結果コードを返します。
        //!
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        ResultCode GetCode() const { return m_Code; }

    private:
        ResultCode m_Code;
    };

    // static const char* detail_ConvertStartResultToString( StartResult startResult );

    //---------------------------------------------------------------------------
    //! @brief  サウンド再生時に渡される、詳細な再生パラメータです。
    //!
    //!         @ref SoundStartable::StartSound 等の関数に渡して使用します。
    //!         この構造体を使用しなくてもサウンドは再生出来ます。
    //!
    //!         startOffsetType, startOffset を設定すると、
    //!         サウンドを途中から再生することが可能です。
    //!         この 2 つのパラメータは同時に設定する必要があります。
    //!         設定した値を有効にするには、
    //!         enableFlag の ENABLE_START_OFFSET ビットを設定してください。
    //!
    //!         オフセット値が同じでも、
    //!         データの種類によって途中再生のために必要な処理時間が異なります。
    //!         大きな処理時間がかかる場合があることに注意してください。
    //!
    //!         【シーケンスサウンド】 @n
    //!         シーケンスを途中まで空回しするため、
    //!         開始オフセットの大きさに応じた処理時間がかかります。
    //!
    //!         【ストリームサウンド】 @n
    //!         データが PCM の場合は、開始オフセットの大きさに関わらず、
    //!         途中再生のための処理時間はほとんどかかりません。
    //!         データが ADPCM の場合は、1 チャンネルあたり最大 14336 サンプルのデータを
    //!         CPU でデコードするための処理がかかります。
    //!         ADPCM では、再生開始位置が指定された位置の直前にある
    //!         14 サンプル境界に変更されます。
    //!
    //!         【ウェーブサウンド】 @n
    //!         データが PCM の場合は、開始オフセットの大きさに関わらず、
    //!         途中再生のための処理時間はほとんどかかりません。
    //!         データが ADPCM の場合は、
    //!         データ先頭から再生開始位置までの全てのサンプルを
    //!         CPU でデコードする必要があるため、
    //!         開始オフセットの大きさに応じた処理時間がかかります。
    //!         ADPCM では、再生開始位置が指定された位置の直前にある
    //!         14 サンプル境界に変更されます。
    //!
    //!         サウンドはサウンドアーカイブ中で設定された
    //!         ID のプレイヤーを使用して再生されますが、
    //!         これとは異なるプレイヤーを指定して再生したい場合、
    //!         playerId を設定します。設定した値を有効にするには、
    //!         enableFlag の ENABLE_PLAYER_ID ビットを設定してください。
    //!
    //!         サウンドはサウンドアーカイブ中で設定された
    //!         プレイヤープライオリティを使用して再生されますが、
    //!         これとは異なるプライオリティ値を指定して再生したい場合、
    //!         playerPriority を設定します。設定した値を有効にするには、
    //!         enableFlag の ENABLE_PLAYER_PRIORITY ビットを設定してください。
    //!
    //!         actorPlayerId は、@ref SoundActor
    //!         クラスで再生する際に使用するアクタープレイヤー番号を設定します。
    //!         値の範囲は 0 ～ 3 です。設定した値を有効にするには、
    //!         enableFlag の ENABLE_ACTOR_PLAYER_ID ビットを設定してください。
    //!
    //!         seqSoundInfo は、シーケンスサウンドに関するパラメータです。
    //!         シーケンスサウンドを再生するときにのみ有効です。
    //!         サウンドアーカイブ中で設定されている、
    //!         シーケンスサウンドに関する情報を上書きすることができます。
    //!         詳しくは SeqSoundInfo 構造体を参照してください。
    //!         設定した値を有効にするには、 enableFlag の ENABLE_SEQ_SOUND_INFO
    //!         ビットを設定してください。
    //!
    //          // streamSoundInfo は、ストリームサウンドに関するパラメータです。
    //          // ストリームサウンドを再生するときのみ有効です。
    //          // ストリームサウンドのリージョンジャンプ機能を利用する場合に、
    //          // 本構造体を使用する必要があります。
    //          // 詳しくは StreamSoundInfo 構造体を参照してください。
    //          // 設定した値を有効にするには、 enableFlag の ENABLE_STRM_SOUND_INFO
    //          // ビットを設定してください。
    //!
    //! @see  SoundStartable::StartSound
    //!
    //  // @date 2012/03/06 メンバ変数に streamSoundInfo を追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    struct StartInfo
    {
        //---------------------------------------------------------------------------
        //! @brief    StartInfo 構造体のパラメータを有効にするためのビットフラグの定義です。
        //!
        //!           それぞれの値の論理和を enableFlag に設定します。
        //!
        //! @see      SoundStartable::StartInfo
        //!
        //! @date 2012/12/04 関連する StartInfo 構造体のメンバ変数を明記
        //  // @date 2012/03/06 ENABLE_STRM_SOUND_INFO を追加
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        enum EnableFlagBit
        {
            /*!
              @brief    startOffsetType, startOffset に設定された値を有効にします。
                        2 つのパラメータは同時に設定する必要があります。
             */
            ENABLE_START_OFFSET      = (1<<0),

            //! playerId に設定された値を有効にします。
            ENABLE_PLAYER_ID         = (1<<1),

            //! playerPriority に設定された値を有効にします。
            ENABLE_PLAYER_PRIORITY   = (1<<2),

            //! actorPlayerId に設定された値を有効にします。
            ENABLE_ACTOR_PLAYER_ID   = (1<<3),

            //! seqSoundInfo に設定された値を有効にします。
            ENABLE_SEQ_SOUND_INFO    = (1<<4),

            //! streamSoundInfo に設定された値を有効にします。
            ENABLE_STRM_SOUND_INFO   = (1<<5),

            //! waveSoundInfo に設定された値を有効にします。
            ENABLE_WAVE_SOUND_INFO   = (1<<6),

            //! @briefprivate
            ENABLE_VOICE_RENDERER_TYPE = (1<<7),

            //! @briefprivate
            //! fadeFrame に設定された値を有効にします。
            ENABLE_FADE_FRAME          = (1<<8),

            //! @briefprivate
            //! soundStopCallback に設定された値を有効にします。
            ENABLE_SOUND_STOP_CALLBACK = (1<<9),

            //! streamSoundMetaInfo に設定された値を有効にします。
            ENABLE_STRM_SOUND_META_INFO = (1<<10),

            //! @briefprivate
            //! delayTimeに設定された値を有効にします。
            ENABLE_DELAY_TIME  = (1<<11)
        };

        //---------------------------------------------------------------------------
        //! @brief    サウンドの開始オフセット値の単位の定義です。
        //!
        //! @see      SoundStartable::StartInfo
        //!
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        enum StartOffsetType
        {
            START_OFFSET_TYPE_MILLISEC, //!< オフセット値の単位はミリ秒です。

            /*!
                @brief  オフセット値の単位は Tick です。
                        この指定はシーケンスサウンドに対してのみ有効です。
                        それ以外では、オフセット値の指定が無効になります。
             */
            START_OFFSET_TYPE_TICK,

            /*!
                @brief  オフセット値の単位は波形のサンプル数です。
                        この指定はストリームサウンドとウェーブサウンドに対して有効です。
                        それ以外では、オフセット値の指定が無効になります。
             */
            START_OFFSET_TYPE_SAMPLE
        };

        //---------------------------------------------------------------------------
        //! @brief    シーケンスサウンドに関するパラメータ構造体です。
        //!
        //!           シーケンスサウンド再生時に渡されます。
        //!
        //!           seqDataAddress にシーケンスデータのアドレスを指定すると、
        //!           サウンドアーカイブ中で設定されているシーケンスデータの代わりに、
        //!           メモリ上に置かれたシーケンスデータをアドレスで指定して再生することができます。
        //!           NULL を指定すると、
        //!           元々サウンドアーカイブで設定されているシーケンスデータを再生します。
        //!
        //!           startLocationLabel には、シーケンスデータ内の再生開始位置を指定します。
        //!           サウンドアーカイブ中で設定されているラベル文字列の代わりに、
        //!           ここで指定した文字列を使用するようになります。
        //!           NULL を指定すると、
        //!           元々サウンドアーカイブで設定されている再生開始位置を使用して再生します。
        //!
        //!           bankIds には、シーケンスデータと関連付けるバンク ID を指定します。
        //!           サウンドアーカイブ中で設定されているバンク ID の代わりに、
        //!           ここで指定した ID を仕様するようになります。
        //!           NULL を指定すると、
        //!           元々サウンドアーカイブで設定されているバンク ID を仕様して再生します。
        //!
        //! @see      SoundStartable::StartInfo
        //!
        //! @date 2012/03/06 誤字修正 (渡されるます → 渡されます)
        //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
        //---------------------------------------------------------------------------
        struct SeqSoundInfo
        {
            //! シーケンスデータのアドレスです。
            const void* seqDataAddress;

            //! シーケンスデータ内の再生開始位置を示すラベル文字列です。
            const char* startLocationLabel;

            //! シーケンスデータと関連付けるバンク ID です。
            SoundArchive::ItemId bankIds[ SoundArchive::SEQ_BANK_MAX ];

            //! コンストラクタです。
            SeqSoundInfo() : seqDataAddress( NULL ), startLocationLabel( NULL )
            {
                for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
                {
                    bankIds[ i ] = SoundArchive::INVALID_ID;
                }
            }
        };

        //---------------------------------------------------------------------------
        //! @brief    ストリームサウンドに関するパラメータ構造体です。
        //!
        //!           ストリームサウンド再生時に渡されます。
        //!
        //!           externalPath にストリームサウンドバイナリへのパスを指定すると、
        //!           サウンドアーカイブ中で設定されているストリームサウンドが参照する
        //!           ストリームサウンドバイナリの代わりに、指定したものを使用して再生することができます。
        //!           NULL を指定すると、
        //!           元々サウンドアーカイブで設定されているストリームサウンドバイナリが使用されます。
        //!
        //  @see StreamRegionCallback
        //  @date 2013/11/15 prefetchData を追加
        //! @date 2012/12/04 externalPath を追加
        //! @date 2012/03/06 初版
        //---------------------------------------------------------------------------
        struct StreamSoundInfo
        {
            //! サウンドアーカイブ管理外の bfstm ファイルのパスです。フルパスで指定する必要があります。
            const char* externalPath;

            //! @briefprivate
            //! リージョンジャンプ先を指定するコールバック関数です。
            StreamRegionCallback regionCallback;

            //! @briefprivate
            //! リージョンジャンプ先を指定するコールバック関数に渡すユーザー引数です。
            void* regionCallbackArg;

            //! @briefprivate
            //! プリフェッチデータです。指定していない場合は使用されません。
            const void* prefetchData;

            //! @briefprivate
            //! 指定している場合、メモリ配置の問題でプリフェッチデータが再生できない際に強制的にストリーム再生します。
            bool forcePlayPrefetchFlag;

            //! @brief コンストラクタです。
            //! @date 2012/03/06 初版
            StreamSoundInfo()
            : externalPath(NULL), regionCallback(NULL), regionCallbackArg(NULL), prefetchData(NULL), forcePlayPrefetchFlag(false)
            {}
        };

        //---------------------------------------------------------------------------
        //! @brief    ウェーブサウンドに関するパラメータ構造体です。
        //!
        //!           ウェーブサウンド再生時に渡されます。
        //!
        //!           waveAddress に波形データのアドレスを指定すると、
        //!           サウンドアーカイブ中で設定されている波形データの代わりに、
        //!           メモリ上に置かれた波形データを使用して再生することができます。
        //!           NULL を指定すると、
        //!           元々サウンドアーカイブで設定されている波形データが使用されます。
        //!
        //! @date 2012/06/14 初版
        //---------------------------------------------------------------------------
        struct WaveSoundInfo
        {
            //! 使用する波形データのアドレスです。
            const void* waveAddress;

            //! @ref nw::snd::WaveType を指定します。
            s8 waveType;

            //! @briefprivate
            bool useContextInfo;

            //! @briefprivate
            u8 padding[2];

            //! @briefprivate
            internal::WaveFileReader::OffsetContextInfo contextInfo;

            //! @brief コンストラクタです。
            //! @date 2012/06/14 初版
            WaveSoundInfo()
            : waveAddress(NULL)
            , waveType(WAVE_TYPE_NWWAV)
            , useContextInfo(false)
            {
            }
        };

        /*!
            @brief  構造体に設定された値を有効にするためのビットフラグです。
                    デフォルト値では全て無効になっています。
         */
        u32 enableFlag;

        //! サウンドの開始オフセット値の単位です。
        StartOffsetType startOffsetType;

        //! サウンドの開始オフセット値です。
        int startOffset;

        //! サウンドの再生に使用するプレイヤーの ID です。
        SoundArchive::ItemId playerId;

        //! サウンドに設定するプレイヤープライオリティです。
        int playerPriority;

        //! SoundActorで再生する場合に使用するアクタープレイヤーの番号です。
        int actorPlayerId;

        //! シーケンスサウンドに関するパラメータです。
        SeqSoundInfo seqSoundInfo;

        //! ストリームサウンドに関するパラメータです。
        StreamSoundInfo streamSoundInfo;

        //! ストリームサウンドのメタ情報に関するパラメータです。
        SoundArchive::StreamSoundInfo streamSoundMetaInfo;

        //! ウェーブサウンドに関するパラメータです。
        WaveSoundInfo waveSoundInfo;

        //! @briefprivate
        //! ボイスレンダリングモードです。
        u8 voiceRendererType;

        //! @briefprivate
        //! HoldSound を指定したフレーム数をかけてフェードアウトさせます。
        //! StartSound では使用されません。
        int fadeFrame;

        //! @briefprivate
        //! サウンドの停止時に呼ばれるコールバック関数です。
        SoundStopCallback soundStopCallback;

        //! @briefprivate
        //! サウンドを指定した時間だけ待ってから再生させます。
        //! 単位はミリ秒（ms)で、3 の倍数のみ設定することができます。
        //! 3 の倍数ではない値を指定した場合、アサートで停止しますのでご注意ください。
        s32 delayTime;

        //! コンストラクタです
        StartInfo()
        : enableFlag( 0 )
        , voiceRendererType(VOICE_RENDERER_SDK)
        , soundStopCallback(NULL)
        , delayTime( 0 ) {}
    };

    /* ------------------------------------------------------------------------
            member definition
       ------------------------------------------------------------------------ */
    // デストラクタ
    //! @briefprivate
    virtual ~SoundStartable() {};

    //----------------------------------------
    //! @name 再生
    //@{
    // 再生関数
    //---------------------------------------------------------------------------
    //! @brief  指定した ID のサウンドを再生します。
    //!
    //!         第一引数には、サウンドハンドルを指定します。
    //!         再生に成功したサウンドは、このサウンドハンドルと関連付けられます。
    //!
    //!         第二引数には、再生するサウンドの識別子としてサウンド ID を指定します。
    //!
    //!         pStartInfo は、再生時に設定できる詳細なパラメータです。
    //!         このパラメータは、設定せずに再生を開始することが可能です。
    //!
    //!         この関数を呼び出すことは、@ref PrepareSound を呼び出した後、
    //!         すぐにハンドルクラスの @ref SoundHandle::StartPrepared
    //!         を呼び出すことと同じです。
    //!
    //! @param[in] pHandle        再生されるサウンドと関連付けられるハンドルです。
    //! @param[in] soundId        再生するサウンドの ID です。
    //! @param[in] pStartInfo     詳細な再生パラメータです。
    //!
    //! @return   再生処理結果を @ref StartResult 型で返します。
    //!
    //! @see SoundHandle クラス
    //! @see StartResult
    //! @see PrepareSound
    //! @see StartInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    StartResult StartSound(
            SoundHandle* pHandle,
            SoundArchive::ItemId soundId,
            const StartInfo* pStartInfo = NULL );

    //---------------------------------------------------------------------------
    //! @brief  指定したラベル文字列のサウンドを再生します。
    //!
    //!         第一引数には、サウンドハンドルを指定します。
    //!         再生に成功したサウンドは、このサウンドハンドルと関連付けられます。
    //!
    //!         第二引数には、再生するサウンドの識別子として、
    //!         サウンドのラベル文字列を指定します。
    //!
    //!         pStartInfo は、再生時に設定できる詳細なパラメータです。
    //!         このパラメータは、設定せずに再生を開始することが可能です。
    //!
    //!         この関数を呼び出すことは、@ref PrepareSound を呼び出した後、
    //!         すぐにハンドルクラスの @ref SoundHandle::StartPrepared
    //!         を呼び出すことと同じです。
    //!
    //! @param[in] pHandle        再生されるサウンドと関連付けられるハンドルです。
    //! @param[in] pSoundName     再生するサウンドのラベル文字列です。
    //! @param[in] pStartInfo     詳細な再生パラメータです。
    //!
    //! @return   再生処理結果を @ref StartResult 型で返します。
    //!
    //! @see SoundHandle クラス
    //! @see StartResult
    //! @see PrepareSound
    //! @see StartInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    StartResult StartSound(
            SoundHandle* pHandle,
            const char* pSoundName,
            const StartInfo* pStartInfo = NULL );

    //---------------------------------------------------------------------------
    //! @brief  指定した ID のサウンドを再生または継続します。
    //!
    //!         同じ ID で毎フレームこの関数を呼び続けている間、
    //!         サウンドを再生することができます。
    //!         呼び出しが途切れた時はサウンドが停止します。
    //!
    //!         第一引数には、サウンドハンドルを指定します。
    //!
    //!         第二引数には、再生するサウンドの識別子としてサウンド ID を指定します。
    //!
    //!         pHoldInfo は、再生時に設定できる詳細なパラメータです。
    //!         このパラメータは、設定せずに再生を開始することが可能です。
    //!
    //! @param[in] pHandle        再生されるサウンドと関連付けられるハンドルです。
    //! @param[in] soundId        再生するサウンドの ID です。
    //! @param[in] pHoldInfo      詳細な再生パラメータです。
    //!
    //! @return   再生処理結果を @ref StartResult 型で返します。
    //!
    //! @see SoundHandle クラス
    //! @see StartResult
    //! @see StartInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    StartResult HoldSound(
            SoundHandle* pHandle,
            SoundArchive::ItemId soundId,
            const StartInfo* pHoldInfo = NULL );

    //---------------------------------------------------------------------------
    //! @brief  指定したラベル文字列のサウンドを再生または継続します。
    //!
    //!         同じ ID で毎フレームこの関数を呼び続けている間、
    //!         サウンドを再生することができます。
    //!         呼び出しが途切れた時はサウンドが停止します。
    //!
    //!         第一引数には、サウンドハンドルを指定します。
    //!
    //!         第二引数には、再生するサウンドの識別子として、
    //!         サウンドのラベル文字列を指定します。
    //!
    //!         pHoldInfo は、再生時に設定できる詳細なパラメータです。
    //!         このパラメータは、設定せずに再生を開始することが可能です。
    //!
    //! @param[in] pHandle        再生されるサウンドと関連付けられるハンドルです。
    //! @param[in] pSoundName     再生するサウンドのラベル文字列です。
    //! @param[in] pHoldInfo      詳細な再生パラメータです。
    //!
    //! @return   再生処理結果を @ref StartResult 型で返します。
    //!
    //! @see SoundHandle クラス
    //! @see StartResult
    //! @see StartInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    StartResult HoldSound(
            SoundHandle* pHandle,
            const char* pSoundName,
            const StartInfo* pHoldInfo = NULL );

    //---------------------------------------------------------------------------
    //! @brief  指定した ID のサウンドを再生する準備をします。
    //!
    //!         準備が完了したサウンドは、ハンドルクラスの
    //!         @ref SoundHandle::StartPrepared
    //!         を呼び出して再生を開始することができます。
    //!
    //!         第一引数には、サウンドハンドルを指定します。
    //!         再生に成功したサウンドは、このサウンドハンドルと関連付けられます。
    //!
    //!         第二引数には、再生するサウンドの識別子としてサウンド ID を指定します。
    //!
    //!         pStartInfo は、再生時に設定できる詳細なパラメータです。
    //!         このパラメータは、設定せずに再生を開始することが可能です。
    //!
    //! @param[in] pHandle        再生されるサウンドと関連付けられるハンドルです。
    //! @param[in] soundId        再生するサウンドの ID です。
    //! @param[in] pStartInfo     詳細な再生パラメータです。
    //!
    //! @return   再生準備処理結果を @ref StartResult 型で返します。
    //!
    //! @see SoundHandle クラス
    //! @see StartResult
    //! @see StartSound
    //! @see StartInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    StartResult PrepareSound(
            SoundHandle* pHandle,
            SoundArchive::ItemId soundId,
            const StartInfo* pStartInfo = NULL );

    //---------------------------------------------------------------------------
    //! @brief  指定したラベル文字列のサウンドを再生する準備をします。
    //!
    //!         準備が完了したサウンドは、ハンドルクラスの
    //!         @ref SoundHandle::StartPrepared
    //!         を呼び出して再生を開始することができます。
    //!
    //!         第一引数には、サウンドハンドルを指定します。
    //!         再生に成功したサウンドは、このサウンドハンドルと関連付けられます。
    //!
    //!         第二引数には、再生するサウンドの識別子として、
    //!         サウンドのラベル文字列を指定します。
    //!
    //!         pStartInfo は、再生時に設定できる詳細なパラメータです。
    //!         このパラメータは、設定せずに再生を開始することが可能です。
    //!
    //! @param[in] pHandle        再生されるサウンドと関連付けられるハンドルです。
    //! @param[in] pSoundName     再生するサウンドのラベル文字列です。
    //! @param[in] pStartInfo     詳細な再生パラメータです。
    //!
    //! @return   再生準備処理結果を @ref StartResult 型で返します。
    //!
    //! @see SoundHandle クラス
    //! @see StartResult
    //! @see StartSound
    //! @see StartInfo
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    StartResult PrepareSound(
            SoundHandle* pHandle,
            const char* pSoundName,
            const StartInfo* pStartInfo = NULL );
    //@}

protected:
    //! @briefprivate
    //! @param handle :private
    //! @param soundId :private
    //! @param holdFlag :private
    //! @param startInfo :private
    //! @return :private
    virtual StartResult detail_SetupSound(
        SoundHandle* handle,
        u32 soundId,
        bool holdFlag,
        const StartInfo* startInfo
    ) = 0;
    //! @briefprivate
    //! @param pString :private
    //! @return :private
    virtual SoundArchive::ItemId detail_GetItemId( const char* pString ) = 0;
};

} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_SOUND_STARTABLE_H_ */

